/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.base.Predicate;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.SuperLiteral;
import org.eclipse.n4js.n4JS.ThisLiteral;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesbuilder.AbstractFunctionDefinitionTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSTypesBuilderHelper;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@Singleton
@SuppressWarnings("all")
class N4JSMethodTypesBuilder extends AbstractFunctionDefinitionTypesBuilder {
  @Inject
  @Extension
  private N4JSTypesBuilderHelper _n4JSTypesBuilderHelper;
  
  boolean relinkMethod(final N4MethodDeclaration methodDecl, final TClassifier classifier, final boolean preLinkingPhase, final int idx) {
    Object _eGet = methodDecl.eGet(N4JSPackage.eINSTANCE.getTypeDefiningElement_DefinedType(), false);
    final EObject methodDefinedType = ((EObject) _eGet);
    if (((methodDefinedType != null) && (!methodDefinedType.eIsProxy()))) {
      throw new IllegalStateException("TMethod already created for N4MethodDeclaration");
    }
    if ((((methodDecl.getName() == null) && (!methodDecl.hasComputedPropertyName())) && (!methodDecl.isCallableConstructor()))) {
      return false;
    }
    TMember _get = classifier.getOwnedMembers().get(idx);
    final TMethod methodType = ((TMethod) _get);
    this._n4JSTypesBuilderHelper.ensureEqualName(methodDecl, methodType);
    this.relinkFormalParameters(methodType, methodDecl, preLinkingPhase);
    methodType.setAstElement(methodDecl);
    methodDecl.setDefinedType(methodType);
    return true;
  }
  
  boolean relinkCallableCtor(final N4MethodDeclaration methodDecl, final TClassifier classifier, final boolean preLinkingPhase) {
    Object _eGet = methodDecl.eGet(N4JSPackage.eINSTANCE.getTypeDefiningElement_DefinedType(), false);
    final EObject methodDefinedType = ((EObject) _eGet);
    if (((methodDefinedType != null) && (!methodDefinedType.eIsProxy()))) {
      throw new IllegalStateException("TMethod already created for N4MethodDeclaration");
    }
    boolean _isCallableConstructor = methodDecl.isCallableConstructor();
    boolean _not = (!_isCallableConstructor);
    if (_not) {
      throw new RuntimeException("Provided method was neither constructor nor callable constructor.");
    }
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(methodDecl.getName());
    boolean _not_1 = (!_isNullOrEmpty);
    if (_not_1) {
      String _name = methodDecl.getName();
      String _plus = ("Callable ctor cannot have a name, had " + _name);
      throw new RuntimeException(_plus);
    }
    boolean _hasComputedPropertyName = methodDecl.hasComputedPropertyName();
    if (_hasComputedPropertyName) {
      throw new RuntimeException("Callable constructor cannot have computed name.");
    }
    final TMethod methodType = classifier.getCallableCtor();
    this.relinkFormalParameters(methodType, methodDecl, preLinkingPhase);
    methodType.setAstElement(methodDecl);
    methodDecl.setDefinedType(methodType);
    return true;
  }
  
  /**
   * Creates TMethod for the given method declaration (and links it to that method).
   * 
   * @param methodDecl declaration for which the TMethod is created, must not be linked to a TMethod yet (i.e. its defined type must be null).
   * @param preLinkingPhase
   */
  TMethod createMethod(final N4MethodDeclaration methodDecl, final boolean preLinkingPhase) {
    Object _eGet = methodDecl.eGet(N4JSPackage.eINSTANCE.getTypeDefiningElement_DefinedType(), false);
    final EObject methodDefinedType = ((EObject) _eGet);
    if (((methodDefinedType != null) && (!methodDefinedType.eIsProxy()))) {
      throw new IllegalStateException("TMethod already created for N4MethodDeclaration");
    }
    if ((((methodDecl.getName() == null) && (!methodDecl.hasComputedPropertyName())) && (!methodDecl.isCallableConstructor()))) {
      return null;
    }
    final TMethod methodType = TypesFactory.eINSTANCE.createTMethod();
    this._n4JSTypesBuilderHelper.setMemberName(methodType, methodDecl);
    methodType.setDeclaredAbstract(methodDecl.isAbstract());
    methodType.setDeclaredStatic(methodDecl.isDeclaredStatic());
    methodType.setDeclaredFinal(methodDecl.isDeclaredFinal());
    methodType.setDeclaredOverride(AnnotationDefinition.OVERRIDE.hasAnnotation(methodDecl));
    methodType.setConstructor(methodDecl.isConstructor());
    methodType.setDeclaredAsync(methodDecl.isAsync());
    methodType.setDeclaredGenerator(methodDecl.isGenerator());
    final boolean providesDefaultImpl = AnnotationDefinition.PROVIDES_DEFAULT_IMPLEMENTATION.hasAnnotation(methodDecl);
    methodType.setHasNoBody(((methodDecl.getBody() == null) && (!providesDefaultImpl)));
    methodType.setLacksThisOrSuperUsage((this.hasNonNullBody(methodDecl.getBody()) && (!this.containsThisOrSuperUsage(methodDecl.getBody()))));
    final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(methodDecl.eResource().getResourceSet());
    this.setMemberAccessModifier(methodType, methodDecl);
    this.addTypeVariables(methodType, methodDecl, preLinkingPhase);
    this.addFormalParameters(methodType, methodDecl, builtInTypeScope, preLinkingPhase);
    this.setReturnTypeConsideringThis(methodType, methodDecl, builtInTypeScope, preLinkingPhase);
    this._n4JSTypesBuilderHelper.setDeclaredThisTypeFromAnnotation(methodType, methodDecl, preLinkingPhase);
    this._n4JSTypesBuilderHelper.copyAnnotations(methodType, methodDecl, preLinkingPhase);
    methodType.setAstElement(methodDecl);
    methodDecl.setDefinedType(methodType);
    return methodType;
  }
  
  private void setMemberAccessModifier(final TMethod methodType, final N4MethodDeclaration n4Method) {
    final Procedure1<MemberAccessModifier> _function = (MemberAccessModifier modifier) -> {
      methodType.setDeclaredMemberAccessModifier(modifier);
    };
    this._n4JSTypesBuilderHelper.setMemberAccessModifier(_function, 
      n4Method.getDeclaredModifiers(), n4Method.getAnnotations());
  }
  
  /**
   * Sets the return type. If the declared return type is 'this', a ComputedTypeRef will
   * be created to generate a bound this type.
   */
  private void setReturnTypeConsideringThis(final TMethod methodType, final N4MethodDeclaration methodDecl, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    if ((methodDecl.isConstructor() || (methodDecl.getReturnTypeRef() instanceof ThisTypeRef))) {
      methodType.setReturnTypeRef(TypeUtils.createDeferredTypeRef());
    } else {
      this.setReturnType(methodType, methodDecl, builtInTypeScope, preLinkingPhase);
    }
  }
  
  private boolean hasNonNullBody(final Block body) {
    return ((null != body) && (null != body.getAllStatements()));
  }
  
  /**
   * Checks for the presence of 'this' or 'super' usages in the given body,
   * also including sub-expressions (eg, 'if (sub-expr)'),
   * without delving inside function definitions or declarations.
   * <p>
   * Static methods refer to static members via ThisLiteral.
   */
  private boolean containsThisOrSuperUsage(final Block body) {
    final Function1<Statement, Boolean> _function = (Statement stmt) -> {
      return Boolean.valueOf((this.isThisOrSuperUsage(stmt) || 
        IteratorExtensions.<EObject>exists(EcoreUtilN4.getAllContentsFiltered(stmt, ((Predicate<EObject>) (EObject it) -> {
          boolean _isFnDefOrDecl = this.isFnDefOrDecl(it);
          return (!_isFnDefOrDecl);
        })), ((Function1<EObject, Boolean>) (EObject it) -> {
          return Boolean.valueOf(this.isThisOrSuperUsage(it));
        }))));
    };
    return IteratorExtensions.<Statement>exists(body.getAllStatements(), _function);
  }
  
  private boolean isFnDefOrDecl(final EObject ast) {
    return ((ast instanceof FunctionDeclaration) || (ast instanceof FunctionDefinition));
  }
  
  private boolean isThisOrSuperUsage(final EObject expr) {
    return ((expr instanceof SuperLiteral) || (expr instanceof ThisLiteral));
  }
}
