/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.Collection;
import java.util.List;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.AnnotationArgument;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.LiteralAnnotationArgument;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.ModifiableElement;
import org.eclipse.n4js.n4JS.ModifierUtils;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4Modifier;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.PropertyNameKind;
import org.eclipse.n4js.n4JS.PropertyNameOwner;
import org.eclipse.n4js.n4JS.TypeRefAnnotationArgument;
import org.eclipse.n4js.postprocessing.ComputedNameProcessor;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.AccessibleTypeElement;
import org.eclipse.n4js.ts.types.DeclaredTypeWithAccessModifier;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TAnnotableElement;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TAnnotationArgument;
import org.eclipse.n4js.ts.types.TAnnotationStringArgument;
import org.eclipse.n4js.ts.types.TAnnotationTypeRefArgument;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TypeAccessModifier;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@Singleton
@SuppressWarnings("all")
class N4JSTypesBuilderHelper {
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  private static Logger logger = Logger.getLogger(N4JSTypesBuilderHelper.class);
  
  protected <T extends AnnotableElement & ModifiableElement> void setTypeAccessModifier(final AccessibleTypeElement classifier, final T definition) {
    final boolean isPlainJS = this.jsVariantHelper.isPlainJS(definition);
    if (isPlainJS) {
      classifier.setDeclaredTypeAccessModifier(TypeAccessModifier.PUBLIC);
    } else {
      classifier.setDeclaredTypeAccessModifier(ModifierUtils.convertToTypeAccessModifier(
        definition.getDeclaredModifiers(), definition.getAllAnnotations()));
    }
  }
  
  void setProvidedByRuntime(final DeclaredTypeWithAccessModifier declaredType, final AnnotableElement annotableElement, final boolean preLinkingPhase) {
    declaredType.setDeclaredProvidedByRuntime(AnnotationDefinition.PROVIDED_BY_RUNTIME.hasAnnotation(annotableElement));
  }
  
  /**
   * Adds references to a feature (via the closure), but copies the references in order to avoid problems with containment relations.
   * The references are copied with proxies (see {@link TypeUtils#copyWithProxies(EObject)} in order to avoid
   * resolving of proxies here.
   * 
   * @param typeListAssignment closure, actually adds the processed list
   * @param listToAssign the list with references, there must be no proxy in the list
   * @param preLinkingPhase if true, references are not set (they are only set in the linking phase)
   * @param <T> reference type, e.g., ParameterizedTypeRef
   */
  <T extends EObject> void addCopyOfReferences(final List<? super T> target, final List<T> values) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(values);
    if (_isNullOrEmpty) {
      return;
    }
    final Function1<T, Boolean> _function = (T it) -> {
      return Boolean.valueOf(((it != null) && it.eIsProxy()));
    };
    boolean _exists = IterableExtensions.<T>exists(values, _function);
    if (_exists) {
      throw new IllegalStateException("There is a proxy in the list, cannot copy and set references");
    }
    final Function1<T, Boolean> _function_1 = (T it) -> {
      return Boolean.valueOf((it == null));
    };
    boolean _exists_1 = IterableExtensions.<T>exists(values, _function_1);
    if (_exists_1) {
      return;
    }
    final Function1<T, T> _function_2 = (T it) -> {
      return TypeUtils.<T>copyWithProxies(it);
    };
    List<? extends T> _map = ListExtensions.map(values, _function_2);
    Iterables.addAll(target, _map);
  }
  
  /**
   * Initializes the name of a TMember in the TModule based the member/property declaration in the AST. In case of
   * computed property names, this method will keep the name in the TModule set to <code>null</code> and
   * {@link ComputedNameProcessor} will later change it to the actual name during post-processing.
   */
  void setMemberName(final TMember tMember, final PropertyNameOwner n4MemberOrPropertyAssignment) {
    tMember.setName(n4MemberOrPropertyAssignment.getName());
    LiteralOrComputedPropertyName _declaredName = n4MemberOrPropertyAssignment.getDeclaredName();
    PropertyNameKind _kind = null;
    if (_declaredName!=null) {
      _kind=_declaredName.getKind();
    }
    boolean _tripleEquals = (_kind == PropertyNameKind.COMPUTED);
    tMember.setHasComputedName(_tripleEquals);
  }
  
  /**
   * Translates AST related member access modifier (and annotation {@code @Interanl}) to type model related member access modifier.
   */
  void setMemberAccessModifier(final Procedure1<? super MemberAccessModifier> memberAccessModifierAssignment, final Collection<? extends N4Modifier> modifiers, final List<Annotation> annotations) {
    memberAccessModifierAssignment.apply(ModifierUtils.convertToMemberAccessModifier(modifiers, annotations));
  }
  
  /**
   * Returns newly created reference to built-in type <code>any</code>.
   */
  ParameterizedTypeRef createAnyTypeRef(final EObject object) {
    ParameterizedTypeRef _xblockexpression = null;
    {
      Resource _eResource = null;
      if (object!=null) {
        _eResource=object.eResource();
      }
      ResourceSet _resourceSet = null;
      if (_eResource!=null) {
        _resourceSet=_eResource.getResourceSet();
      }
      final ResourceSet rs = _resourceSet;
      ParameterizedTypeRef _xifexpression = null;
      if ((rs != null)) {
        _xifexpression = BuiltInTypeScope.get(rs).getAnyTypeRef();
      } else {
        _xifexpression = null;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Copies annotations from AST to Type model. Note that not all annotations are copied, only the ones listed in
   * {@link #ANNOTATIONS_IN_TYPE_MODEL}. Also annotations contained in containing export declaration are copied, as this
   * construct is not present in type model and its annotations are to be defined at the type.
   * <p>
   * Since this mechanism is changed anyway, no type check (whether annotation is allowed for given element type) is
   * performed.
   */
  void copyAnnotations(final TAnnotableElement typeTarget, final AnnotableElement ast, final boolean preLinkingPhase) {
    EList<TAnnotation> _annotations = typeTarget.getAnnotations();
    final Function1<Annotation, Boolean> _function = (Annotation it) -> {
      return Boolean.valueOf(AnnotationDefinition.isInTypeModel(it.getName()));
    };
    final Function1<Annotation, TAnnotation> _function_1 = (Annotation it) -> {
      final TAnnotation ta = TypesFactory.eINSTANCE.createTAnnotation();
      ta.setName(it.getName());
      final Function1<AnnotationArgument, TAnnotationArgument> _function_2 = (AnnotationArgument it_1) -> {
        boolean _matched = false;
        if (it_1 instanceof LiteralAnnotationArgument) {
          _matched=true;
          final TAnnotationStringArgument arg = TypesFactory.eINSTANCE.createTAnnotationStringArgument();
          arg.setValue(((LiteralAnnotationArgument)it_1).getLiteral().getValueAsString());
          return arg;
        }
        if (!_matched) {
          if (it_1 instanceof TypeRefAnnotationArgument) {
            _matched=true;
            final TAnnotationTypeRefArgument arg = TypesFactory.eINSTANCE.createTAnnotationTypeRefArgument();
            arg.setTypeRef(TypeUtils.<TypeRef>copyWithProxies(((TypeRefAnnotationArgument)it_1).getTypeRef()));
            return arg;
          }
        }
        return null;
      };
      ta.getArgs().addAll(ListExtensions.<AnnotationArgument, TAnnotationArgument>map(it.getArgs(), _function_2));
      return ta;
    };
    Iterable<TAnnotation> _map = IterableExtensions.<Annotation, TAnnotation>map(IterableExtensions.<Annotation>filter(ast.getAllAnnotations(), _function), _function_1);
    Iterables.<TAnnotation>addAll(_annotations, _map);
  }
  
  /**
   * Returns newly created reference to built-in type <code>any</code>.
   */
  TClassifier getObjectType(final EObject object) {
    TClassifier _xblockexpression = null;
    {
      Resource _eResource = null;
      if (object!=null) {
        _eResource=object.eResource();
      }
      ResourceSet _resourceSet = null;
      if (_eResource!=null) {
        _resourceSet=_eResource.getResourceSet();
      }
      final ResourceSet rs = _resourceSet;
      TClassifier _xifexpression = null;
      if ((rs != null)) {
        _xifexpression = BuiltInTypeScope.get(rs).getObjectType();
      } else {
        _xifexpression = null;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * @see TClassifier#isDeclaredCovariantConstructor()
   */
  boolean isDeclaredCovariantConstructor(final N4ClassifierDeclaration classifierDecl) {
    boolean _hasAnnotation = AnnotationDefinition.COVARIANT_CONSTRUCTOR.hasAnnotation(classifierDecl);
    if (_hasAnnotation) {
      return true;
    }
    final Function1<N4MemberDeclaration, Boolean> _function = (N4MemberDeclaration it) -> {
      return Boolean.valueOf(it.isConstructor());
    };
    final N4MemberDeclaration ctor = IterableExtensions.<N4MemberDeclaration>findFirst(classifierDecl.getOwnedMembers(), _function);
    return ((ctor != null) && AnnotationDefinition.COVARIANT_CONSTRUCTOR.hasAnnotation(ctor));
  }
  
  /**
   * Handle optional "@This" annotation and set its argument as declared this type in types model element.
   */
  protected void setDeclaredThisTypeFromAnnotation(final TFunction functionType, final FunctionDefinition functionDef, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      functionType.setDeclaredThisType(TypeUtils.<TypeRef>copyWithProxies(
        this.internalGetDeclaredThisTypeFromAnnotation(functionDef)));
    }
  }
  
  /**
   * Handle optional "@This" annotation and set its argument as declared this type in types model element.
   */
  protected void setDeclaredThisTypeFromAnnotation(final FieldAccessor accessorType, final org.eclipse.n4js.n4JS.FieldAccessor accessorDecl, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      accessorType.setDeclaredThisType(TypeUtils.<TypeRef>copyWithProxies(
        this.internalGetDeclaredThisTypeFromAnnotation(accessorDecl)));
    }
  }
  
  private TypeRef internalGetDeclaredThisTypeFromAnnotation(final AnnotableElement element) {
    final Annotation annThis = AnnotationDefinition.THIS.getAnnotation(element);
    EList<AnnotationArgument> _args = null;
    if (annThis!=null) {
      _args=annThis.getArgs();
    }
    Iterable<TypeRefAnnotationArgument> _filter = null;
    if (_args!=null) {
      _filter=Iterables.<TypeRefAnnotationArgument>filter(_args, TypeRefAnnotationArgument.class);
    }
    TypeRefAnnotationArgument _head = null;
    if (_filter!=null) {
      _head=IterableExtensions.<TypeRefAnnotationArgument>head(_filter);
    }
    TypeRef _typeRef = null;
    if (_head!=null) {
      _typeRef=_head.getTypeRef();
    }
    return _typeRef;
  }
  
  /**
   * Used during {@link N4JSTypesBuilder#relinkTModuleToSource(org.eclipse.xtext.resource.DerivedStateAwareResource,
   * boolean) relinking}, to ensure consistency of named elements between newly loaded AST and original TModule.
   */
  protected void ensureEqualName(final NamedElement astNode, final IdentifiableElement moduleElement) {
    final String nameInAST = astNode.getName();
    final String nameInModule = moduleElement.getName();
    if ((nameInAST != null)) {
      boolean _equals = nameInAST.equals(nameInModule);
      boolean _not = (!_equals);
      if (_not) {
        Resource _eResource = astNode.eResource();
        URI _uRI = null;
        if (_eResource!=null) {
          _uRI=_eResource.getURI();
        }
        final String msg = (((((((("inconsistency between newly loaded AST and to-be-linked TModule: " + "nameInAST=") + nameInAST) + ", ") + "nameInModule=") + nameInModule) + ", ") + "in: ") + _uRI);
        N4JSTypesBuilderHelper.logger.error(msg);
        throw new IllegalStateException(msg);
      }
    }
  }
}
