/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem.utils;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Collection;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.BoundType;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelperStrategy;

/**
 * Type System Helper Strategy for deriving a new, slightly modified TypeRef from an existing TypeRef.
 * For example, creating a FunctionTypeExpression representing the upper/lower bound of an existing
 * FunctionTypeExprOrRef.
 * <p>
 * Main reason for factoring out the below code from n4js.xsemantics is to avoid code duplication and
 * to have such code closer together that is similar and has to be kept aligned over time.
 * <p>
 * If you change one method, check the others if the same change might be required there as well!
 */
@SuppressWarnings("all")
class DerivationComputer extends TypeSystemHelperStrategy {
  @Inject
  private N4JSTypeSystem ts;
  
  public FunctionTypeExpression createSubstitutionOfFunctionTypeExprOrRef(final RuleEnvironment G, final FunctionTypeExprOrRef F) {
    final FunctionTypeExpression result = TypeRefsFactory.eINSTANCE.createFunctionTypeExpression();
    result.setBinding(true);
    result.setDeclaredType(F.getFunctionType());
    EList<TypeVariable> _typeVars = F.getTypeVars();
    for (final TypeVariable currTV : _typeVars) {
      Object _get = G.get(currTV);
      boolean _tripleEquals = (_get == null);
      if (_tripleEquals) {
        EList<TypeVariable> _unboundTypeVars = result.getUnboundTypeVars();
        _unboundTypeVars.add(currTV);
        this.performSubstitutionOnUpperBounds(G, F, currTV, result);
      }
    }
    TypeRef _declaredThisType = F.getDeclaredThisType();
    boolean _tripleNotEquals = (_declaredThisType != null);
    if (_tripleNotEquals) {
      final TypeRef resultDeclaredThisType = this.ts.substTypeVariables(G, F.getDeclaredThisType());
      result.setDeclaredThisType(TypeUtils.<TypeRef>copy(resultDeclaredThisType));
    }
    TypeRef _returnTypeRef = F.getReturnTypeRef();
    boolean _tripleNotEquals_1 = (_returnTypeRef != null);
    if (_tripleNotEquals_1) {
      final TypeRef resultReturnTypeRef = this.ts.substTypeVariables(G, F.getReturnTypeRef());
      result.setReturnTypeRef(TypeUtils.<TypeRef>copyIfContained(resultReturnTypeRef));
    }
    result.setReturnValueMarkedOptional(F.isReturnValueOptional());
    EList<TFormalParameter> _fpars = F.getFpars();
    for (final TFormalParameter fpar : _fpars) {
      if ((fpar != null)) {
        final TFormalParameter newPar = TypesFactory.eINSTANCE.createTFormalParameter();
        newPar.setName(fpar.getName());
        newPar.setVariadic(fpar.isVariadic());
        newPar.setHasInitializerAssignment(fpar.isHasInitializerAssignment());
        TypeRef _typeRef = fpar.getTypeRef();
        boolean _tripleNotEquals_2 = (_typeRef != null);
        if (_tripleNotEquals_2) {
          final TypeRef resultParTypeRef = this.ts.substTypeVariables(G, fpar.getTypeRef());
          newPar.setTypeRef(TypeUtils.<TypeRef>copyIfContained(resultParTypeRef));
        }
        EList<TFormalParameter> _fpars_1 = result.getFpars();
        _fpars_1.add(newPar);
      } else {
        result.getFpars().add(null);
      }
    }
    TypeUtils.copyTypeModifiers(result, F);
    return result;
  }
  
  public FunctionTypeExpression createUpperBoundOfFunctionTypeExprOrRef(final RuleEnvironment G, final FunctionTypeExprOrRef F) {
    return this.createBoundOfFunctionTypeExprOrRef(G, F, BoundType.UPPER);
  }
  
  public FunctionTypeExpression createLowerBoundOfFunctionTypeExprOrRef(final RuleEnvironment G, final FunctionTypeExprOrRef F) {
    return this.createBoundOfFunctionTypeExprOrRef(G, F, BoundType.LOWER);
  }
  
  public FunctionTypeExpression createBoundOfFunctionTypeExprOrRef(final RuleEnvironment G, final FunctionTypeExprOrRef F, final BoundType boundType) {
    final FunctionTypeExpression result = TypeRefsFactory.eINSTANCE.createFunctionTypeExpression();
    result.setBinding(true);
    result.setDeclaredType(F.getFunctionType());
    EList<TypeVariable> _unboundTypeVars = result.getUnboundTypeVars();
    EList<TypeVariable> _typeVars = F.getTypeVars();
    Iterables.<TypeVariable>addAll(_unboundTypeVars, _typeVars);
    if ((F instanceof FunctionTypeExpression)) {
      EList<TypeRef> _unboundTypeVarsUpperBounds = result.getUnboundTypeVarsUpperBounds();
      Collection<TypeRef> _copyAll = TypeUtils.<TypeRef>copyAll(((FunctionTypeExpression)F).getUnboundTypeVarsUpperBounds());
      Iterables.<TypeRef>addAll(_unboundTypeVarsUpperBounds, _copyAll);
    }
    TypeRef _declaredThisType = F.getDeclaredThisType();
    boolean _tripleNotEquals = (_declaredThisType != null);
    if (_tripleNotEquals) {
      result.setDeclaredThisType(
        TypeUtils.<TypeRef>copy(F.getDeclaredThisType()));
    }
    TypeRef _returnTypeRef = F.getReturnTypeRef();
    boolean _tripleNotEquals_1 = (_returnTypeRef != null);
    if (_tripleNotEquals_1) {
      TypeRef _switchResult = null;
      if (boundType != null) {
        switch (boundType) {
          case UPPER:
            _switchResult = this.ts.upperBound(G, F.getReturnTypeRef());
            break;
          case LOWER:
            _switchResult = this.ts.lowerBound(G, F.getReturnTypeRef());
            break;
          default:
            break;
        }
      }
      final TypeRef resultReturnTypeRef = _switchResult;
      result.setReturnTypeRef(
        TypeUtils.<TypeRef>copyIfContained(resultReturnTypeRef));
    }
    EList<TFormalParameter> _fpars = F.getFpars();
    for (final TFormalParameter fpar : _fpars) {
      if ((fpar != null)) {
        final TFormalParameter newPar = TypesFactory.eINSTANCE.createTFormalParameter();
        newPar.setName(fpar.getName());
        newPar.setVariadic(fpar.isVariadic());
        newPar.setHasInitializerAssignment(fpar.isHasInitializerAssignment());
        TypeRef _typeRef = fpar.getTypeRef();
        boolean _tripleNotEquals_2 = (_typeRef != null);
        if (_tripleNotEquals_2) {
          TypeRef _switchResult_1 = null;
          if (boundType != null) {
            switch (boundType) {
              case UPPER:
                _switchResult_1 = this.ts.lowerBound(G, fpar.getTypeRef());
                break;
              case LOWER:
                _switchResult_1 = this.ts.upperBound(G, fpar.getTypeRef());
                break;
              default:
                break;
            }
          }
          final TypeRef resultParTypeRef = _switchResult_1;
          newPar.setTypeRef(TypeUtils.<TypeRef>copyIfContained(resultParTypeRef));
        }
        EList<TFormalParameter> _fpars_1 = result.getFpars();
        _fpars_1.add(newPar);
      } else {
        result.getFpars().add(null);
      }
    }
    TypeUtils.copyTypeModifiers(result, F);
    return result;
  }
  
  /**
   * Performing substitution on the upper bound of an unbound(!) type variable is non-trivial, because we aren't
   * allowed to copy the type variable and change its upper bound (short version: a type variable is a type and
   * therefore needs to be contained in a Resource; but our new FunctionTypeExpression 'result' is a TypeRef which
   * may not be contained in any Resource).
   * <p>
   * If type variable substitution on <code>currTV</code>'s upper bound leads to a change of that upper bound (and
   * only then!), the modified upper bound will be stored in property 'unboundTypeVarsUpperBounds' of
   * <code>result</code>.
   * <p>
   * This has to be carefully aligned with {@link FunctionTypeExpression#getUnboundTypeVarsUpperBounds()} and
   * {@link FunctionTypeExpression#getTypeVarUpperBound(TypeVariable)}.
   */
  private void performSubstitutionOnUpperBounds(final RuleEnvironment G, final FunctionTypeExprOrRef F, final TypeVariable currTV, final FunctionTypeExpression result) {
    final TypeRef currTV_declUB = currTV.getDeclaredUpperBound();
    if ((currTV_declUB != null)) {
      final TypeRef oldUB = F.getTypeVarUpperBound(currTV);
      final TypeRef newUB = this.ts.substTypeVariables(G, oldUB);
      final boolean unchanged = (newUB == currTV_declUB);
      if ((!unchanged)) {
        final int idx = result.getUnboundTypeVars().indexOf(currTV);
        while ((result.getUnboundTypeVarsUpperBounds().size() < idx)) {
          result.getUnboundTypeVarsUpperBounds().add(null);
        }
        final TypeRef ubSubst = newUB;
        EList<TypeRef> _unboundTypeVarsUpperBounds = result.getUnboundTypeVarsUpperBounds();
        _unboundTypeVarsUpperBounds.add(ubSubst);
      } else {
      }
    }
  }
}
