/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem.utils;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.util.EcoreUtil.EqualityHelper;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.ts.typeRefs.ExistentialTypeRef;
import org.eclipse.n4js.ts.typeRefs.OptionalFieldStrategy;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeCompareUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.constraints.TypeConstraint;
import org.eclipse.n4js.typesystem.utils.Result;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.StructuralTypingResult;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelperStrategy;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.StructuralMembersPredicates;
import org.eclipse.n4js.utils.StructuralMembersTriple;
import org.eclipse.n4js.utils.StructuralMembersTripleIterator;
import org.eclipse.n4js.utils.StructuralTypesHelper;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

@Singleton
@SuppressWarnings("all")
public class StructuralTypingComputer extends TypeSystemHelperStrategy {
  @Data
  public static class StructTypingInfo {
    private final RuleEnvironment G;
    
    private final TypeRef left;
    
    private final TypeRef right;
    
    private final TypingStrategy leftStrategy;
    
    private final TypingStrategy rightStrategy;
    
    private final List<String> missingMembers = CollectionLiterals.<String>newArrayList();
    
    private final List<String> wrongMembers = CollectionLiterals.<String>newArrayList();
    
    public StructTypingInfo(final RuleEnvironment G, final TypeRef left, final TypeRef right, final TypingStrategy leftStrategy, final TypingStrategy rightStrategy) {
      super();
      this.G = G;
      this.left = left;
      this.right = right;
      this.leftStrategy = leftStrategy;
      this.rightStrategy = rightStrategy;
    }
    
    @Override
    @Pure
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((this.G== null) ? 0 : this.G.hashCode());
      result = prime * result + ((this.left== null) ? 0 : this.left.hashCode());
      result = prime * result + ((this.right== null) ? 0 : this.right.hashCode());
      result = prime * result + ((this.leftStrategy== null) ? 0 : this.leftStrategy.hashCode());
      result = prime * result + ((this.rightStrategy== null) ? 0 : this.rightStrategy.hashCode());
      result = prime * result + ((this.missingMembers== null) ? 0 : this.missingMembers.hashCode());
      return prime * result + ((this.wrongMembers== null) ? 0 : this.wrongMembers.hashCode());
    }
    
    @Override
    @Pure
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      StructuralTypingComputer.StructTypingInfo other = (StructuralTypingComputer.StructTypingInfo) obj;
      if (this.G == null) {
        if (other.G != null)
          return false;
      } else if (!this.G.equals(other.G))
        return false;
      if (this.left == null) {
        if (other.left != null)
          return false;
      } else if (!this.left.equals(other.left))
        return false;
      if (this.right == null) {
        if (other.right != null)
          return false;
      } else if (!this.right.equals(other.right))
        return false;
      if (this.leftStrategy == null) {
        if (other.leftStrategy != null)
          return false;
      } else if (!this.leftStrategy.equals(other.leftStrategy))
        return false;
      if (this.rightStrategy == null) {
        if (other.rightStrategy != null)
          return false;
      } else if (!this.rightStrategy.equals(other.rightStrategy))
        return false;
      if (this.missingMembers == null) {
        if (other.missingMembers != null)
          return false;
      } else if (!this.missingMembers.equals(other.missingMembers))
        return false;
      if (this.wrongMembers == null) {
        if (other.wrongMembers != null)
          return false;
      } else if (!this.wrongMembers.equals(other.wrongMembers))
        return false;
      return true;
    }
    
    @Override
    @Pure
    public String toString() {
      ToStringBuilder b = new ToStringBuilder(this);
      b.add("G", this.G);
      b.add("left", this.left);
      b.add("right", this.right);
      b.add("leftStrategy", this.leftStrategy);
      b.add("rightStrategy", this.rightStrategy);
      b.add("missingMembers", this.missingMembers);
      b.add("wrongMembers", this.wrongMembers);
      return b.toString();
    }
    
    @Pure
    public RuleEnvironment getG() {
      return this.G;
    }
    
    @Pure
    public TypeRef getLeft() {
      return this.left;
    }
    
    @Pure
    public TypeRef getRight() {
      return this.right;
    }
    
    @Pure
    public TypingStrategy getLeftStrategy() {
      return this.leftStrategy;
    }
    
    @Pure
    public TypingStrategy getRightStrategy() {
      return this.rightStrategy;
    }
    
    @Pure
    public List<String> getMissingMembers() {
      return this.missingMembers;
    }
    
    @Pure
    public List<String> getWrongMembers() {
      return this.wrongMembers;
    }
  }
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private N4JSElementKeywordProvider keywordProvider;
  
  @Inject
  private StructuralTypesHelper structuralTypesHelper;
  
  public StructuralTypingResult isStructuralSubtype(final RuleEnvironment G, final TypeRef left, final TypeRef right) {
    final TypingStrategy rightStrategy = right.getTypingStrategy();
    final TypingStrategy leftStrategy = left.getTypingStrategy();
    if ((((((((((left.getDeclaredType() instanceof TN4Classifier) || (left.getDeclaredType() instanceof TypeVariable)) && (left.getTypingStrategy() == right.getTypingStrategy())) && (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER != leftStrategy)) && (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER != rightStrategy)) && (left.getDeclaredType() == right.getDeclaredType())) && left.getStructuralMembers().isEmpty()) && right.getStructuralMembers().isEmpty()) && left.getTypeArgs().isEmpty())) {
      return StructuralTypingResult.result(left, right, CollectionLiterals.<String>emptyList(), CollectionLiterals.<String>emptyList());
    }
    if (((!right.isUseSiteStructuralTyping()) && right.isDefSiteStructuralTyping())) {
      boolean _subtypeSucceeded = this.ts.subtypeSucceeded(G, left, RuleEnvironmentExtensions.n4ObjectTypeRef(G));
      if (_subtypeSucceeded) {
        String _typeRefAsString = right.getTypeRefAsString();
        String _plus = ("All N4Objects must explicitly extend/implement definition site structural type " + _typeRefAsString);
        String _plus_1 = (_plus + ".");
        return StructuralTypingResult.failureDefSiteWithN4Object(_plus_1);
      }
    }
    final StructuralTypingResult primitiveSubtypingResult = this.isPrimitiveStructuralSubtype(G, left, right);
    if ((null != primitiveSubtypingResult)) {
      return primitiveSubtypingResult;
    }
    boolean _isStructuralSubtypingInProgressFor = this.isStructuralSubtypingInProgressFor(G, left, right);
    if (_isStructuralSubtypingInProgressFor) {
      return StructuralTypingResult.result(left, right, CollectionLiterals.<String>emptyList(), CollectionLiterals.<String>emptyList());
    }
    final RuleEnvironment G2 = RuleEnvironmentExtensions.wrap(G);
    this.rememberStructuralSubtypingInProgressFor(G2, left, right);
    final StructuralTypingComputer.StructTypingInfo info = new StructuralTypingComputer.StructTypingInfo(G2, left, right, leftStrategy, rightStrategy);
    final StructuralMembersTripleIterator iter = this.structuralTypesHelper.getMembersTripleIterator(G2, left, right, true);
    while (iter.hasNext()) {
      this.checkMembers(left, iter.next(), info);
    }
    return StructuralTypingResult.result(left, right, info.missingMembers, info.wrongMembers);
  }
  
  /**
   * Special handling for primitive-structural types.
   * 
   * <p>Note that this method only returns a non-null {@link StructuralTypingResult} if primitive structural subtyping is
   * applicable for the given operands {@code left} and {@code right}.</p>
   * 
   * @returns A {@link StructuralTypingResult} if primitive structural typing is applicable. {@code null} otherwise.
   */
  public StructuralTypingResult isPrimitiveStructuralSubtype(final RuleEnvironment G, final TypeRef leftRaw, final TypeRef right) {
    final TypeRef left = this.changeStringBasedEnumToString(G, leftRaw);
    Type _declaredType = right.getDeclaredType();
    final boolean rightIsPrimitive = (_declaredType instanceof PrimitiveType);
    Type _declaredType_1 = left.getDeclaredType();
    final boolean leftIsPrimitive = (_declaredType_1 instanceof PrimitiveType);
    if ((rightIsPrimitive && (!leftIsPrimitive))) {
      String _typeRefAsString = leftRaw.getTypeRefAsString();
      String _plus = (_typeRefAsString + " is not a subtype of ");
      String _typeRefAsString_1 = right.getTypeRefAsString();
      String _plus_1 = (_plus + _typeRefAsString_1);
      return StructuralTypingResult.failure(_plus_1);
    } else {
      if ((leftIsPrimitive && (!rightIsPrimitive))) {
        String _typeRefAsString_2 = leftRaw.getTypeRefAsString();
        String _plus_2 = (_typeRefAsString_2 + " is not a subtype of ");
        String _typeRefAsString_3 = right.getTypeRefAsString();
        String _plus_3 = (_plus_2 + _typeRefAsString_3);
        return StructuralTypingResult.failure(_plus_3);
      } else {
        if ((leftIsPrimitive && rightIsPrimitive)) {
          StructuralTypingResult _xifexpression = null;
          Type _declaredType_2 = left.getDeclaredType();
          Type _declaredType_3 = right.getDeclaredType();
          boolean _tripleEquals = (_declaredType_2 == _declaredType_3);
          if (_tripleEquals) {
            _xifexpression = StructuralTypingResult.success();
          } else {
            String _typeRefAsString_4 = leftRaw.getTypeRefAsString();
            String _plus_4 = (_typeRefAsString_4 + " is not a subtype of ");
            String _typeRefAsString_5 = right.getTypeRefAsString();
            String _plus_5 = (_plus_4 + _typeRefAsString_5);
            _xifexpression = StructuralTypingResult.failure(_plus_5);
          }
          return _xifexpression;
        } else {
          return null;
        }
      }
    }
  }
  
  /**
   * Replace type references pointing to the type of a <code>@StringBased</code> enum by a reference to built-in type
   * <code>string</code>, leaving all other types unchanged.
   */
  private TypeRef changeStringBasedEnumToString(final RuleEnvironment G, final TypeRef typeRef) {
    final Type declType = typeRef.getDeclaredType();
    if (((declType instanceof TEnum) && AnnotationDefinition.STRING_BASED.hasAnnotation(declType))) {
      return RuleEnvironmentExtensions.stringTypeRef(G);
    }
    return typeRef;
  }
  
  private void checkMembers(final TypeRef leftTypeRef, final StructuralMembersTriple triple, final StructuralTypingComputer.StructTypingInfo info) {
    final TMember leftMember = triple.getLeft();
    final TMember rightMember = triple.getRight();
    final FieldAccessor leftOtherAccessor = triple.getLeftOtherAccessor();
    final TypingStrategy leftStrategy = info.leftStrategy;
    final TypingStrategy rightStrategy = info.rightStrategy;
    this.checkMembers(leftTypeRef, leftMember, rightMember, info, rightStrategy);
    if (rightStrategy != null) {
      switch (rightStrategy) {
        case STRUCTURAL_READ_ONLY_FIELDS:
          final boolean handleOptionality = N4JSLanguageUtils.isOptionalityLessRestrictedOrEqual(leftTypeRef.getASTNodeOptionalFieldStrategy(), OptionalFieldStrategy.GETTERS_OPTIONAL);
          final boolean memberNecessary = ((!rightMember.isOptional()) || (rightMember.isOptional() && (!handleOptionality)));
          if ((memberNecessary && (StructuralMembersPredicates.READABLE_FIELDS_PREDICATE.apply(rightMember)).booleanValue())) {
            if ((((TypingStrategy.STRUCTURAL_WRITE_ONLY_FIELDS == leftStrategy) && (!(StructuralMembersPredicates.GETTERS_PREDICATE.apply(leftMember)).booleanValue())) && (!(StructuralMembersPredicates.GETTERS_PREDICATE.apply(leftOtherAccessor)).booleanValue()))) {
              String _name = rightMember.getName();
              String _plus = (_name + " failed: readable field requires a getter in subtype.");
              info.wrongMembers.add(_plus);
            } else {
              if (((!(StructuralMembersPredicates.READABLE_FIELDS_PREDICATE.apply(leftMember)).booleanValue()) && (!(StructuralMembersPredicates.READABLE_FIELDS_PREDICATE.apply(leftOtherAccessor)).booleanValue()))) {
                String _name_1 = rightMember.getName();
                String _plus_1 = (_name_1 + " failed: readable field requires a readable field or a getter in subtype.");
                info.wrongMembers.add(_plus_1);
              }
            }
          }
          break;
        case STRUCTURAL_WRITE_ONLY_FIELDS:
          if (((StructuralMembersPredicates.WRITABLE_FIELDS_PREDICATE.apply(rightMember)).booleanValue() && (!Objects.equal(leftTypeRef.getASTNodeOptionalFieldStrategy(), OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL)))) {
            if (((((TypingStrategy.STRUCTURAL_READ_ONLY_FIELDS == leftStrategy) || (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER == leftStrategy)) && (!(StructuralMembersPredicates.SETTERS_PREDICATE.apply(leftMember)).booleanValue())) && (!(StructuralMembersPredicates.SETTERS_PREDICATE.apply(leftOtherAccessor)).booleanValue()))) {
              String _name_2 = rightMember.getName();
              String _plus_2 = (_name_2 + " failed: writable field requires a setter in subtype.");
              info.wrongMembers.add(_plus_2);
            } else {
              if (((!(StructuralMembersPredicates.WRITABLE_FIELDS_PREDICATE.apply(leftMember)).booleanValue()) && (!(StructuralMembersPredicates.WRITABLE_FIELDS_PREDICATE.apply(leftOtherAccessor)).booleanValue()))) {
                String _name_3 = rightMember.getName();
                String _plus_3 = (_name_3 + " failed: writable field requires a writable field or a setter in subtype.");
                info.wrongMembers.add(_plus_3);
              }
            }
          }
          break;
        case STRUCTURAL_FIELD_INITIALIZER:
          if (((StructuralMembersPredicates.WRITABLE_FIELDS_PREDICATE.apply(rightMember)).booleanValue() && this.isMandatoryField(rightMember))) {
            if (((TypingStrategy.STRUCTURAL_WRITE_ONLY_FIELDS == leftStrategy) && (!((StructuralMembersPredicates.GETTERS_PREDICATE.apply(leftMember)).booleanValue() || (StructuralMembersPredicates.GETTERS_PREDICATE.apply(leftOtherAccessor)).booleanValue())))) {
              String _name_4 = rightMember.getName();
              String _plus_4 = (_name_4 + " failed: non-optional writable field requires a getter in subtype.");
              info.wrongMembers.add(_plus_4);
            } else {
              boolean _not = (!((StructuralMembersPredicates.READABLE_FIELDS_PREDICATE.apply(leftMember)).booleanValue() || (StructuralMembersPredicates.READABLE_FIELDS_PREDICATE.apply(leftOtherAccessor)).booleanValue()));
              if (_not) {
                String _name_5 = rightMember.getName();
                String _plus_5 = (_name_5 + " failed: non-optional writable field requires a readable field or a getter in subtype.");
                info.wrongMembers.add(_plus_5);
              }
            }
          }
          break;
        default:
          if ((((TypingStrategy.STRUCTURAL_READ_ONLY_FIELDS == leftStrategy) || (TypingStrategy.STRUCTURAL_WRITE_ONLY_FIELDS == leftStrategy)) || (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER == leftStrategy))) {
            boolean _matched = false;
            boolean _isWriteableField = N4JSLanguageUtils.isWriteableField(rightMember);
            if (_isWriteableField) {
              _matched=true;
              boolean _isGetterSetterPair = this.isGetterSetterPair(leftMember, leftOtherAccessor);
              boolean _not_1 = (!_isGetterSetterPair);
              if (_not_1) {
                String _name_6 = rightMember.getName();
                String _plus_6 = (_name_6 + " failed: writable field requires a getter/setter pair in subtype.");
                info.wrongMembers.add(_plus_6);
              }
            }
            if (!_matched) {
              Boolean _apply = StructuralMembersPredicates.READABLE_FIELDS_PREDICATE.apply(rightMember);
              if (_apply) {
                _matched=true;
                boolean _not_2 = (!((StructuralMembersPredicates.GETTERS_PREDICATE.apply(leftMember)).booleanValue() || (StructuralMembersPredicates.GETTERS_PREDICATE.apply(leftOtherAccessor)).booleanValue()));
                if (_not_2) {
                  String _name_7 = rightMember.getName();
                  String _plus_7 = (_name_7 + " failed: read-only field requires a getter in subtype.");
                  info.wrongMembers.add(_plus_7);
                }
              }
            }
            if (!_matched) {
              Boolean _apply_1 = StructuralMembersPredicates.SETTERS_PREDICATE.apply(rightMember);
              if (_apply_1) {
                _matched=true;
                boolean _not_3 = (!((StructuralMembersPredicates.SETTERS_PREDICATE.apply(leftMember)).booleanValue() || (StructuralMembersPredicates.SETTERS_PREDICATE.apply(leftOtherAccessor)).booleanValue()));
                if (_not_3) {
                  String _name_8 = rightMember.getName();
                  String _plus_8 = (_name_8 + " failed: setter requires a setter in subtype.");
                  info.wrongMembers.add(_plus_8);
                }
              }
            }
          } else {
            if ((N4JSLanguageUtils.isWriteableField(rightMember) && (leftMember instanceof FieldAccessor))) {
              if ((!(leftOtherAccessor instanceof TSetter))) {
                OptionalFieldStrategy _aSTNodeOptionalFieldStrategy = leftTypeRef.getASTNodeOptionalFieldStrategy();
                boolean _notEquals = (!Objects.equal(_aSTNodeOptionalFieldStrategy, OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL));
                if (_notEquals) {
                  final boolean isSpecialCaseOfDispensableGetterForOptionalField = (((leftMember instanceof TSetter) && rightMember.isOptional()) && Objects.equal(leftTypeRef.getASTNodeOptionalFieldStrategy(), OptionalFieldStrategy.GETTERS_OPTIONAL));
                  if ((!isSpecialCaseOfDispensableGetterForOptionalField)) {
                    String _xifexpression = null;
                    if (((leftMember instanceof TGetter) && rightMember.isOptional())) {
                      _xifexpression = "optional writable field requires at least a setter in subtype.";
                    } else {
                      _xifexpression = "writable field requires a field or a getter/setter pair in subtype.";
                    }
                    final String msgSpecial = _xifexpression;
                    String _name_9 = rightMember.getName();
                    String _plus_9 = (_name_9 + " failed: ");
                    String _plus_10 = (_plus_9 + msgSpecial);
                    info.wrongMembers.add(_plus_10);
                  }
                }
              } else {
                this.checkMembers(leftTypeRef, leftOtherAccessor, rightMember, info, rightStrategy);
              }
            }
          }
          break;
      }
    } else {
      if ((((TypingStrategy.STRUCTURAL_READ_ONLY_FIELDS == leftStrategy) || (TypingStrategy.STRUCTURAL_WRITE_ONLY_FIELDS == leftStrategy)) || (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER == leftStrategy))) {
        boolean _matched = false;
        boolean _isWriteableField = N4JSLanguageUtils.isWriteableField(rightMember);
        if (_isWriteableField) {
          _matched=true;
          boolean _isGetterSetterPair = this.isGetterSetterPair(leftMember, leftOtherAccessor);
          boolean _not_1 = (!_isGetterSetterPair);
          if (_not_1) {
            String _name_6 = rightMember.getName();
            String _plus_6 = (_name_6 + " failed: writable field requires a getter/setter pair in subtype.");
            info.wrongMembers.add(_plus_6);
          }
        }
        if (!_matched) {
          Boolean _apply = StructuralMembersPredicates.READABLE_FIELDS_PREDICATE.apply(rightMember);
          if (_apply) {
            _matched=true;
            boolean _not_2 = (!((StructuralMembersPredicates.GETTERS_PREDICATE.apply(leftMember)).booleanValue() || (StructuralMembersPredicates.GETTERS_PREDICATE.apply(leftOtherAccessor)).booleanValue()));
            if (_not_2) {
              String _name_7 = rightMember.getName();
              String _plus_7 = (_name_7 + " failed: read-only field requires a getter in subtype.");
              info.wrongMembers.add(_plus_7);
            }
          }
        }
        if (!_matched) {
          Boolean _apply_1 = StructuralMembersPredicates.SETTERS_PREDICATE.apply(rightMember);
          if (_apply_1) {
            _matched=true;
            boolean _not_3 = (!((StructuralMembersPredicates.SETTERS_PREDICATE.apply(leftMember)).booleanValue() || (StructuralMembersPredicates.SETTERS_PREDICATE.apply(leftOtherAccessor)).booleanValue()));
            if (_not_3) {
              String _name_8 = rightMember.getName();
              String _plus_8 = (_name_8 + " failed: setter requires a setter in subtype.");
              info.wrongMembers.add(_plus_8);
            }
          }
        }
      } else {
        if ((N4JSLanguageUtils.isWriteableField(rightMember) && (leftMember instanceof FieldAccessor))) {
          if ((!(leftOtherAccessor instanceof TSetter))) {
            OptionalFieldStrategy _aSTNodeOptionalFieldStrategy = leftTypeRef.getASTNodeOptionalFieldStrategy();
            boolean _notEquals = (!Objects.equal(_aSTNodeOptionalFieldStrategy, OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL));
            if (_notEquals) {
              final boolean isSpecialCaseOfDispensableGetterForOptionalField = (((leftMember instanceof TSetter) && rightMember.isOptional()) && Objects.equal(leftTypeRef.getASTNodeOptionalFieldStrategy(), OptionalFieldStrategy.GETTERS_OPTIONAL));
              if ((!isSpecialCaseOfDispensableGetterForOptionalField)) {
                String _xifexpression = null;
                if (((leftMember instanceof TGetter) && rightMember.isOptional())) {
                  _xifexpression = "optional writable field requires at least a setter in subtype.";
                } else {
                  _xifexpression = "writable field requires a field or a getter/setter pair in subtype.";
                }
                final String msgSpecial = _xifexpression;
                String _name_9 = rightMember.getName();
                String _plus_9 = (_name_9 + " failed: ");
                String _plus_10 = (_plus_9 + msgSpecial);
                info.wrongMembers.add(_plus_10);
              }
            }
          } else {
            this.checkMembers(leftTypeRef, leftOtherAccessor, rightMember, info, rightStrategy);
          }
        }
      }
    }
  }
  
  /**
   * Returns with {@code true} iff the  the arguments are a getter-setter accessor pair.
   */
  private boolean isGetterSetterPair(final TMember firstLeft, final TMember secondLeft) {
    return (((StructuralMembersPredicates.GETTERS_PREDICATE.apply(firstLeft)).booleanValue() || (StructuralMembersPredicates.GETTERS_PREDICATE.apply(secondLeft)).booleanValue()) && ((StructuralMembersPredicates.SETTERS_PREDICATE.apply(secondLeft)).booleanValue() || (StructuralMembersPredicates.SETTERS_PREDICATE.apply(secondLeft)).booleanValue()));
  }
  
  /**
   * Checks if the member 'left' (may be <code>null</code> if not found) fulfills the structural requirement represented
   * by member 'right'.  In the error case, the two lists of error messages in 'info' are updated accordingly.
   * <p>
   * NOTE: in case of a field on right-hand side, this method accepts a getter OR(!) a setter of appropriate type
   * on left side; the requirement that BOTH a getter AND setter must be provided for a writable field must be
   * checked outside this method.
   */
  private void checkMembers(final TypeRef leftTypeRef, final TMember left, final TMember right, final StructuralTypingComputer.StructTypingInfo info, final TypingStrategy rightStrategy) {
    final RuleEnvironment G = info.G;
    if ((left == null)) {
      boolean _memberIsMissing = this.memberIsMissing(leftTypeRef, right, info);
      if (_memberIsMissing) {
        String _keyword = this.keywordProvider.keyword(right, rightStrategy);
        String _plus = (_keyword + " ");
        String _name = right.getName();
        String _plus_1 = (_plus + _name);
        info.missingMembers.add(_plus_1);
      }
    } else {
      final Pair<TypeArgument, TypeArgument> mtypes = this.getMemberTypes(left, right, info);
      Result subtypeResult = ((Result) null);
      if ((left.isOptional() && (!right.isOptional()))) {
        String _name_1 = left.getName();
        String _plus_2 = (_name_1 + " failed: non-optional member requires a corresponding non-optional member in the structural subtype.");
        info.missingMembers.add(_plus_2);
      } else {
        if ((N4JSLanguageUtils.isWriteableField(right) && (left instanceof TField))) {
          if (((N4JSLanguageUtils.isReadOnlyField(left) && (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER != rightStrategy)) && (TypingStrategy.STRUCTURAL_READ_ONLY_FIELDS != rightStrategy))) {
            String _name_2 = right.getName();
            String _plus_3 = (_name_2 + " failed: field is read-only.");
            info.wrongMembers.add(_plus_3);
          } else {
            subtypeResult = this.ts.equaltype(G, mtypes.getKey(), mtypes.getValue());
          }
        } else {
          if (((right instanceof TSetter) || (left instanceof TSetter))) {
            subtypeResult = this.ts.supertype(G, mtypes.getKey(), mtypes.getValue());
          } else {
            subtypeResult = this.ts.subtype(G, mtypes.getKey(), mtypes.getValue());
          }
        }
      }
      if (((subtypeResult != null) && subtypeResult.isFailure())) {
        String _name_3 = right.getName();
        String _plus_4 = (_name_3 + " failed: ");
        String _failureMessage = subtypeResult.getFailureMessage();
        String _plus_5 = (_plus_4 + _failureMessage);
        info.wrongMembers.add(_plus_5);
      }
    }
  }
  
  /**
   * Same as previous method, but instead of actually checking the types, we return a constraint. This would normally
   * belong into class <code>InferenceContext</code>, but is placed here to keep it aligned with above method more
   * easily.
   */
  public TypeConstraint reduceMembers(final TypeRef leftTypeRef, final TMember left, final TMember right, final Variance variance, final StructuralTypingComputer.StructTypingInfo info) {
    if ((variance == Variance.CONTRA)) {
      return this.reduceMembers(leftTypeRef, right, left, Variance.CO, info);
    }
    if ((left == null)) {
      boolean _memberIsMissing = this.memberIsMissing(leftTypeRef, right, info);
      if (_memberIsMissing) {
        return TypeConstraint.FALSE;
      } else {
        return TypeConstraint.TRUE;
      }
    } else {
      final Pair<TypeArgument, TypeArgument> mtypes = this.getMemberTypes(left, right, info);
      if ((left.isOptional() && (!right.isOptional()))) {
        return TypeConstraint.FALSE;
      } else {
        if ((N4JSLanguageUtils.isWriteableField(right) && (left instanceof TField))) {
          if (((N4JSLanguageUtils.isReadOnlyField(left) && (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER != info.rightStrategy)) && (TypingStrategy.STRUCTURAL_READ_ONLY_FIELDS != info.rightStrategy))) {
            return TypeConstraint.FALSE;
          } else {
            TypeArgument _key = mtypes.getKey();
            TypeArgument _value = mtypes.getValue();
            return new TypeConstraint(_key, _value, Variance.INV);
          }
        } else {
          if (((right instanceof TSetter) || (left instanceof TSetter))) {
            TypeArgument _key_1 = mtypes.getKey();
            TypeArgument _value_1 = mtypes.getValue();
            Variance _inverse = variance.inverse();
            return new TypeConstraint(_key_1, _value_1, _inverse);
          } else {
            TypeArgument _key_2 = mtypes.getKey();
            TypeArgument _value_2 = mtypes.getValue();
            return new TypeConstraint(_key_2, _value_2, variance);
          }
        }
      }
    }
  }
  
  private boolean memberIsMissing(final TypeRef leftTypeRef, final TMember right, final StructuralTypingComputer.StructTypingInfo info) {
    final boolean rightMemberIsOptional = this.rightMemberIsOptional(leftTypeRef, right, info.rightStrategy);
    if (rightMemberIsOptional) {
      return false;
    } else {
      ContainerType<?> _containingType = null;
      if (right!=null) {
        _containingType=right.getContainingType();
      }
      TClassifier _objectType = RuleEnvironmentExtensions.objectType(info.G);
      boolean _tripleEquals = (_containingType == _objectType);
      if (_tripleEquals) {
        return false;
      } else {
        return true;
      }
    }
  }
  
  /**
   * This method implements Req-IDE-240500 in language spec.
   */
  private boolean rightMemberIsOptional(final TypeRef leftTypeRef, final TMember right, final TypingStrategy rightStrategy) {
    final OptionalFieldStrategy leftOptionalStrategy = leftTypeRef.getASTNodeOptionalFieldStrategy();
    boolean _switchResult = false;
    if (rightStrategy != null) {
      switch (rightStrategy) {
        case STRUCTURAL:
        case STRUCTURAL_FIELDS:
          boolean _xifexpression = false;
          boolean _isOptional = right.isOptional();
          boolean _not = (!_isOptional);
          if (_not) {
            _xifexpression = false;
          } else {
            _xifexpression = (Objects.equal(leftOptionalStrategy, OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL) || ((right instanceof TGetter) && N4JSLanguageUtils.isOptionalityLessRestrictedOrEqual(leftOptionalStrategy, OptionalFieldStrategy.GETTERS_OPTIONAL)));
          }
          _switchResult = _xifexpression;
          break;
        case STRUCTURAL_WRITE_ONLY_FIELDS:
          _switchResult = (right.isOptional() && Objects.equal(leftOptionalStrategy, OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL));
          break;
        case STRUCTURAL_READ_ONLY_FIELDS:
          _switchResult = (right.isOptional() && N4JSLanguageUtils.isOptionalityLessRestrictedOrEqual(leftOptionalStrategy, OptionalFieldStrategy.GETTERS_OPTIONAL));
          break;
        case STRUCTURAL_FIELD_INITIALIZER:
          _switchResult = (((right.isOptional() && N4JSLanguageUtils.isOptionalityLessRestrictedOrEqual(leftOptionalStrategy, OptionalFieldStrategy.GETTERS_OPTIONAL)) || 
            this.isInitializedField(right)) || this.isOptionalSetter(right));
          break;
        default:
          _switchResult = false;
          break;
      }
    } else {
      _switchResult = false;
    }
    final boolean rightMemberIsOptional = _switchResult;
    return rightMemberIsOptional;
  }
  
  /**
   * Store a guard in the given rule environment to note that we are in the process of inferring
   * left ~<: right.
   * <p>
   * IDEBUG-171:
   * If we are already computing the structural subtype relation left ~<: right and we are again asked
   * whether left ~<: right, then we simply want to return success.
   * The rationale is that if we run into a cycle while checking the members' types, we can simply say the
   * members causing the cycle won't render the overall evaluation false ("an uns soll es nicht scheitern").
   * <p>
   * EXAMPLE 1:
   * <pre>
   * class Element {
   * 	public ~Element child;
   * }
   * var ~Element e1;
   * var ~Element e2;
   * //e1 = e2;   // remove comment to get stack overflow when disabling if-clause in #isSubtype()
   * </pre>
   * <p>
   * EXAMPLE 2:
   * <pre>
   * class A {
   * 	public ~B f;
   * }
   * class B {
   * 	public ~A f;
   * }
   * var ~A a;
   * var ~B b;
   * //a = b;   // remove comment to get stack overflow when disabling if-clause in #isSubtype()
   * </pre>
   * Note how this is analogous to what EMF is doing when computing structural equality as explained
   * in the paragraph on "populating a two way map" in the following EMF API documentation:
   * {@link EqualityHelper}
   */
  private void rememberStructuralSubtypingInProgressFor(final RuleEnvironment G, final TypeRef left, final TypeRef right) {
    TypeCompareUtils.SemanticEqualsWrapper _wrap = this.wrap(left);
    TypeCompareUtils.SemanticEqualsWrapper _wrap_1 = this.wrap(right);
    Pair<TypeCompareUtils.SemanticEqualsWrapper, TypeCompareUtils.SemanticEqualsWrapper> _mappedTo = Pair.<TypeCompareUtils.SemanticEqualsWrapper, TypeCompareUtils.SemanticEqualsWrapper>of(_wrap, _wrap_1);
    Pair<String, Pair<TypeCompareUtils.SemanticEqualsWrapper, TypeCompareUtils.SemanticEqualsWrapper>> _mappedTo_1 = Pair.<String, Pair<TypeCompareUtils.SemanticEqualsWrapper, TypeCompareUtils.SemanticEqualsWrapper>>of(RuleEnvironmentExtensions.GUARD_STRUCTURAL_TYPING_COMPUTER, _mappedTo);
    G.put(_mappedTo_1, Boolean.TRUE);
  }
  
  private boolean isStructuralSubtypingInProgressFor(final RuleEnvironment G, final TypeRef left, final TypeRef right) {
    TypeCompareUtils.SemanticEqualsWrapper _wrap = this.wrap(left);
    TypeCompareUtils.SemanticEqualsWrapper _wrap_1 = this.wrap(right);
    Pair<TypeCompareUtils.SemanticEqualsWrapper, TypeCompareUtils.SemanticEqualsWrapper> _mappedTo = Pair.<TypeCompareUtils.SemanticEqualsWrapper, TypeCompareUtils.SemanticEqualsWrapper>of(_wrap, _wrap_1);
    Pair<String, Pair<TypeCompareUtils.SemanticEqualsWrapper, TypeCompareUtils.SemanticEqualsWrapper>> _mappedTo_1 = Pair.<String, Pair<TypeCompareUtils.SemanticEqualsWrapper, TypeCompareUtils.SemanticEqualsWrapper>>of(RuleEnvironmentExtensions.GUARD_STRUCTURAL_TYPING_COMPUTER, _mappedTo);
    Object _get = G.get(_mappedTo_1);
    return (_get != null);
  }
  
  private TypeCompareUtils.SemanticEqualsWrapper wrap(final TypeRef typeRef) {
    return new TypeCompareUtils.SemanticEqualsWrapper(typeRef);
  }
  
  private Pair<TypeArgument, TypeArgument> getMemberTypes(final TMember leftMember, final TMember rightMember, final StructuralTypingComputer.StructTypingInfo info) {
    final RuleEnvironment G = info.G;
    final TypeRef typeLeftRaw = this.ts.type(G, leftMember);
    final TypeRef typeRightRaw = this.ts.type(G, rightMember);
    final RuleEnvironment G_left = RuleEnvironmentExtensions.wrap(G);
    final RuleEnvironment G_right = RuleEnvironmentExtensions.wrap(G);
    this._typeSystemHelper.addSubstitutions(G_left, info.left);
    this._typeSystemHelper.addSubstitutions(G_right, info.right);
    final ArrayList<ExistentialTypeRef> reopen = CollectionLiterals.<ExistentialTypeRef>newArrayList();
    this.collectExistentialTypeRefs(G_right, reopen);
    final TypeRef typeLeft = this.ts.substTypeVariables(G_left, typeLeftRaw);
    final TypeRef typeRight = this.ts.substTypeVariables(G_right, typeRightRaw);
    final Consumer<ExistentialTypeRef> _function = (ExistentialTypeRef it) -> {
      RuleEnvironmentExtensions.addExistentialTypeToBeReopened(G, it);
    };
    reopen.forEach(_function);
    return Pair.<TypeArgument, TypeArgument>of(typeLeft, typeRight);
  }
  
  /**
   * Searches the values of all type variable mappings in the given rule environment for
   * {@link ExistentialTypeRef}s and adds them to the given list.
   */
  private void collectExistentialTypeRefs(final RuleEnvironment G, final List<? super ExistentialTypeRef> addHere) {
    RuleEnvironment next = G;
    while ((next != null)) {
      {
        Set<Map.Entry<Object, Object>> _entrySet = next.entrySet();
        for (final Map.Entry<Object, Object> entry : _entrySet) {
          {
            final Object key = entry.getKey();
            if ((key instanceof TypeVariable)) {
              final Object value = entry.getValue();
              if ((value instanceof Collection<?>)) {
                for (final Object currValue : ((Collection<?>)value)) {
                  if ((currValue instanceof TypeRef)) {
                    this.collectExistentialTypeRefs(((TypeRef)currValue), addHere);
                  }
                }
              } else {
                if ((value instanceof TypeRef)) {
                  this.collectExistentialTypeRefs(((TypeRef)value), addHere);
                }
              }
            }
          }
        }
        next = G.getNext();
      }
    }
  }
  
  private void collectExistentialTypeRefs(final TypeRef typeRef, final List<? super ExistentialTypeRef> addHere) {
    if ((typeRef instanceof ExistentialTypeRef)) {
      addHere.add(((ExistentialTypeRef)typeRef));
    } else {
      addHere.addAll(EcoreUtil2.<ExistentialTypeRef>getAllContentsOfType(typeRef, ExistentialTypeRef.class));
    }
  }
  
  /**
   * Returns with {@code true} if the member argument is
   * <ul>
   * <li>*NOT* {@link TMember#isOptional() optional} member,</li>
   * <li>*NOT* {@link #isInitializedField(TMember) initialized field} and</li>
   * <li>*NOT* {@link #isOptionalSetter(TMember) optional setter}.</li>
   * </ul>
   * Otherwise returns with {@code false}.
   */
  private boolean isMandatoryField(final TMember it) {
    return ((((null != it) && (!it.isOptional())) && (!this.isInitializedField(it))) && (!this.isOptionalSetter(it)));
  }
  
  /**
   * Argument is an instance of a {@link TField field} and {@link TField#isHasExpression() has initializer expression}.
   * This method is {@code null} safe.
   */
  private boolean isInitializedField(final TMember it) {
    boolean _xifexpression = false;
    if ((it instanceof TField)) {
      _xifexpression = ((TField)it).isHasExpression();
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  /**
   * Returns {@code true} if the member argument is an instance of a {@link TSetter setter} and has
   * {@link AnnotationDefinition#IDE_1996 IDE_1996} annotation. Otherwise returns with {@code false}.
   */
  private boolean isOptionalSetter(final TMember it) {
    return ((it instanceof TSetter) && AnnotationDefinition.PROVIDES_INITIALZER.hasAnnotation(it));
  }
}
