/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.assistants.TypeAssistant;
import org.eclipse.n4js.transpiler.es.assistants.DelegationAssistant;
import org.eclipse.n4js.transpiler.im.DelegatingMember;
import org.eclipse.n4js.transpiler.utils.ConcreteMembersOrderedForTranspiler;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.util.AccessorTuple;
import org.eclipse.n4js.utils.Log;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Handles some special cases where code has to be emitted for non-owned members, e.g. for members consumed by an
 * interface, <b>EXCEPT</b> for {@link StaticPolyfillTransformation static polyfills} and
 * {@link ApiImplStubGenerationTransformation API/Impl stubs} (those cases are handled in other transformations).
 * <p>
 * This transformation will add new "full" members or {@link DelegatingMember}s to the <code>ownedMembersRaw</code>
 * property of {@link ContainerType}s in the intermediate model.
 */
@Log
@SuppressWarnings("all")
public class MemberPatchingTransformation extends Transformation {
  @Inject
  private DelegationAssistant delegationAssistant;
  
  @Inject
  private TypeAssistant typeAssistant;
  
  @Override
  public void assertPreConditions() {
    this.typeAssistant.assertClassifierPreConditions();
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<N4ClassifierDeclaration> _function = (N4ClassifierDeclaration it) -> {
      this.transformClassifierDecl(it);
    };
    this.<N4ClassifierDeclaration>collectNodes(this.getState().im, N4ClassifierDeclaration.class, false).forEach(_function);
  }
  
  private void _transformClassifierDecl(final N4InterfaceDeclaration ifcDecl) {
    final TInterface tIfc = this.getState().info.getOriginalDefinedType(ifcDecl);
    final ConcreteMembersOrderedForTranspiler cmoft = this.typeAssistant.getOrCreateCMOFT(tIfc);
    for (final TMember m : cmoft.ownedAndMixedInConcreteMembers) {
      if ((!(m instanceof TField))) {
        ContainerType<?> _containingType = m.getContainingType();
        final boolean isInherited = (_containingType != tIfc);
        if (isInherited) {
          final DelegatingMember delegator = this.delegationAssistant.createDelegatingMember(tIfc, m);
          this.getState().info.markAsConsumedFromInterface(delegator);
          EList<N4MemberDeclaration> _ownedMembersRaw = ifcDecl.getOwnedMembersRaw();
          _ownedMembersRaw.add(delegator);
        }
      } else {
      }
    }
  }
  
  private void _transformClassifierDecl(final N4ClassDeclaration classDecl) {
    final TClass tClass = this.getState().info.getOriginalDefinedType(classDecl);
    final ConcreteMembersOrderedForTranspiler cmoft = this.typeAssistant.getOrCreateCMOFT(tClass);
    final Function1<TMethod, Boolean> _function = (TMethod m) -> {
      EObject _eContainer = m.eContainer();
      return Boolean.valueOf((_eContainer != tClass));
    };
    final List<TMethod> consumedMethods = IterableExtensions.<TMethod>toList(IterableExtensions.<TMethod>filter(Iterables.<TMethod>filter(cmoft.ownedAndMixedInConcreteMembers, TMethod.class), _function));
    for (final TMethod m : consumedMethods) {
      {
        final DelegatingMember member = this.delegationAssistant.createDelegatingMember(tClass, m);
        this.getState().info.markAsConsumedFromInterface(member);
        EList<N4MemberDeclaration> _ownedMembersRaw = classDecl.getOwnedMembersRaw();
        _ownedMembersRaw.add(member);
      }
    }
    for (final AccessorTuple accTuple : cmoft.concreteAccessorTuples) {
      {
        if ((((accTuple.getGetter() != null) && (accTuple.getGetter().getContainingType() != tClass)) && (accTuple.getInheritedGetter() == null))) {
          final TGetter g = accTuple.getGetter();
          final DelegatingMember member = this.delegationAssistant.createDelegatingMember(tClass, g);
          this.getState().info.markAsConsumedFromInterface(member);
          EList<N4MemberDeclaration> _ownedMembersRaw = classDecl.getOwnedMembersRaw();
          _ownedMembersRaw.add(member);
        }
        if ((((accTuple.getSetter() != null) && (accTuple.getSetter().getContainingType() != tClass)) && (accTuple.getInheritedSetter() == null))) {
          final TSetter s = accTuple.getSetter();
          final DelegatingMember member_1 = this.delegationAssistant.createDelegatingMember(tClass, s);
          this.getState().info.markAsConsumedFromInterface(member_1);
          EList<N4MemberDeclaration> _ownedMembersRaw_1 = classDecl.getOwnedMembersRaw();
          _ownedMembersRaw_1.add(member_1);
        }
      }
    }
    for (final TField field : cmoft.fieldsPurelyMixedInNotOverridingAccessor) {
      {
        final N4MemberDeclaration member = TranspilerBuilderBlocks._N4MemberDecl(field);
        EList<N4MemberDeclaration> _ownedMembersRaw = classDecl.getOwnedMembersRaw();
        _ownedMembersRaw.add(member);
        this.getState().info.setOriginalDefinedMember(member, field);
        this.getState().info.markAsConsumedFromInterface(member);
      }
    }
    for (final AccessorTuple accTuple_1 : cmoft.concreteAccessorTuples) {
      {
        if ((((accTuple_1.getInheritedGetter() != null) && (accTuple_1.getGetter() == null)) && (accTuple_1.getSetter() != null))) {
          String _name = accTuple_1.getSetter().getName();
          String _plus = ("Encountered an invalid getter shadowing. Setter " + _name);
          String _plus_1 = (_plus + 
            " of classifier ");
          ContainerType<?> _containingType = accTuple_1.getSetter().getContainingType();
          String _plus_2 = (_plus_1 + _containingType);
          String _plus_3 = (_plus_2 + "");
          IllegalStateException _illegalStateException = new IllegalStateException("Invalid shadowing of inherited getter. Getter should be implemented explicitly.");
          MemberPatchingTransformation.logger.error(_plus_3, _illegalStateException);
        }
        if ((((accTuple_1.getInheritedSetter() != null) && (accTuple_1.getGetter() != null)) && (accTuple_1.getSetter() == null))) {
          String _name_1 = accTuple_1.getGetter().getName();
          String _plus_4 = ("Encountered an invalid inherited setter shadowing. Getter " + _name_1);
          String _plus_5 = (_plus_4 + 
            " of classifier ");
          ContainerType<?> _containingType_1 = accTuple_1.getGetter().getContainingType();
          String _plus_6 = (_plus_5 + _containingType_1);
          String _plus_7 = (_plus_6 + "");
          IllegalStateException _illegalStateException_1 = new IllegalStateException("Invalid shadowing of inherited setter. Setter should be implemented explicitly.");
          MemberPatchingTransformation.logger.error(_plus_7, _illegalStateException_1);
        }
      }
    }
  }
  
  private void transformClassifierDecl(final EObject classDecl) {
    if (classDecl instanceof N4ClassDeclaration) {
      _transformClassifierDecl((N4ClassDeclaration)classDecl);
      return;
    } else if (classDecl instanceof N4InterfaceDeclaration) {
      _transformClassifierDecl((N4InterfaceDeclaration)classDecl);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(classDecl).toString());
    }
  }
  
  private final static Logger logger = Logger.getLogger(MemberPatchingTransformation.class);
}
