/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.generator.GeneratorOption;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclarationOrBinding;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.es.transform.ArrowFunction_Part1_Transformation;
import org.eclipse.n4js.transpiler.es.transform.BlockTransformation;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Transforms ES2015 rest parameters to an ES5 equivalent.
 * 
 * Dependencies:
 * <ul>
 * <li>requiresBefore {@link ArrowFunction_Part1_Transformation}:<br>
 *     the transpiled output code for varargs requires the implicit local arguments parameter, which is not available in
 *     arrow functions; therefore, this transformation relies on arrow functions already having been transformed into
 *     ordinary function expressions.
 * <li>requiresBefore {@link BlockTransformation}:<br>
 *     the block transformation sometimes wraps the entire body into a newly created function (for async functions);
 *     the handling of varargs added below must NOT be wrapped in such an inner function; the easiest way to ensure this
 *     is to execute BlockTransformation before this transformation.
 * </ul>
 */
@TransformationDependency.Optional(GeneratorOption.RestParameters)
@TransformationDependency.RequiresBefore({ ArrowFunction_Part1_Transformation.class, BlockTransformation.class })
@SuppressWarnings("all")
public class RestParameterTransformation extends Transformation {
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
    final List<FormalParameter> allFormalParameter = EcoreUtil2.<FormalParameter>eAllOfType(this.getState().im, FormalParameter.class);
    final Function1<FormalParameter, Boolean> _function = (FormalParameter it) -> {
      return Boolean.valueOf(it.isVariadic());
    };
    final List<FormalParameter> stillVariadic = IterableExtensions.<FormalParameter>toList(IterableExtensions.<FormalParameter>filter(allFormalParameter, _function));
    int _size = stillVariadic.size();
    boolean _tripleEquals = (_size == 0);
    this.assertTrue("no rest parameter should be in the model.", _tripleEquals);
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<FunctionDefinition> _function = (FunctionDefinition it) -> {
      this.transform(it);
    };
    this.<FunctionDefinition>collectNodes(this.getState().im, FunctionDefinition.class, true).forEach(_function);
  }
  
  private void transform(final FunctionDefinition fdef) {
    boolean _isEmpty = fdef.getFpars().isEmpty();
    if (_isEmpty) {
      return;
    }
    int _size = fdef.getFpars().size();
    final int lastFparIdx = (_size - 1);
    final FormalParameter lastFpar = fdef.getFpars().get(lastFparIdx);
    boolean _isVariadic = lastFpar.isVariadic();
    boolean _not = (!_isVariadic);
    if (_not) {
      return;
    }
    VariableStatement __VariableStatement = TranspilerBuilderBlocks._VariableStatement();
    final Procedure1<VariableStatement> _function = (VariableStatement it) -> {
      EList<VariableDeclarationOrBinding> _varDeclsOrBindings = it.getVarDeclsOrBindings();
      VariableDeclaration __VariableDeclaration = TranspilerBuilderBlocks._VariableDeclaration(lastFpar.getName());
      final Procedure1<VariableDeclaration> _function_1 = (VariableDeclaration it_1) -> {
        it_1.setExpression(TranspilerBuilderBlocks._CallExpr(
          TranspilerBuilderBlocks._PropertyAccessExpr(this.steFor_Array(), this.steFor_prototype(), this.steFor_slice(), this.steFor_call()), 
          TranspilerBuilderBlocks._IdentRef(this.steFor_arguments()), 
          TranspilerBuilderBlocks._IntLiteral(lastFparIdx)));
      };
      VariableDeclaration _doubleArrow = ObjectExtensions.<VariableDeclaration>operator_doubleArrow(__VariableDeclaration, _function_1);
      _varDeclsOrBindings.add(_doubleArrow);
    };
    final VariableStatement stmt = ObjectExtensions.<VariableStatement>operator_doubleArrow(__VariableStatement, _function);
    Block _body = fdef.getBody();
    boolean _tripleEquals = (_body == null);
    if (_tripleEquals) {
      fdef.setBody(TranspilerBuilderBlocks._Block());
    }
    this.replaceAndRelocate(lastFpar, stmt, stmt.getVarDecl().get(0), fdef.getBody());
  }
}
