/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling;

import com.google.inject.Inject;
import java.util.Arrays;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.text.IRegion;
import org.eclipse.n4js.jsdoc.JSDoc2HoverSerializer;
import org.eclipse.n4js.jsdoc.N4JSDocletParser;
import org.eclipse.n4js.jsdoc.dom.Doclet;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.ui.labeling.TypesHoverProvider;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.ui.editor.hover.html.DefaultEObjectHoverProvider;
import org.eclipse.xtext.ui.editor.hover.html.XtextBrowserInformationControlInput;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;

@SuppressWarnings("all")
public class N4JSHoverProvider extends DefaultEObjectHoverProvider {
  @Inject
  @Extension
  private N4JSTypeSystem _n4JSTypeSystem;
  
  @Inject
  @Extension
  private N4JSElementKeywordProvider _n4JSElementKeywordProvider;
  
  @Inject
  private TypesHoverProvider typesHoverProvider;
  
  @Inject
  private OperationCanceledManager cancelManager;
  
  @Inject
  private N4JSDocletParser docletParser;
  
  @Override
  protected String getFirstLine(final EObject o) {
    if ((o instanceof LiteralOrComputedPropertyName)) {
      return this.getFirstLine(((LiteralOrComputedPropertyName)o).eContainer());
    }
    return TypesHoverProvider.composeFirstLine(this._n4JSElementKeywordProvider.keyword(o), this.getLabel(o));
  }
  
  @Override
  protected String getLabel(final EObject o) {
    return UtilN4.sanitizeForHTML(this.doGetLabel(o));
  }
  
  @Override
  protected String getDocumentation(final EObject o) {
    try {
      String jsdocString = super.getDocumentation(o);
      if ((jsdocString == null)) {
        return null;
      }
      final Doclet doclet = this.docletParser.parse(jsdocString);
      final String hoverHTML = JSDoc2HoverSerializer.toJSDocString(doclet);
      return hoverHTML;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        return ("Error generating documentation:  " + ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private String _doGetLabel(final EObject o) {
    final EObject tElem = N4JSASTUtils.getCorrespondingTypeModelElement(o);
    String _xifexpression = null;
    if ((null == tElem)) {
      _xifexpression = super.getLabel(o);
    } else {
      _xifexpression = this.typesHoverProvider.getLabel(tElem);
    }
    return _xifexpression;
  }
  
  private String _doGetLabel(final VariableDeclaration vd) {
    String _xifexpression = null;
    if ((vd instanceof ExportedVariableDeclaration)) {
      _xifexpression = this._doGetLabel(((EObject) vd));
    } else {
      _xifexpression = this.getLabelFromTypeSystem(vd);
    }
    return _xifexpression;
  }
  
  private String _doGetLabel(final PropertyNameValuePair nameValuePair) {
    return this.getLabelFromTypeSystem(nameValuePair);
  }
  
  private String _doGetLabel(final FormalParameter fp) {
    String _xblockexpression = null;
    {
      String _xifexpression = null;
      boolean _isHasInitializerAssignment = fp.isHasInitializerAssignment();
      if (_isHasInitializerAssignment) {
        _xifexpression = "=…";
      } else {
        _xifexpression = "";
      }
      final String optinonalMarker = _xifexpression;
      String _labelFromTypeSystem = this.getLabelFromTypeSystem(fp);
      _xblockexpression = (_labelFromTypeSystem + optinonalMarker);
    }
    return _xblockexpression;
  }
  
  private String _doGetLabel(final FunctionExpression fe) {
    return this.getLabelFromTypeSystem(fe);
  }
  
  private String _doGetLabel(final LiteralOrComputedPropertyName name) {
    EObject _eContainer = name.eContainer();
    if ((_eContainer instanceof TypableElement)) {
      EObject _eContainer_1 = name.eContainer();
      return this.getLabelFromTypeSystem(((TypableElement) _eContainer_1));
    }
    return name.getName();
  }
  
  private String getLabelFromTypeSystem(final TypableElement o) {
    if (((null == o) || (null == o.eResource()))) {
      return null;
    }
    final TypeRef typeRef = this._n4JSTypeSystem.type(RuleEnvironmentExtensions.newRuleEnvironment(o), o);
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _name = this.getName(o);
    _builder.append(_name);
    _builder.append(": ");
    String _typeRefAsString = typeRef.getTypeRefAsString();
    _builder.append(_typeRefAsString);
    return _builder.toString();
  }
  
  private CharSequence _getName(final EObject o) {
    return "";
  }
  
  private CharSequence _getName(final NamedElement nameValuePair) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(" ");
    String _name = nameValuePair.getName();
    _builder.append(_name, " ");
    return _builder;
  }
  
  @Override
  protected boolean hasHover(final EObject o) {
    return this.doHasHover(o);
  }
  
  private boolean _doHasHover(final EObject o) {
    final EObject tElem = N4JSASTUtils.getCorrespondingTypeModelElement(o);
    boolean _xifexpression = false;
    if ((null == tElem)) {
      _xifexpression = super.hasHover(o);
    } else {
      _xifexpression = this.typesHoverProvider.hasHover(tElem);
    }
    return _xifexpression;
  }
  
  private boolean _doHasHover(final VariableDeclaration vd) {
    return true;
  }
  
  private boolean _doHasHover(final PropertyNameValuePair nameValuePair) {
    return true;
  }
  
  private boolean _doHasHover(final FormalParameter fp) {
    return true;
  }
  
  private boolean _doHasHover(final FunctionExpression fe) {
    return true;
  }
  
  private boolean _doHasHover(final N4TypeDeclaration md) {
    return true;
  }
  
  private boolean _doHasHover(final LiteralOrComputedPropertyName name) {
    EObject _eContainer = name.eContainer();
    return (_eContainer instanceof N4MemberDeclaration);
  }
  
  @Override
  protected XtextBrowserInformationControlInput getHoverInfo(final EObject element, final IRegion hoverRegion, final XtextBrowserInformationControlInput previous) {
    Object _xtrycatchfinallyexpression = null;
    try {
      return super.getHoverInfo(element, hoverRegion, previous);
    } catch (final Throwable _t) {
      if (_t instanceof Throwable) {
        final Throwable t = (Throwable)_t;
        Object _xifexpression = null;
        boolean _isOperationCanceledException = this.cancelManager.isOperationCanceledException(t);
        boolean _not = (!_isOperationCanceledException);
        if (_not) {
          throw Exceptions.sneakyThrow(t);
        }
        _xtrycatchfinallyexpression = _xifexpression;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return ((XtextBrowserInformationControlInput)_xtrycatchfinallyexpression);
  }
  
  private String doGetLabel(final EObject nameValuePair) {
    if (nameValuePair instanceof PropertyNameValuePair) {
      return _doGetLabel((PropertyNameValuePair)nameValuePair);
    } else if (nameValuePair instanceof FormalParameter) {
      return _doGetLabel((FormalParameter)nameValuePair);
    } else if (nameValuePair instanceof FunctionExpression) {
      return _doGetLabel((FunctionExpression)nameValuePair);
    } else if (nameValuePair instanceof VariableDeclaration) {
      return _doGetLabel((VariableDeclaration)nameValuePair);
    } else if (nameValuePair instanceof LiteralOrComputedPropertyName) {
      return _doGetLabel((LiteralOrComputedPropertyName)nameValuePair);
    } else if (nameValuePair != null) {
      return _doGetLabel(nameValuePair);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(nameValuePair).toString());
    }
  }
  
  private CharSequence getName(final EObject nameValuePair) {
    if (nameValuePair instanceof NamedElement) {
      return _getName((NamedElement)nameValuePair);
    } else if (nameValuePair != null) {
      return _getName(nameValuePair);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(nameValuePair).toString());
    }
  }
  
  private boolean doHasHover(final EObject nameValuePair) {
    if (nameValuePair instanceof PropertyNameValuePair) {
      return _doHasHover((PropertyNameValuePair)nameValuePair);
    } else if (nameValuePair instanceof FormalParameter) {
      return _doHasHover((FormalParameter)nameValuePair);
    } else if (nameValuePair instanceof FunctionExpression) {
      return _doHasHover((FunctionExpression)nameValuePair);
    } else if (nameValuePair instanceof N4TypeDeclaration) {
      return _doHasHover((N4TypeDeclaration)nameValuePair);
    } else if (nameValuePair instanceof VariableDeclaration) {
      return _doHasHover((VariableDeclaration)nameValuePair);
    } else if (nameValuePair instanceof LiteralOrComputedPropertyName) {
      return _doHasHover((LiteralOrComputedPropertyName)nameValuePair);
    } else if (nameValuePair != null) {
      return _doHasHover(nameValuePair);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(nameValuePair).toString());
    }
  }
}
