/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils.io;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collection;
import java.util.Set;
import org.apache.log4j.Logger;
import org.eclipse.n4js.utils.io.FileMatchingMode;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * File visitor implementation for collecting files (with their path) that matches a specified pattern.
 */
@SuppressWarnings("all")
public class FileMatcher extends SimpleFileVisitor<Path> {
  private final static Logger LOGGER = Logger.getLogger(FileMatcher.class);
  
  private final static String GLOB_SYNTAX_PREFIX = "glob:";
  
  private final PathMatcher matcher;
  
  private final Collection<Path> matchedPaths;
  
  private final FileMatchingMode mode;
  
  /**
   * Recursively scans for files in the given root location. Returns with a collection of file paths
   * that fulfills the given file name filter pattern.
   */
  public static Set<Path> scanFiles(final Path root, final String pattern) {
    return FileMatcher.scan(root, pattern, FileMatchingMode.FILES);
  }
  
  /**
   * Recursively scans for directories in the given root location. Returns with a collection of directory paths
   * that fulfills the given directory name filter pattern.
   */
  public static Set<Path> scanDirectories(final Path root, final String pattern) {
    return FileMatcher.scan(root, pattern, FileMatchingMode.DIRECTORIES);
  }
  
  private static Set<Path> scan(final Path root, final String pattern, final FileMatchingMode mode) {
    try {
      boolean _exists = Preconditions.<Path>checkNotNull(root).toFile().exists();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("File does not exist: ");
      _builder.append(root);
      _builder.append(".");
      Preconditions.checkArgument(_exists, _builder);
      String _checkNotNull = Preconditions.<String>checkNotNull(pattern);
      final FileMatcher matcher = new FileMatcher(_checkNotNull, mode);
      Files.walkFileTree(root, matcher);
      final Function1<Path, Path> _function = (Path it) -> {
        return root.relativize(it);
      };
      return IterableExtensions.<Path>toSet(IterableExtensions.<Path, Path>map(matcher.matchedPaths, _function));
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("Error while recursively scanning for files in ");
        _builder_1.append(root);
        _builder_1.append(" with pattern: ");
        _builder_1.append(pattern);
        _builder_1.append(".");
        throw new RuntimeException(_builder_1.toString(), e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private FileMatcher(final String pattern, final FileMatchingMode mode) {
    final String escapedPattern = Preconditions.<String>checkNotNull(pattern).replaceAll("\\\\", "/");
    this.mode = mode;
    FileSystem _default = FileSystems.getDefault();
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(FileMatcher.GLOB_SYNTAX_PREFIX);
    _builder.append("**");
    _builder.append(escapedPattern);
    this.matcher = _default.getPathMatcher(_builder.toString());
    this.matchedPaths = CollectionLiterals.<Path>newArrayList();
  }
  
  @Override
  public FileVisitResult preVisitDirectory(final Path dir, final BasicFileAttributes attrs) throws IOException {
    FileVisitResult _xblockexpression = null;
    {
      this.match(dir);
      _xblockexpression = FileVisitResult.CONTINUE;
    }
    return _xblockexpression;
  }
  
  @Override
  public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException {
    FileVisitResult _xblockexpression = null;
    {
      this.match(file);
      _xblockexpression = FileVisitResult.CONTINUE;
    }
    return _xblockexpression;
  }
  
  @Override
  public FileVisitResult visitFileFailed(final Path file, final IOException e) throws IOException {
    FileVisitResult _xblockexpression = null;
    {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Error while visiting file at: ");
      _builder.append(file);
      _builder.append(".");
      FileMatcher.LOGGER.error(_builder, e);
      _xblockexpression = FileVisitResult.CONTINUE;
    }
    return _xblockexpression;
  }
  
  public boolean match(final Path path) {
    boolean _xifexpression = false;
    if ((((null != path) && this.mode.apply(path)) && this.matcher.matches(path))) {
      _xifexpression = this.matchedPaths.add(path);
    }
    return _xifexpression;
  }
}
