/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.resource;

import com.google.inject.Inject;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.resource.N4JSUnloader;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.typesbuilder.N4JSTypesBuilder;
import org.eclipse.xtext.resource.DerivedStateAwareResource;
import org.eclipse.xtext.resource.IDerivedStateComputer;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Derives the types model from the AST and stores it at the second index of the resource.
 * See {@link N4JSTypesBuilder}.
 */
@SuppressWarnings("all")
public class N4JSDerivedStateComputer implements IDerivedStateComputer {
  @Inject
  @Extension
  private N4JSUnloader _n4JSUnloader;
  
  @Inject
  private N4JSTypesBuilder typesBuilder;
  
  /**
   * Creates an {@link TModule} on the second slot of the resource. when the resource contents is not empty.
   */
  @Override
  public void installDerivedState(final DerivedStateAwareResource resource, final boolean preLinkingPhase) {
    final EList<EObject> contents = resource.getContents();
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(contents);
    if (_isNullOrEmpty) {
      String _string = resource.getURI().toString();
      String _plus = ("cannot install derived state in resource \'" + _string);
      final String msg = (_plus + "\' without AST");
      throw new IllegalStateException(msg);
    } else {
      int _size = contents.size();
      boolean _equals = (_size == 1);
      if (_equals) {
        this.typesBuilder.createTModuleFromSource(resource, preLinkingPhase);
      } else {
        int _size_1 = contents.size();
        boolean _equals_1 = (_size_1 == 2);
        if (_equals_1) {
          this.typesBuilder.relinkTModuleToSource(resource, preLinkingPhase);
        } else {
          String _string_1 = resource.getURI().toString();
          String _plus_1 = ("resource \'" + _string_1);
          String _plus_2 = (_plus_1 + "\' with more than two roots");
          throw new IllegalStateException(_plus_2);
        }
      }
    }
  }
  
  /**
   * Calls {@link N4JSUnloader#unloadRoot(EObject} for the second slot root.
   * Then all contents of the resource are cleared.
   */
  @Override
  public void discardDerivedState(final DerivedStateAwareResource resource) {
    final EList<EObject> contents = resource.getContents();
    boolean _isEmpty = contents.isEmpty();
    if (_isEmpty) {
      return;
    }
    final List<EObject> tail = contents.subList(1, contents.size());
    final Consumer<EObject> _function = (EObject it) -> {
      this._n4JSUnloader.unloadRoot(it);
    };
    tail.forEach(_function);
    tail.clear();
  }
}
