/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.scoping.utils;

import com.google.common.base.Function;
import com.google.inject.Inject;
import com.google.inject.Provider;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.scoping.N4JSScopeProvider;
import org.eclipse.n4js.scoping.imports.ImportedElementsScopingHelper;
import org.eclipse.n4js.scoping.utils.ScopesHelper;
import org.eclipse.n4js.scoping.utils.WrongStaticAccessDescription;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.TStructMethod;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.xtext.resource.EObjectDescription;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.IScopeProvider;
import org.eclipse.xtext.scoping.Scopes;
import org.eclipse.xtext.scoping.impl.SingletonScope;
import org.eclipse.xtext.util.IResourceScopeCache;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Helper for {@link N4JSScopeProvider N4JSScopeProvider} using
 * {@link ImportedElementsScopingHelper ImportedElementsScopingHelper}
 * for providing scope for types provider.
 */
@SuppressWarnings("all")
public class LocallyKnownTypesScopingHelper {
  @Inject
  private IResourceScopeCache cache;
  
  @Inject
  private ImportedElementsScopingHelper importedElementsScopingHelper;
  
  @Inject
  private ScopesHelper scopesHelper;
  
  /**
   * Returns the type itself and type variables in case the type is generic.
   */
  public IScope scopeWithTypeAndItsTypeVariables(final IScope parent, final Type type, final boolean staticAccess) {
    IScope result = parent;
    if ((type != null)) {
      String _name = type.getName();
      boolean _tripleNotEquals = (_name != null);
      if (_tripleNotEquals) {
        IEObjectDescription _create = EObjectDescription.create(type.getName(), type);
        SingletonScope _singletonScope = new SingletonScope(_create, result);
        result = _singletonScope;
      }
      boolean _isGeneric = type.isGeneric();
      if (_isGeneric) {
        if (((type instanceof TClassifier) && staticAccess)) {
          final Function<IEObjectDescription, IEObjectDescription> _function = (IEObjectDescription it) -> {
            return new WrongStaticAccessDescription(it, staticAccess);
          };
          result = this.scopesHelper.scopeFor(type.getTypeVars(), _function, result);
        } else {
          result = Scopes.scopeFor(type.getTypeVars(), result);
        }
      }
    }
    return result;
  }
  
  /**
   * Returns the type variables if the TStructMethod is generic.
   */
  public IScope scopeWithTypeVarsOfTStructMethod(final IScope parent, final TStructMethod m) {
    final TStructMember mDef = m.getDefinedMember();
    if ((mDef instanceof TStructMethod)) {
      boolean _isGeneric = ((TStructMethod)mDef).isGeneric();
      if (_isGeneric) {
        return Scopes.scopeFor(((TStructMethod)mDef).getTypeVars(), parent);
      }
    }
    return parent;
  }
  
  /**
   * Returns the type variables if the function type expression is generic.
   */
  public IScope scopeWithTypeVarsOfFunctionTypeExpression(final IScope parent, final FunctionTypeExpression funTypeExpr) {
    if (((funTypeExpr != null) && funTypeExpr.isGeneric())) {
      return Scopes.scopeFor(funTypeExpr.getTypeVars(), parent);
    }
    return parent;
  }
  
  /**
   * Returns scope with locally known types and (as parent) import scope; the result is cached.
   */
  public IScope scopeWithLocallyKnownTypes(final Script script, final IScopeProvider delegate) {
    Pair<Script, String> _mappedTo = Pair.<Script, String>of(script, "locallyKnownTypes");
    final Provider<IScope> _function = () -> {
      final IScope parent = delegate.getScope(script, 
        TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__DECLARED_TYPE);
      final IScope importScope = this.importedElementsScopingHelper.getImportedTypes(parent, script);
      final IScope localTypes = this.scopeWithLocallyDeclaredTypes(script, importScope);
      return localTypes;
    };
    return this.cache.<IScope>get(_mappedTo, script.eResource(), _function);
  }
  
  /**
   * Returns scope with locally declared types (without import scope).
   */
  public IScope scopeWithLocallyDeclaredTypes(final Script script, final IScope parent) {
    final TModule local = script.getModule();
    if (((local == null) || local.eIsProxy())) {
      return parent;
    }
    final Function1<Type, IEObjectDescription> _function = (Type topLevelType) -> {
      return EObjectDescription.create(topLevelType.getName(), topLevelType);
    };
    return this.scopesHelper.mapBasedScopeFor(script, parent, ListExtensions.<Type, IEObjectDescription>map(local.getTopLevelTypes(), _function));
  }
  
  /**
   * Returns scope with locally known types specially configured for super reference in case of polyfill definitions.
   * It is comparable to {@link #getLocallyKnownTypes(Script, EReference, IScopeProvider), but it does not
   * add the polyfillType itself. Instead, only its type variables are added, which are otherwise hidden in case of polyfills.
   * The result is not cached as this scope is needed only one time.
   */
  public IScope scopeWithLocallyKnownTypesForPolyfillSuperRef(final Script script, final IScopeProvider delegate, final Type polyfillType) {
    final IScope parent = delegate.getScope(script, 
      TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__DECLARED_TYPE);
    final IScope importScope = this.importedElementsScopingHelper.getImportedTypes(parent, script);
    final TModule local = script.getModule();
    final Function1<Type, Boolean> _function = (Type it) -> {
      return Boolean.valueOf((it != polyfillType));
    };
    final Function1<Type, IEObjectDescription> _function_1 = (Type it) -> {
      return EObjectDescription.create(it.getName(), it);
    };
    final IScope localTypesScope = this.scopesHelper.mapBasedScopeFor(script, importScope, IterableExtensions.<Type, IEObjectDescription>map(IterableExtensions.<Type>filter(local.getTopLevelTypes(), _function), _function_1));
    boolean _isGeneric = polyfillType.isGeneric();
    if (_isGeneric) {
      return Scopes.scopeFor(polyfillType.getTypeVars(), localTypesScope);
    }
    return localTypesScope;
  }
}
