/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.collect.Iterables;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * A types builder that creates/relinks {@link ModuleNamespaceVirtualType} instances
 * based on given {@link ImportDeclaration}s.
 */
@SuppressWarnings("all")
public class N4JSNamespaceImportTypesBuilder {
  /**
   * Relinks the namespace types after reconciling the given {@link TModule}
   * 
   * Relinks existing instances of {@link ModuleNamespaceVirtualType} instances to the new AST.
   * 
   * If for a given {@link NamespaceImportSpecifier} no {@link ModuleNamespaceVirtualType}
   * instance exists yet, a new one is created and added to the module's {@link TModule#internalTypes}
   */
  public void relinkNamespaceTypes(final Script script, final TModule target, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      final Map<String, ModuleNamespaceVirtualType> namespaceTypesByName = this.getExistingNamespaceTypesByName(target);
      List<ImportDeclaration> _list = IterableExtensions.<ImportDeclaration>toList(Iterables.<ImportDeclaration>filter(script.getScriptElements(), ImportDeclaration.class));
      for (final ImportDeclaration importDeclaration : _list) {
        {
          final NamespaceImportSpecifier namespaceImport = this.getNamespaceImportSpecifier(importDeclaration);
          if ((namespaceImport != null)) {
            Object _eGet = importDeclaration.eGet(N4JSPackage.eINSTANCE.getImportDeclaration_Module(), false);
            final TModule importedModule = ((TModule) _eGet);
            final ModuleNamespaceVirtualType existingNamespaceType = namespaceTypesByName.get(namespaceImport.getAlias());
            if ((existingNamespaceType != null)) {
              this.relinkNamespaceType(existingNamespaceType, namespaceImport, importedModule);
            } else {
              EList<Type> _internalTypes = target.getInternalTypes();
              ModuleNamespaceVirtualType _createModuleNamespaceVirtualType = this.createModuleNamespaceVirtualType(namespaceImport, importedModule);
              _internalTypes.add(_createModuleNamespaceVirtualType);
            }
          }
        }
      }
    }
  }
  
  /**
   * Creates a new {@link ModuleNamespaceVirtualType} instance for the given {@link ImportDeclaration}
   * and return it.
   */
  public ModuleNamespaceVirtualType createModuleNamespaceVirtualType(final NamespaceImportSpecifier importSpecifier, final TModule importedModule) {
    final ModuleNamespaceVirtualType type = TypesFactory.eINSTANCE.createModuleNamespaceVirtualType();
    type.setName(importSpecifier.getAlias());
    type.setModule(importedModule);
    type.setDeclaredDynamic(importSpecifier.isDeclaredDynamic());
    type.setAstElement(importSpecifier);
    importSpecifier.setDefinedType(type);
    return type;
  }
  
  /**
   * Obtains the {@link NamespaceImportSpecifier} of the given {@code importDeclaration}.
   * 
   * Returns {@code null}, if the given import declaration does not represent a namespace import.
   */
  public NamespaceImportSpecifier getNamespaceImportSpecifier(final ImportDeclaration importDeclaration) {
    final List<NamespaceImportSpecifier> namespaceImportSpecifiers = IterableExtensions.<NamespaceImportSpecifier>toList(Iterables.<NamespaceImportSpecifier>filter(importDeclaration.getImportSpecifiers(), NamespaceImportSpecifier.class));
    boolean _isEmpty = namespaceImportSpecifiers.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      return IterableExtensions.<NamespaceImportSpecifier>head(namespaceImportSpecifiers);
    } else {
      return null;
    }
  }
  
  /**
   * Relinks the given {@link ModuleNamespaceVirtualType} to the given import specifier.
   */
  private void relinkNamespaceType(final ModuleNamespaceVirtualType namespaceType, final NamespaceImportSpecifier namespaceImportSpecifier, final TModule importedModule) {
    namespaceType.setModule(importedModule);
    namespaceType.setAstElement(namespaceImportSpecifier);
    namespaceImportSpecifier.setDefinedType(namespaceType);
  }
  
  /**
   * Returns a map of all existing {@link ModuleNamespaceVirtualType} contained in the
   * given {@link TModule}.
   * 
   * Includes exposed and non-exposed internal types.
   */
  private Map<String, ModuleNamespaceVirtualType> getExistingNamespaceTypesByName(final TModule module) {
    final HashMap<String, ModuleNamespaceVirtualType> namespaceTypesByName = new HashMap<String, ModuleNamespaceVirtualType>();
    EList<Type> _internalTypes = module.getInternalTypes();
    boolean _tripleNotEquals = (_internalTypes != null);
    if (_tripleNotEquals) {
      final Consumer<ModuleNamespaceVirtualType> _function = (ModuleNamespaceVirtualType type) -> {
        namespaceTypesByName.put(type.getName(), type);
      };
      Iterables.<ModuleNamespaceVirtualType>filter(module.getInternalTypes(), ModuleNamespaceVirtualType.class).forEach(_function);
    }
    EList<Type> _exposedInternalTypes = module.getExposedInternalTypes();
    boolean _tripleNotEquals_1 = (_exposedInternalTypes != null);
    if (_tripleNotEquals_1) {
      final Consumer<ModuleNamespaceVirtualType> _function_1 = (ModuleNamespaceVirtualType type) -> {
        namespaceTypesByName.put(type.getName(), type);
      };
      Iterables.<ModuleNamespaceVirtualType>filter(module.getExposedInternalTypes(), ModuleNamespaceVirtualType.class).forEach(_function_1);
    }
    return namespaceTypesByName;
  }
}
