/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.tests.helper;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.n4idl.migrations.MigrationSwitchComputer;
import org.eclipse.n4js.n4idl.migrations.OrSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.SwitchCondition;
import org.eclipse.n4js.n4idl.migrations.TypeSwitchCondition;
import org.eclipse.n4js.n4idl.tests.helper.N4IDLTypeRefTestHelper;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.junit.Assert;

/**
 * A test extension class for N4IDL-related tests, which allows to create valid
 * {@link TypeRef} instances from a given (N4IDL compliant) type-expression string.
 */
@SuppressWarnings("all")
public class AbstractN4IDLTypeSwitchTest extends Assert {
  @Inject
  protected MigrationSwitchComputer switchComputer;
  
  @Inject
  protected N4JSTypeSystem typeSystem;
  
  @Inject
  @Extension
  protected N4IDLTypeRefTestHelper _n4IDLTypeRefTestHelper;
  
  /**
   * Computes a {@link SwitchCondition} based on the given typeExpression using the given preamble.
   * 
   * Converts the resulting condition back to a {@link TypeRef} using {@link MigrationSwitchComputer#toTypeRef}
   * and returns the string representation of the result.
   */
  public String computeAndConvertToTypeRef(final String typeExpression, final String preamble) {
    try {
      final TypeRef typeRef = this._n4IDLTypeRefTestHelper.makeTypeRef(typeExpression, preamble);
      final RuleEnvironment env = this.typeSystem.createRuleEnvironmentForContext(typeRef, typeRef.eResource());
      final TypeRef result = this.switchComputer.toSwitchRecognizableTypeRef(env, typeRef);
      return result.getTypeRefAsString();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Creates a {@link TypeRef} for the given type expression string and
   * computes the corresponding {@link SwitchCondition} using {@link MigrationSwitchComputer}.
   * 
   * Prepends the given preamble to the N4IDL module before parsing it.
   */
  public SwitchCondition computeSwitch(final String typeExpression, final String preamble) {
    try {
      return this.switchComputer.compute(this._n4IDLTypeRefTestHelper.makeTypeRef(typeExpression, preamble));
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Returns the context {@link Resource} of the given {@link SwitchCondition} or
   * {@code null} if no context can be inferred.
   */
  protected Resource findContextResource(final SwitchCondition condition) {
    TypeSwitchCondition _head = IterableExtensions.<TypeSwitchCondition>head(Iterables.<TypeSwitchCondition>filter(condition, TypeSwitchCondition.class));
    Type _type = null;
    if (_head!=null) {
      _type=_head.type;
    }
    Resource _eResource = null;
    if (_type!=null) {
      _eResource=_type.eResource();
    }
    return _eResource;
  }
  
  protected RuleEnvironment createRuleEnvironment(final SwitchCondition condition) {
    return RuleEnvironmentExtensions.newRuleEnvironment(this.findContextResource(condition));
  }
  
  public TypeRef toTypeRef(final SwitchCondition condition) {
    final RuleEnvironment ruleEnv = RuleEnvironmentExtensions.newRuleEnvironment(this.findContextResource(condition));
    return this.switchComputer.toTypeRef(ruleEnv, condition);
  }
  
  /**
   * Creates a {@link TypeRef} for the given type expression string.
   * 
   * Prepends the given preamble to the N4IDL module before parsing it.
   * 
   * Computes the corresponding TypeSwitchCondition and returns its string representation.
   */
  public String compute(final String typeExpression, final String preamble) {
    return this.computeSwitch(typeExpression, preamble).toString();
  }
  
  /**
   * Assert that the given {@link SwitchCondition} is not an OrSwitchCondition nor
   * does it contain any sub-conditions of that type.
   */
  public void assertSwitchDoesNotContainOr(final SwitchCondition condition) {
    this.assertSwitchDoesNotContainOr(condition, condition);
  }
  
  private void assertSwitchDoesNotContainOr(final SwitchCondition rootCondition, final SwitchCondition condition) {
    if ((condition instanceof OrSwitchCondition)) {
      Assert.fail(("Switch condition did contain OR condition: " + rootCondition));
    }
    final Consumer<SwitchCondition> _function = (SwitchCondition c) -> {
      this.assertSwitchDoesNotContainOr(rootCondition, c);
    };
    condition.subConditions().forEach(_function);
  }
}
