/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.n4idl.assistants;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.eclipse.n4js.n4JS.BooleanLiteral;
import org.eclipse.n4js.n4JS.EqualityOperator;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.RelationalExpression;
import org.eclipse.n4js.n4JS.RelationalOperator;
import org.eclipse.n4js.n4JS.UnaryOperator;
import org.eclipse.n4js.n4idl.migrations.AndSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.ArrayTypeSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.ConstantSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.OrSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.SwitchCondition;
import org.eclipse.n4js.n4idl.migrations.TypeSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.TypeTypeCondition;
import org.eclipse.n4js.transpiler.TransformationAssistant;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.utils.ResourceNameComputer;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * A sub-transpiler which transforms {@link SwitchCondition}s to IM elements
 * and thus JavaScript conditions/expressions.
 * 
 * Use the method {@link #transform} to recursively trigger the transformation of a
 * given {@link SwitchCondition}.
 */
@SuppressWarnings("all")
public class TypeSwitchTranspiler extends TransformationAssistant {
  @Inject
  private ResourceNameComputer resourceNameComputer;
  
  /**
   * Transforms the given {@link SwitchCondition} to a corresponding IM model {@link Expression}.
   * 
   * @param lhs The operand of the switch condition. If used more than once, this element will be copied.
   */
  public List<Expression> transform(final SwitchCondition condition, final Expression lhs) {
    final Function1<Expression, Expression> _function = (Expression e) -> {
      return TranspilerBuilderBlocks._Parenthesis(e);
    };
    return ListExtensions.<Expression, Expression>map(this.doTransform(condition, this.<Expression>copy(lhs)), _function);
  }
  
  private List<Expression> _doTransform(final OrSwitchCondition or, final Expression lhs) {
    final Function1<SwitchCondition, List<Expression>> _function = (SwitchCondition op) -> {
      return this.transform(op, lhs);
    };
    return IterableExtensions.<Expression>toList(Iterables.<Expression>concat(ListExtensions.<SwitchCondition, List<Expression>>map(or.operands, _function)));
  }
  
  private List<Expression> _doTransform(final AndSwitchCondition and, final Expression lhs) {
    final Function1<SwitchCondition, List<Expression>> _function = (SwitchCondition op) -> {
      return this.transform(op, lhs);
    };
    final Function1<List<Expression>, Expression> _function_1 = (List<Expression> subExpressions) -> {
      return TranspilerBuilderBlocks._AND(subExpressions);
    };
    return ListExtensions.<List<Expression>, Expression>map(ListExtensions.<SwitchCondition, List<Expression>>map(and.operands, _function), _function_1);
  }
  
  private List<Expression> _doTransform(final ArrayTypeSwitchCondition arrayCondition, final Expression lhs) {
    final ParameterizedPropertyAccessExpression_IM arrayIsArray = TranspilerBuilderBlocks._PropertyAccessExpr(TranspilerBuilderBlocks._IdentRef(this.getSymbolTableEntryInternal("Array", true)), this.getSymbolTableEntryInternal("isArray", true));
    final ParameterizedCallExpression isArrayExpression = TranspilerBuilderBlocks._CallExpr(arrayIsArray, this.<Expression>copy(lhs));
    final RelationalExpression notEmptyExpression = TranspilerBuilderBlocks._RelationalExpr(TranspilerBuilderBlocks._PropertyAccessExpr(this.<Expression>copy(lhs), this.getSymbolTableEntryInternal("length", true)), RelationalOperator.GT, TranspilerBuilderBlocks._NumericLiteral(0));
    final List<Expression> elementTypeExpression = this.transform(arrayCondition.elementTypeCondition, TranspilerBuilderBlocks._IndexAccessExpr(this.<Expression>copy(lhs), TranspilerBuilderBlocks._NumericLiteral(0)));
    Iterable<Expression> _plus = Iterables.<Expression>concat(Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(isArrayExpression, notEmptyExpression)), elementTypeExpression);
    Expression __AND = TranspilerBuilderBlocks._AND(_plus);
    return Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(__AND));
  }
  
  private List<Expression> _doTransform(final TypeTypeCondition typeTypeCondition, final Expression lhs) {
    ParameterizedPropertyAccessExpression_IM __PropertyAccessExpr = TranspilerBuilderBlocks._PropertyAccessExpr(lhs, this.getSymbolTableEntryInternal("n4type", true));
    return Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(__PropertyAccessExpr));
  }
  
  private List<Expression> _doTransform(final TypeSwitchCondition typeCondition, final Expression lhs) {
    Expression _runtimeTypeCheck = this.runtimeTypeCheck(typeCondition.type, lhs);
    return Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(_runtimeTypeCheck));
  }
  
  /**
   * Creates a runtime type-check expression for the given type and lhs.
   */
  private Expression _runtimeTypeCheck(final TClassifier type, final Expression lhs) {
    final SymbolTableEntryOriginal typeSTE = this.getState().steCache.mapOriginal.get(type);
    return TranspilerBuilderBlocks._RelationalExpr(lhs, RelationalOperator.INSTANCEOF, TranspilerBuilderBlocks._IdentRef(typeSTE));
  }
  
  private Expression _runtimeTypeCheck(final TInterface tInterface, final Expression lhs) {
    ParameterizedCallExpression _xblockexpression = null;
    {
      final SymbolTableEntryInternal $implementsSTE = this.steFor_$implements();
      final String fqn = this.resourceNameComputer.getFullyQualifiedTypeName_WITH_LEGACY_SUPPORT(tInterface);
      _xblockexpression = TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._IdentRef($implementsSTE), lhs, TranspilerBuilderBlocks._StringLiteral(fqn));
    }
    return _xblockexpression;
  }
  
  /**
   * @see {@link #runtimeTypeCheck(TClassifier, Expression) }
   */
  private Expression _runtimeTypeCheck(final PrimitiveType type, final Expression lhs) {
    Expression _switchResult = null;
    String _name = type.getName();
    if (_name != null) {
      switch (_name) {
        case "any":
          _switchResult = TranspilerBuilderBlocks._TRUE();
          break;
        case "int":
          _switchResult = this._TypeOfCheck(lhs, "number");
          break;
        case "number":
          _switchResult = this._TypeOfCheck(lhs, "number");
          break;
        case "string":
          _switchResult = this._TypeOfCheck(lhs, "string");
          break;
        case "boolean":
          _switchResult = this._TypeOfCheck(lhs, "boolean");
          break;
        default:
          throw new IllegalStateException(("Unhandled primitive type in TypeSwitchTranspiler: " + type));
      }
    } else {
      throw new IllegalStateException(("Unhandled primitive type in TypeSwitchTranspiler: " + type));
    }
    return _switchResult;
  }
  
  private Expression _runtimeTypeCheck(final Type type, final Expression lhs) {
    throw new IllegalStateException(("Cannot produce runtime type-check for type " + type));
  }
  
  /**
   * Creates a new typeof check using the given lhs and typeofResult:
   * {@code typeof <lhs> === "<typeofResult>"}.
   * 
   * @param lhs The left-hand side to check the type of
   * @param typeofResult The desired result of the typeof operator
   */
  private Expression _TypeOfCheck(final Expression lhs, final String typeofResult) {
    return TranspilerBuilderBlocks._EqualityExpr(TranspilerBuilderBlocks._UnaryExpr(UnaryOperator.TYPEOF, lhs), EqualityOperator.SAME, TranspilerBuilderBlocks._StringLiteral(typeofResult));
  }
  
  /**
   * @see {@link #runtimeTypeCheck(TClassifier, Expression) }
   */
  private List<Expression> _doTransform(final ConstantSwitchCondition constantCondition, final Expression lhs) {
    boolean _equals = constantCondition.constant.equals("true");
    if (_equals) {
      BooleanLiteral __TRUE = TranspilerBuilderBlocks._TRUE();
      return Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(__TRUE));
    } else {
      boolean _equals_1 = constantCondition.constant.equals("true");
      if (_equals_1) {
        BooleanLiteral __FALSE = TranspilerBuilderBlocks._FALSE();
        return Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(__FALSE));
      } else {
        throw new IllegalStateException((("Unhandled ConstantSwitchCondition with constant \'" + constantCondition.constant) + "\'"));
      }
    }
  }
  
  private List<Expression> _doTransform(final SwitchCondition unhandledCondition, final Expression lhs) {
    String _simpleName = unhandledCondition.getClass().getSimpleName();
    String _plus = ("Encountered unhandled switch-condition of type " + _simpleName);
    String _plus_1 = (_plus + " in transpiler: ");
    String _string = unhandledCondition.toString();
    String _plus_2 = (_plus_1 + _string);
    throw new IllegalStateException(_plus_2);
  }
  
  private List<Expression> doTransform(final SwitchCondition and, final Expression lhs) {
    if (and instanceof AndSwitchCondition) {
      return _doTransform((AndSwitchCondition)and, lhs);
    } else if (and instanceof ArrayTypeSwitchCondition) {
      return _doTransform((ArrayTypeSwitchCondition)and, lhs);
    } else if (and instanceof ConstantSwitchCondition) {
      return _doTransform((ConstantSwitchCondition)and, lhs);
    } else if (and instanceof OrSwitchCondition) {
      return _doTransform((OrSwitchCondition)and, lhs);
    } else if (and instanceof TypeSwitchCondition) {
      return _doTransform((TypeSwitchCondition)and, lhs);
    } else if (and instanceof TypeTypeCondition) {
      return _doTransform((TypeTypeCondition)and, lhs);
    } else if (and != null) {
      return _doTransform(and, lhs);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(and, lhs).toString());
    }
  }
  
  private Expression runtimeTypeCheck(final Type tInterface, final Expression lhs) {
    if (tInterface instanceof TInterface) {
      return _runtimeTypeCheck((TInterface)tInterface, lhs);
    } else if (tInterface instanceof PrimitiveType) {
      return _runtimeTypeCheck((PrimitiveType)tInterface, lhs);
    } else if (tInterface instanceof TClassifier) {
      return _runtimeTypeCheck((TClassifier)tInterface, lhs);
    } else if (tInterface != null) {
      return _runtimeTypeCheck(tInterface, lhs);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(tInterface, lhs).toString());
    }
  }
}
