/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.organize.imports;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Iterator;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.documentation.N4JSDocumentationProvider;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.parser.InternalSemicolonInjectingParser;
import org.eclipse.n4js.ts.services.TypeExpressionsGrammarAccess;
import org.eclipse.n4js.ui.organize.imports.InsertionPoint;
import org.eclipse.n4js.ui.organize.imports.XtextResourceUtils;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.xtext.TerminalRule;
import org.eclipse.xtext.nodemodel.BidiTreeIterator;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.ILeafNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.impl.HiddenLeafNode;
import org.eclipse.xtext.nodemodel.impl.LeafNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Helper used to calculate imports region in the resource.
 */
@SuppressWarnings("all")
public class ImportsRegionHelper {
  @Inject
  private N4JSDocumentationProvider documentationProvider;
  
  @Inject
  private TypeExpressionsGrammarAccess typeExpressionGrammmarAccess;
  
  /**
   * Calculates import offset by analyzing the provided resource.
   * @See {@link #getImportRegion(Script)}
   */
  public int getImportOffset(final XtextResource resource) {
    return this.getImportRegion(resource).offset;
  }
  
  /**
   * Calculates import offset by analyzing the provided script.
   * @See {@link #getImportRegion(Script)}
   */
  public int getImportOffset(final Script script) {
    return this.getImportRegion(script).offset;
  }
  
  /**
   * Calculates import region by analyzing the provided resource.
   * @See {@link #getImportRegion(Script)}
   */
  InsertionPoint getImportRegion(final XtextResource xtextResource) {
    return this.getImportRegion(XtextResourceUtils.getScript(xtextResource));
  }
  
  /**
   * Calculate destination region in Document for imports. If the offset is not 0,
   * then it has to be advanced by current line feed length
   * 
   * Note reduced visibility - it is internal method used only by helpers related to organizing imports.
   * 
   * Using first position after script-annotation ("@@") and after any directive in the prolog section, that is
   * just before the first statement or, in cases where the first statement is js-style documented, before the jsdoc-style
   * documentation.
   * 
   * Examples:
   * <pre>
   *  // (A)
   *  // (B)
   *  &#64;&#64;StaticPolyfillAware
   *  // (C)
   *  "use strict";
   *  // (D)
   *  /&#42; non-jsdoc comment (E) &#42;/
   *  /&#42;&#42; jsdoc comment (F) &#42;/
   *  /&#42; non-jsdoc comment (G) &#42;/
   *  // (H)
   *  export public class A { // (I)
   *     method(): B {
   *        return new B(); // requires import
   *    }
   *  }
   * </pre>
   * Will put the insertion-point in front of line (F), since this is the active jsdoc for class A.
   * {@link InsertionPoint#isBeforeJsdocDocumentation} will be set to true. Lowest possible insertion
   * is the begin of line (D), stored in {@link InsertionPoint#notBeforeTotalOffset}. If the directive <code>"use strict";</code>
   * between lines (C) and (D) is omitted, then the lowest insertion point would be in front of line (C). In any case the insertion of
   * the import must be in front of the <code>export</code> keyword line (I).
   * 
   * <p>Region has length 0.
   * 
   * @param xtextResource
   *            n4js resource
   * @return region for import statements, length 0
   */
  InsertionPoint getImportRegion(final Script script) {
    final InsertionPoint insertionPoint = new InsertionPoint();
    int begin = (-1);
    if ((script != null)) {
      begin = 0;
      final List<INode> scriptAnnos = NodeModelUtils.findNodesForFeature(script, N4JSPackage.Literals.SCRIPT__ANNOTATIONS);
      boolean _isEmpty = scriptAnnos.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        int _size = scriptAnnos.size();
        int _minus = (_size - 1);
        final INode lastAnno = scriptAnnos.get(_minus);
        begin = lastAnno.getTotalEndOffset();
        insertionPoint.notBeforeTotalOffset = begin;
      }
      final EList<ScriptElement> elements = script.getScriptElements();
      int lastSeenDirective = (-1);
      int idxNondirectiveStatemnt = (-1);
      for (int i = 0; ((i < elements.size()) && (idxNondirectiveStatemnt == (-1))); i++) {
        {
          final ScriptElement curr = elements.get(i);
          boolean _isStringLiteralExpression = N4JSASTUtils.isStringLiteralExpression(curr);
          if (_isStringLiteralExpression) {
            lastSeenDirective = i;
          } else {
            if ((curr instanceof ImportDeclaration)) {
            } else {
              idxNondirectiveStatemnt = i;
            }
          }
        }
      }
      if ((idxNondirectiveStatemnt != (-1))) {
        final ScriptElement realScriptElement = elements.get(idxNondirectiveStatemnt);
        final ICompositeNode realScriptElementNode = NodeModelUtils.findActualNodeFor(realScriptElement);
        final List<INode> docuNodes = this.documentationProvider.getDocumentationNodes(realScriptElement);
        boolean _isEmpty_1 = docuNodes.isEmpty();
        boolean _not_1 = (!_isEmpty_1);
        if (_not_1) {
          final INode docuNode = docuNodes.get(0);
          INode previousNode = docuNode;
          INode lastEOL = null;
          boolean continue_ = true;
          while (((continue_ && previousNode.hasPreviousSibling()) && 
            (previousNode.getPreviousSibling() instanceof HiddenLeafNode))) {
            {
              final EObject grammar = previousNode.getPreviousSibling().getGrammarElement();
              TerminalRule _wSRule = this.typeExpressionGrammmarAccess.getWSRule();
              boolean _equals = Objects.equal(grammar, _wSRule);
              if (_equals) {
                previousNode = previousNode.getPreviousSibling();
              } else {
                TerminalRule _eOLRule = this.typeExpressionGrammmarAccess.getEOLRule();
                boolean _equals_1 = Objects.equal(grammar, _eOLRule);
                if (_equals_1) {
                  previousNode = previousNode.getPreviousSibling();
                  lastEOL = previousNode;
                } else {
                  continue_ = false;
                }
              }
            }
          }
          int _xifexpression = (int) 0;
          if ((lastEOL != null)) {
            _xifexpression = lastEOL.getEndOffset();
          } else {
            _xifexpression = docuNode.getTotalOffset();
          }
          begin = _xifexpression;
          insertionPoint.isBeforeJsdocDocumentation = true;
        } else {
          List<ILeafNode> listLeafNodes = IterableExtensions.<ILeafNode>toList(realScriptElementNode.getLeafNodes());
          {
            final Iterator<ILeafNode> iterLeaves = listLeafNodes.iterator();
            ILeafNode curr = null;
            ILeafNode firstEOL = null;
            ILeafNode afterFirstEOL = null;
            boolean sawComment = false;
            ILeafNode lastComment = null;
            while ((iterLeaves.hasNext() && (curr = iterLeaves.next()).isHidden())) {
              EObject _grammarElement = curr.getGrammarElement();
              if ((_grammarElement instanceof TerminalRule)) {
                EObject _grammarElement_1 = curr.getGrammarElement();
                TerminalRule _mL_COMMENTRule = this.typeExpressionGrammmarAccess.getML_COMMENTRule();
                boolean _equals = Objects.equal(_grammarElement_1, _mL_COMMENTRule);
                if (_equals) {
                  firstEOL = null;
                  afterFirstEOL = null;
                  sawComment = true;
                  lastComment = curr;
                } else {
                  EObject _grammarElement_2 = curr.getGrammarElement();
                  TerminalRule _sL_COMMENTRule = this.typeExpressionGrammmarAccess.getSL_COMMENTRule();
                  boolean _equals_1 = Objects.equal(_grammarElement_2, _sL_COMMENTRule);
                  if (_equals_1) {
                    firstEOL = null;
                    afterFirstEOL = null;
                    sawComment = true;
                    lastComment = curr;
                  } else {
                    EObject _grammarElement_3 = curr.getGrammarElement();
                    TerminalRule _eOLRule = this.typeExpressionGrammmarAccess.getEOLRule();
                    boolean _equals_2 = Objects.equal(_grammarElement_3, _eOLRule);
                    if (_equals_2) {
                      if ((firstEOL == null)) {
                        firstEOL = curr;
                      } else {
                        if ((afterFirstEOL == null)) {
                          afterFirstEOL = curr;
                        }
                      }
                    } else {
                      EObject _grammarElement_4 = curr.getGrammarElement();
                      TerminalRule _wSRule = this.typeExpressionGrammmarAccess.getWSRule();
                      boolean _equals_3 = Objects.equal(_grammarElement_4, _wSRule);
                      if (_equals_3) {
                        if (((firstEOL != null) && (afterFirstEOL == null))) {
                          afterFirstEOL = curr;
                        }
                      } else {
                        firstEOL = null;
                        afterFirstEOL = null;
                      }
                    }
                  }
                }
              }
            }
            if (((curr == null) || curr.isHidden())) {
              throw new RuntimeException("Expected at least one non-hidden element.");
            }
            insertionPoint.notAfterTotalOffset = curr.getTotalOffset();
            int _xifexpression_1 = (int) 0;
            if (((afterFirstEOL != null) && sawComment)) {
              _xifexpression_1 = afterFirstEOL.getTotalOffset();
            } else {
              int _xifexpression_2 = (int) 0;
              boolean _hasNoCommentUpTo = this.hasNoCommentUpTo(curr);
              if (_hasNoCommentUpTo) {
                _xifexpression_2 = 0;
              } else {
                int _xifexpression_3 = (int) 0;
                if (sawComment) {
                  _xifexpression_3 = lastComment.getEndOffset();
                } else {
                  _xifexpression_3 = IterableExtensions.<ILeafNode>head(listLeafNodes).getTotalOffset();
                }
                _xifexpression_2 = _xifexpression_3;
              }
              _xifexpression_1 = _xifexpression_2;
            }
            int begin2 = _xifexpression_1;
            begin = Math.max(begin, begin2);
          }
          if ((lastSeenDirective > (-1))) {
            final ICompositeNode lastDirectiveNode = NodeModelUtils.findActualNodeFor(elements.get(lastSeenDirective));
            final int lastDirectiveEndOffset = lastDirectiveNode.getTotalEndOffset();
            insertionPoint.notBeforeTotalOffset = Math.max(lastDirectiveEndOffset, 
              insertionPoint.notBeforeTotalOffset);
            begin = Math.max(begin, lastDirectiveEndOffset);
          }
        }
      } else {
        if ((lastSeenDirective > (-1))) {
          final ICompositeNode lastDirectiveNode_1 = NodeModelUtils.findActualNodeFor(elements.get(lastSeenDirective));
          begin = lastDirectiveNode_1.getTotalEndOffset();
          insertionPoint.notBeforeTotalOffset = Math.max(begin, insertionPoint.notBeforeTotalOffset);
        } else {
        }
      }
      insertionPoint.offset = begin;
    }
    return insertionPoint;
  }
  
  /**
   * Goes from the beginning of the RootNode up to the passed in node. Looks only at hidden leafs and at ASI-LeafNodes.
   * @return {@code false} if any comment is encountered on the way.
   */
  private boolean hasNoCommentUpTo(final ILeafNode node) {
    if ((node == null)) {
      return true;
    }
    final BidiTreeIterator<INode> iter = node.getRootNode().getAsTreeIterable().iterator();
    while (iter.hasNext()) {
      {
        final INode curr = iter.next();
        boolean _equals = Objects.equal(curr, node);
        if (_equals) {
          return true;
        }
        if ((curr instanceof LeafNode)) {
          if ((((LeafNode)curr).isHidden() || 
            UtilN4.isIgnoredSyntaxErrorNode(curr, InternalSemicolonInjectingParser.SEMICOLON_INSERTED))) {
            boolean _isEmpty = ((LeafNode)curr).getText().trim().isEmpty();
            boolean _not = (!_isEmpty);
            if (_not) {
              return false;
            }
          }
        }
      }
    }
    throw new IllegalStateException("Iteration over-stepped the passed in node.");
  }
}
