/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.organize.imports;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.ui.changes.ChangeProvider;
import org.eclipse.n4js.ui.changes.IChange;
import org.eclipse.n4js.ui.organize.imports.XtextResourceUtils;
import org.eclipse.xtext.RuleCall;
import org.eclipse.xtext.nodemodel.BidiTreeIterator;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.ILeafNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;

/**
 * Utility that will calculate changes to remove all imports from {@link XtextResource}.
 */
@SuppressWarnings("all")
public class ImportsRemovalChangesComputer {
  /**
   * Compute changes that will remove all imports.
   * 
   * @param resource the resource to modify
   * @param document the document connected to the xtextResource,  for textual changes.
   * @return list of changes to the document.
   */
  public static List<IChange> getImportDeletionChanges(final XtextResource resource, final IXtextDocument document) throws BadLocationException {
    final ArrayList<IChange> changes = CollectionLiterals.<IChange>newArrayList();
    final EList<ScriptElement> elements = XtextResourceUtils.getScript(resource).getScriptElements();
    for (final ScriptElement el : elements) {
      if ((el instanceof ImportDeclaration)) {
        final ICompositeNode nodeToRemove = NodeModelUtils.findActualNodeFor(el);
        changes.add(ImportsRemovalChangesComputer.removeNodeButKeepComments(document, nodeToRemove));
      }
    }
    return changes;
  }
  
  private static IChange removeNodeButKeepComments(final IXtextDocument doc, final INode importNode) throws BadLocationException {
    if ((importNode == null)) {
      return IChange.IDENTITY;
    }
    final BidiTreeIterator<INode> iterator = importNode.getAsTreeIterable().iterator();
    int end = importNode.getEndOffset();
    boolean done = false;
    while (((!done) && iterator.hasPrevious())) {
      {
        final INode prev = iterator.previous();
        if ((prev instanceof ILeafNode)) {
          boolean _isHidden = ((ILeafNode)prev).isHidden();
          boolean _not = (!_isHidden);
          if (_not) {
            boolean _equals = ";".equals(((ILeafNode)prev).getText());
            boolean _not_1 = (!_equals);
            if (_not_1) {
              final EObject grammarElement = ((ILeafNode)prev).getGrammarElement();
              if ((grammarElement instanceof RuleCall)) {
                boolean _equals_1 = "STRING".equals(((RuleCall)grammarElement).getRule().getName());
                if (_equals_1) {
                  done = true;
                }
              }
              while (((!done) && iterator.hasPrevious())) {
                {
                  final INode sibling = iterator.previous();
                  if ((sibling instanceof ILeafNode)) {
                    boolean _isHidden_1 = ((ILeafNode)sibling).isHidden();
                    boolean _not_2 = (!_isHidden_1);
                    if (_not_2) {
                      done = true;
                      end = ((ILeafNode)sibling).getEndOffset();
                    }
                  }
                }
              }
            }
          } else {
            while (((!done) && iterator.hasPrevious())) {
              {
                final INode sibling = iterator.previous();
                if ((sibling instanceof ILeafNode)) {
                  boolean _isHidden_1 = ((ILeafNode)sibling).isHidden();
                  boolean _not_2 = (!_isHidden_1);
                  if (_not_2) {
                    done = true;
                    end = ((ILeafNode)sibling).getEndOffset();
                  }
                }
              }
            }
          }
        }
      }
    }
    final int offset = importNode.getOffset();
    return ChangeProvider.removeText(doc, offset, (end - offset), true);
  }
}
