/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.json.formatting2;

import com.google.common.base.Objects;
import java.util.Arrays;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.json.JSON.JSONArray;
import org.eclipse.n4js.json.JSON.JSONDocument;
import org.eclipse.n4js.json.JSON.JSONObject;
import org.eclipse.n4js.json.JSON.JSONValue;
import org.eclipse.n4js.json.JSON.NameValuePair;
import org.eclipse.xtext.formatting2.AbstractFormatter2;
import org.eclipse.xtext.formatting2.IFormattableDocument;
import org.eclipse.xtext.formatting2.IHiddenRegionFormatter;
import org.eclipse.xtext.formatting2.regionaccess.ISemanticRegion;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;

/**
 * A simple formatter for JSON files.
 * 
 * Generally, puts name-value-pairs of objects and array elements
 * on a separate line.
 */
@SuppressWarnings("all")
public class JSONFormatter extends AbstractFormatter2 {
  protected void _format(final JSONDocument jSONDocument, @Extension final IFormattableDocument document) {
    document.<JSONValue>format(jSONDocument.getContent());
    final Consumer<ISemanticRegion> _function = (ISemanticRegion it) -> {
      final Procedure1<IHiddenRegionFormatter> _function_1 = (IHiddenRegionFormatter it_1) -> {
        it_1.indent();
      };
      document.append(it, _function_1);
    };
    this.textRegionExtensions.allSemanticRegions(jSONDocument).forEach(_function);
  }
  
  /**
   * Put both brackets and every element of a JSONArray on a separate line.
   */
  protected void _format(final JSONArray al, @Extension final IFormattableDocument document) {
    this.configureCommas(al, document);
    final Pair<ISemanticRegion, ISemanticRegion> bracketPair = IterableExtensions.<Pair<ISemanticRegion, ISemanticRegion>>head(this.textRegionExtensions.regionFor(al).keywordPairs("[", "]"));
    if ((bracketPair != null)) {
      final Procedure1<IHiddenRegionFormatter> _function = (IHiddenRegionFormatter it) -> {
        it.indent();
      };
      document.<ISemanticRegion, ISemanticRegion>interior(bracketPair, _function);
      boolean _isEmpty = al.getElements().isEmpty();
      if (_isEmpty) {
        final Procedure1<IHiddenRegionFormatter> _function_1 = (IHiddenRegionFormatter it) -> {
          it.noSpace();
        };
        document.append(bracketPair.getKey(), _function_1);
        final Procedure1<IHiddenRegionFormatter> _function_2 = (IHiddenRegionFormatter it) -> {
          it.noSpace();
        };
        document.prepend(bracketPair.getValue(), _function_2);
        return;
      }
      final Procedure1<IHiddenRegionFormatter> _function_3 = (IHiddenRegionFormatter it) -> {
        it.newLine();
      };
      document.prepend(bracketPair.getValue(), _function_3);
    }
    final Procedure2<JSONValue, Integer> _function_4 = (JSONValue it, Integer num) -> {
      final Procedure1<IHiddenRegionFormatter> _function_5 = (IHiddenRegionFormatter it_1) -> {
        it_1.newLine();
      };
      document.<JSONValue>prepend(it, _function_5);
    };
    IterableExtensions.<JSONValue>forEach(al.getElements(), _function_4);
    final Consumer<JSONValue> _function_5 = (JSONValue it) -> {
      document.<JSONValue>format(it);
    };
    al.getElements().forEach(_function_5);
  }
  
  /**
   * On the direct level of an semantic Object enforce commas to ", " with autoWrap option.
   */
  private void configureCommas(final EObject semEObject, @Extension final IFormattableDocument document) {
    final Consumer<ISemanticRegion> _function = (ISemanticRegion it) -> {
      final Procedure1<IHiddenRegionFormatter> _function_1 = (IHiddenRegionFormatter it_1) -> {
        it_1.noSpace();
      };
      document.prepend(it, _function_1);
      final Procedure1<IHiddenRegionFormatter> _function_2 = (IHiddenRegionFormatter it_1) -> {
        it_1.oneSpace();
        it_1.autowrap();
      };
      document.append(it, _function_2);
    };
    this.textRegionExtensions.regionFor(semEObject).keywords(",").forEach(_function);
  }
  
  /**
   * Put both curly braces and every name-value-pair of a JSONObject on a separate line.
   */
  protected void _format(final JSONObject ol, @Extension final IFormattableDocument document) {
    this.configureCommas(ol, document);
    final Pair<ISemanticRegion, ISemanticRegion> bracePair = IterableExtensions.<Pair<ISemanticRegion, ISemanticRegion>>head(this.textRegionExtensions.regionFor(ol).keywordPairs("{", "}"));
    if ((bracePair != null)) {
      if (bracePair!=null) {
        final Procedure1<IHiddenRegionFormatter> _function = (IHiddenRegionFormatter it) -> {
          it.indent();
        };
        document.<ISemanticRegion, ISemanticRegion>interior(bracePair, _function);
      }
      boolean _isEmpty = ol.getNameValuePairs().isEmpty();
      if (_isEmpty) {
        final Procedure1<IHiddenRegionFormatter> _function_1 = (IHiddenRegionFormatter it) -> {
          it.noSpace();
        };
        document.append(bracePair.getKey(), _function_1);
        final Procedure1<IHiddenRegionFormatter> _function_2 = (IHiddenRegionFormatter it) -> {
          it.noSpace();
        };
        document.prepend(bracePair.getValue(), _function_2);
        return;
      }
      ISemanticRegion _value = null;
      if (bracePair!=null) {
        _value=bracePair.getValue();
      }
      final Procedure1<IHiddenRegionFormatter> _function_3 = (IHiddenRegionFormatter it) -> {
        it.newLine();
      };
      document.prepend(_value, _function_3);
      final Procedure2<NameValuePair, Integer> _function_4 = (NameValuePair it, Integer num) -> {
        final Procedure1<IHiddenRegionFormatter> _function_5 = (IHiddenRegionFormatter it_1) -> {
          it_1.newLine();
        };
        document.<NameValuePair>prepend(it, _function_5);
      };
      IterableExtensions.<NameValuePair>forEach(ol.getNameValuePairs(), _function_4);
      ISemanticRegion _key = null;
      if (bracePair!=null) {
        _key=bracePair.getKey();
      }
      ISemanticRegion _nextSemanticRegion = null;
      if (_key!=null) {
        _nextSemanticRegion=_key.getNextSemanticRegion();
      }
      ISemanticRegion _value_1 = null;
      if (bracePair!=null) {
        _value_1=bracePair.getValue();
      }
      boolean _equals = Objects.equal(_nextSemanticRegion, _value_1);
      if (_equals) {
        final Procedure1<IHiddenRegionFormatter> _function_5 = (IHiddenRegionFormatter it) -> {
          it.newLine();
        };
        document.append(bracePair.getKey(), _function_5);
      }
    }
    final Consumer<NameValuePair> _function_6 = (NameValuePair it) -> {
      document.<NameValuePair>format(it);
    };
    ol.getNameValuePairs().forEach(_function_6);
  }
  
  protected void _format(final NameValuePair nameValuePair, @Extension final IFormattableDocument document) {
    final ISemanticRegion colon = this.textRegionExtensions.regionFor(nameValuePair).keyword(":");
    final JSONValue value = nameValuePair.getValue();
    final Procedure1<IHiddenRegionFormatter> _function = (IHiddenRegionFormatter it) -> {
      it.noSpace();
    };
    document.prepend(colon, _function);
    final Procedure1<IHiddenRegionFormatter> _function_1 = (IHiddenRegionFormatter it) -> {
      it.oneSpace();
    };
    document.append(colon, _function_1);
    document.<JSONValue>format(value);
  }
  
  public void format(final Object al, final IFormattableDocument document) {
    if (al instanceof XtextResource) {
      _format((XtextResource)al, document);
      return;
    } else if (al instanceof JSONArray) {
      _format((JSONArray)al, document);
      return;
    } else if (al instanceof JSONObject) {
      _format((JSONObject)al, document);
      return;
    } else if (al instanceof JSONDocument) {
      _format((JSONDocument)al, document);
      return;
    } else if (al instanceof NameValuePair) {
      _format((NameValuePair)al, document);
      return;
    } else if (al instanceof EObject) {
      _format((EObject)al, document);
      return;
    } else if (al == null) {
      _format((Void)null, document);
      return;
    } else if (al != null) {
      _format(al, document);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(al, document).toString());
    }
  }
}
