/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.organize.imports;

import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.organize.imports.ImportProvidedElement;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Utilities for ImportSpecifiers
 */
@SuppressWarnings("all")
public class ImportSpecifiersUtil {
  /**
   * @return {@link List} of {@link ImportProvidedElement}s describing imported elements
   */
  public static List<ImportProvidedElement> mapToImportProvidedElements(final List<ImportSpecifier> importSpecifiers) {
    final Function1<ImportSpecifier, List<ImportProvidedElement>> _function = (ImportSpecifier specifier) -> {
      boolean _matched = false;
      if (specifier instanceof NamespaceImportSpecifier) {
        _matched=true;
        return ImportSpecifiersUtil.namespaceToProvidedElements(((NamespaceImportSpecifier)specifier));
      }
      if (!_matched) {
        if (specifier instanceof NamedImportSpecifier) {
          _matched=true;
          String _usedName = ImportSpecifiersUtil.usedName(((NamedImportSpecifier)specifier));
          String _importedElementName = ImportSpecifiersUtil.importedElementName(((NamedImportSpecifier)specifier));
          ImportProvidedElement _importProvidedElement = new ImportProvidedElement(_usedName, _importedElementName, 
            ((ImportSpecifier) specifier));
          return CollectionLiterals.<ImportProvidedElement>newArrayList(_importProvidedElement);
        }
      }
      return CollectionLiterals.<ImportProvidedElement>emptyList();
    };
    return IterableExtensions.<ImportProvidedElement>toList(Iterables.<ImportProvidedElement>concat(ListExtensions.<ImportSpecifier, List<ImportProvidedElement>>map(importSpecifiers, _function)));
  }
  
  /**
   * Map all exported elements from namespace target module to the import provided elements.
   */
  private static List<ImportProvidedElement> namespaceToProvidedElements(final NamespaceImportSpecifier specifier) {
    TModule _importedModule = ImportSpecifiersUtil.importedModule(specifier);
    boolean _tripleEquals = (_importedModule == null);
    if (_tripleEquals) {
      return CollectionLiterals.<ImportProvidedElement>emptyList();
    }
    final ArrayList<ImportProvidedElement> importProvidedElements = CollectionLiterals.<ImportProvidedElement>newArrayList();
    String _alias = specifier.getAlias();
    String _computeNamespaceActualName = ImportSpecifiersUtil.computeNamespaceActualName(specifier);
    ImportProvidedElement _importProvidedElement = new ImportProvidedElement(_alias, _computeNamespaceActualName, specifier);
    importProvidedElements.add(_importProvidedElement);
    final Function1<Type, Boolean> _function = (Type it) -> {
      return Boolean.valueOf(it.isExported());
    };
    final Function1<Type, TExportableElement> _function_1 = (Type it) -> {
      return ((TExportableElement) it);
    };
    final Iterable<TExportableElement> topExportedTypes = IterableExtensions.<Type, TExportableElement>map(IterableExtensions.<Type>filter(ImportSpecifiersUtil.importedModule(specifier).getTopLevelTypes(), _function), _function_1);
    final Function1<TVariable, Boolean> _function_2 = (TVariable it) -> {
      return Boolean.valueOf(it.isExported());
    };
    final Function1<TVariable, TExportableElement> _function_3 = (TVariable it) -> {
      return ((TExportableElement) it);
    };
    final Iterable<TExportableElement> topExportedVars = IterableExtensions.<TVariable, TExportableElement>map(IterableExtensions.<TVariable>filter(ImportSpecifiersUtil.importedModule(specifier).getVariables(), _function_2), _function_3);
    final Iterable<TExportableElement> topExported = Iterables.<TExportableElement>concat(topExportedTypes, topExportedVars);
    final Consumer<TExportableElement> _function_4 = (TExportableElement type) -> {
      String _importedElementName = ImportSpecifiersUtil.importedElementName(specifier, type);
      String _exportedName = type.getExportedName();
      ImportProvidedElement _importProvidedElement_1 = new ImportProvidedElement(_importedElementName, _exportedName, 
        ((ImportSpecifier) specifier));
      importProvidedElements.add(_importProvidedElement_1);
    };
    topExported.forEach(_function_4);
    return importProvidedElements;
  }
  
  /**
   * Computes 'actual' name of the namespace for {@link ImportProvidedElement} entry.
   * If processed namespace refers to unresolved module, will return dummy name,
   * otherwise returns artificial name composed of prefix and target module qualified name
   */
  public static String computeNamespaceActualName(final NamespaceImportSpecifier specifier) {
    String _xifexpression = null;
    boolean _eIsProxy = ImportSpecifiersUtil.importedModule(specifier).eIsProxy();
    if (_eIsProxy) {
      int _hashCode = specifier.hashCode();
      _xifexpression = (ImportProvidedElement.NAMESPACE_PREFIX + Integer.valueOf(_hashCode));
    } else {
      String _string = ImportSpecifiersUtil.importedModule(specifier).getQualifiedName().toString();
      _xifexpression = (ImportProvidedElement.NAMESPACE_PREFIX + _string);
    }
    return _xifexpression;
  }
  
  /**
   * Computes exported name of the element imported by this specifier.
   */
  public static String importedElementName(final NamedImportSpecifier specifier) {
    final TExportableElement element = specifier.getImportedElement();
    if ((element == null)) {
      String _importedElementAsText = specifier.getImportedElementAsText();
      String _plus = ("<" + _importedElementAsText);
      return (_plus + ">(null)");
    }
    boolean _eIsProxy = element.eIsProxy();
    if (_eIsProxy) {
      String _importedElementAsText_1 = specifier.getImportedElementAsText();
      String _plus_1 = ("<" + _importedElementAsText_1);
      return (_plus_1 + ">(proxy)");
    }
    return element.getExportedName();
  }
  
  /**
   * returns locally used name of element imported via {@link NamedImportSpecifier}
   */
  public static String usedName(final NamedImportSpecifier it) {
    String _xifexpression = null;
    String _alias = it.getAlias();
    boolean _tripleEquals = (_alias == null);
    if (_tripleEquals) {
      _xifexpression = ImportSpecifiersUtil.importedElementName(it);
    } else {
      _xifexpression = it.getAlias();
    }
    return _xifexpression;
  }
  
  /**
   * returns locally used name of element imported via {@link NamespaceImportSpecifier}
   */
  public static String importedElementName(final NamespaceImportSpecifier is, final TExportableElement element) {
    String _alias = is.getAlias();
    String _plus = (_alias + ".");
    String _exportedName = element.getExportedName();
    return (_plus + _exportedName);
  }
  
  public static TModule importedModule(final ImportSpecifier it) {
    EObject _eContainer = it.eContainer();
    return ((ImportDeclaration) _eContainer).getModule();
  }
  
  /**
   * Returns true if the module that is target of the import declaration containing provided import specifier is invalid (null, proxy, no name).
   * Additionally for {@link NamedImportSpecifier} instances checks if linker failed to resolve target (is null, proxy, or has no name)
   * 
   * @param spec - the ImportSpecifier to investigate
   * @return true import looks broken
   */
  public static boolean isBrokenImport(final ImportSpecifier spec) {
    final TModule module = ImportSpecifiersUtil.importedModule(spec);
    if ((((module == null) || module.eIsProxy()) || StringExtensions.isNullOrEmpty(module.getQualifiedName()))) {
      return true;
    }
    if ((spec instanceof NamedImportSpecifier)) {
      final NamedImportSpecifier nis = ((NamedImportSpecifier)spec);
      if ((((nis == null) || nis.eIsProxy()) || StringExtensions.isNullOrEmpty(nis.getImportedElementAsText()))) {
        return true;
      }
      final TExportableElement imported = nis.getImportedElement();
      if ((imported == null)) {
        return true;
      }
      boolean _eIsProxy = imported.eIsProxy();
      boolean _not = (!_eIsProxy);
      if (_not) {
        return StringExtensions.isNullOrEmpty(imported.getExportedName());
      }
    }
    return false;
  }
}
