/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.organize.imports;

import com.google.common.collect.Iterators;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.AnnotationArgument;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.TypeRefAnnotationArgument;
import org.eclipse.n4js.organize.imports.DIUtility;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Utility for computing  dependencies for N4JS DI mechanisms.
 */
@SuppressWarnings("all")
public class InjectedTypesResolverUtility {
  /**
   * Search through {@link Script} to find all {@link N4ClassifierDefinition}s.
   * From found entries builds collection of {@link Type}s used with DI annotations.
   */
  public static Collection<Type> findAllInjected(final Script script) {
    final ArrayList<Type> injections = CollectionLiterals.<Type>newArrayList();
    final Procedure1<N4ClassifierDefinition> _function = (N4ClassifierDefinition cl) -> {
      final Consumer<TFormalParameter> _function_1 = (TFormalParameter it) -> {
        injections.add(InjectedTypesResolverUtility.getDeclaredTypeFromTypeRef(it.getTypeRef()));
      };
      InjectedTypesResolverUtility.getOwnedInjectedCtorParams(cl).forEach(_function_1);
      final Consumer<N4FieldDeclaration> _function_2 = (N4FieldDeclaration f) -> {
        boolean _hasAnnotation = AnnotationDefinition.INJECT.hasAnnotation(f);
        if (_hasAnnotation) {
          injections.add(InjectedTypesResolverUtility.getDeclaredTypeFromTypeRef(f.getDeclaredTypeRef()));
        }
      };
      cl.getOwnedFields().forEach(_function_2);
      boolean _hasOwnedAnnotation = AnnotationDefinition.BINDER.hasOwnedAnnotation(cl);
      if (_hasOwnedAnnotation) {
        final Consumer<Annotation> _function_3 = (Annotation an) -> {
          EObject _value = IterableExtensions.<AnnotationArgument>head(an.getArgs()).value();
          injections.add(InjectedTypesResolverUtility.getDeclaredTypeFromTypeRef(((TypeRef) _value)));
          EObject _value_1 = IterableExtensions.<AnnotationArgument>last(an.getArgs()).value();
          injections.add(InjectedTypesResolverUtility.getDeclaredTypeFromTypeRef(((TypeRef) _value_1)));
        };
        AnnotationDefinition.BIND.getAllOwnedAnnotations(cl).forEach(_function_3);
        final Consumer<N4MethodDeclaration> _function_4 = (N4MethodDeclaration m) -> {
          boolean _hasAnnotation = AnnotationDefinition.PROVIDES.hasAnnotation(m);
          if (_hasAnnotation) {
            injections.add(InjectedTypesResolverUtility.getDeclaredTypeFromTypeRef(m.getReturnTypeRef()));
            final Consumer<FormalParameter> _function_5 = (FormalParameter it) -> {
              injections.add(InjectedTypesResolverUtility.getDeclaredTypeFromTypeRef(it.getDeclaredTypeRef()));
            };
            m.getFpars().forEach(_function_5);
          }
        };
        cl.getOwnedMethods().forEach(_function_4);
      }
      boolean _hasOwnedAnnotation_1 = AnnotationDefinition.GENERATE_INJECTOR.hasOwnedAnnotation(cl);
      if (_hasOwnedAnnotation_1) {
        boolean _hasOwnedAnnotation_2 = AnnotationDefinition.WITH_PARENT_INJECTOR.hasOwnedAnnotation(cl);
        if (_hasOwnedAnnotation_2) {
          Annotation ann = AnnotationDefinition.WITH_PARENT_INJECTOR.getOwnedAnnotation(cl);
          if ((ann != null)) {
            AnnotationArgument _head = IterableExtensions.<AnnotationArgument>head(ann.getArgs());
            injections.add(((TypeRefAnnotationArgument) _head).getTypeRef().getDeclaredType());
          }
        }
        boolean _hasOwnedAnnotation_3 = AnnotationDefinition.USE_BINDER.hasOwnedAnnotation(cl);
        if (_hasOwnedAnnotation_3) {
          Iterable<Annotation> anns = AnnotationDefinition.USE_BINDER.getAllOwnedAnnotations(cl);
          if ((anns != null)) {
            final Function1<Annotation, TypeRefAnnotationArgument> _function_5 = (Annotation it) -> {
              AnnotationArgument _head_1 = IterableExtensions.<AnnotationArgument>head(it.getArgs());
              return ((TypeRefAnnotationArgument) _head_1);
            };
            final Function1<TypeRefAnnotationArgument, Type> _function_6 = (TypeRefAnnotationArgument it) -> {
              return it.getTypeRef().getDeclaredType();
            };
            Iterable<Type> argsTypes = IterableExtensions.<TypeRefAnnotationArgument, Type>map(IterableExtensions.<TypeRefAnnotationArgument>filterNull(IterableExtensions.<Annotation, TypeRefAnnotationArgument>map(anns, _function_5)), _function_6);
            final Consumer<Type> _function_7 = (Type it) -> {
              injections.add(it);
            };
            argsTypes.forEach(_function_7);
          }
        }
      }
    };
    IteratorExtensions.<N4ClassifierDefinition>forEach(Iterators.<N4ClassifierDefinition>filter(script.eAllContents(), N4ClassifierDefinition.class), _function);
    return IterableExtensions.<Type>toList(IterableExtensions.<Type>filterNull(injections));
  }
  
  private static List<TFormalParameter> getOwnedInjectedCtorParams(final N4ClassifierDefinition it) {
    Type _definedType = null;
    if (it!=null) {
      _definedType=it.getDefinedType();
    }
    if ((_definedType instanceof TClass)) {
      Type _definedType_1 = it.getDefinedType();
      final TMethod ctor = ((TClass) _definedType_1).getOwnedCtor();
      if ((null != ctor)) {
        boolean _hasAnnotation = AnnotationDefinition.INJECT.hasAnnotation(ctor);
        if (_hasAnnotation) {
          return ctor.getFpars();
        }
      }
    }
    return CollectionLiterals.<TFormalParameter>emptyList();
  }
  
  /**
   * resolves declared type from type ref. If declared type is
   * N4Provider (can be nested) resolves to (nested) provided type.
   */
  private static Type getDeclaredTypeFromTypeRef(final TypeRef typeRef) {
    Type _xifexpression = null;
    boolean _isProviderType = DIUtility.isProviderType(typeRef);
    if (_isProviderType) {
      final Type providedType = DIUtility.getProvidedType(typeRef);
      if ((null != providedType)) {
        return providedType;
      }
    } else {
      _xifexpression = typeRef.getDeclaredType();
    }
    return _xifexpression;
  }
}
