/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.scoping.accessModifiers.MemberVisibilityChecker;
import org.eclipse.n4js.scoping.members.TypingStrategyFilter;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.StructuralTypeRef;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.Result;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.IssueUserDataKeys;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.n4js.validation.validators.PolyfillValidatorFragment;
import org.eclipse.n4js.validation.validators.StaticPolyfillValidatorExtension;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * superfluous properties in {@code @Spec} constructor.
 */
@SuppressWarnings("all")
public class N4JSClassValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private PolyfillValidatorFragment polyfillValidatorFragment;
  
  @Inject
  @Extension
  private ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private MemberVisibilityChecker memberVisibilityChecker;
  
  @Inject
  private N4JSElementKeywordProvider n4jsElementKeywordProvider;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * Non abstract test classes have to exported: IDEBUG-572.
   */
  @Check
  public void checkTestClassExported(final N4ClassDeclaration classDecl) {
    Type _definedType = null;
    if (classDecl!=null) {
      _definedType=classDecl.getDefinedType();
    }
    if ((_definedType instanceof TClass)) {
      Type _definedType_1 = classDecl.getDefinedType();
      TClass clazz = ((TClass) _definedType_1);
      if ((((!clazz.isAbstract()) && (!clazz.isExported())) && this.hasTestMethods(clazz))) {
        this.addIssue(IssueCodes.getMessageForCLF_TEST_CLASS_NOT_EXPORTED(), classDecl, 
          N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, IssueCodes.CLF_TEST_CLASS_NOT_EXPORTED);
      }
    }
  }
  
  private boolean hasTestMethods(final TClass clazz) {
    Resource _eResource = null;
    if (clazz!=null) {
      _eResource=clazz.eResource();
    }
    boolean _tripleNotEquals = (null != _eResource);
    if (_tripleNotEquals) {
      final TModule module = N4JSResource.getModule(clazz.eResource());
      if ((null != module)) {
        final Function1<TMethod, Boolean> _function = (TMethod it) -> {
          return Boolean.valueOf(AnnotationDefinition.TEST_METHOD.hasAnnotation(it));
        };
        return IterableExtensions.<TMethod>exists(Iterables.<TMethod>filter(this.containerTypesHelper.fromContext(module).allMembers(clazz, false, false), TMethod.class), _function);
      }
    }
    return false;
  }
  
  /**
   * Checks new expression.
   */
  @Check
  public void checkSpecConstructorArguments(final NewExpression newExpression) {
    Expression _callee = newExpression.getCallee();
    boolean _not = (!(_callee instanceof IdentifierRef));
    if (_not) {
      return;
    }
    Expression _callee_1 = newExpression.getCallee();
    final IdentifierRef identifierRef = ((IdentifierRef) _callee_1);
    IdentifiableElement _id = identifierRef.getId();
    boolean _not_1 = (!(_id instanceof TClass));
    if (_not_1) {
      return;
    }
    IdentifiableElement _id_1 = identifierRef.getId();
    final TClass tclass = ((TClass) _id_1);
    final TMethod ctor = this.containerTypesHelper.fromContext(newExpression).findConstructor(tclass);
    if (((null == ctor) || IterableExtensions.isNullOrEmpty(ctor.getFpars()))) {
      return;
    }
    final Function1<TFormalParameter, Boolean> _function = (TFormalParameter it) -> {
      TAnnotation _annotation = AnnotationDefinition.SPEC.getAnnotation(it);
      return Boolean.valueOf((null != _annotation));
    };
    final List<TFormalParameter> fparsWithSpecAnnotation = IterableExtensions.<TFormalParameter>toList(IterableExtensions.<TFormalParameter>filter(ctor.getFpars(), _function));
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(fparsWithSpecAnnotation);
    if (_isNullOrEmpty) {
      return;
    }
    int _size = fparsWithSpecAnnotation.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      return;
    }
    final TFormalParameter fparWithSpecAnnotation = Iterables.<TFormalParameter>getOnlyElement(fparsWithSpecAnnotation);
    final TypeRef typeRef = fparWithSpecAnnotation.getTypeRef();
    if (((!(typeRef instanceof ThisTypeRefStructural)) || (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER != typeRef.getTypingStrategy()))) {
      return;
    }
    final int indexOfTheSpecFpar = ctor.getFpars().indexOf(fparWithSpecAnnotation);
    if (((0 > indexOfTheSpecFpar) || (newExpression.getArguments().size() <= indexOfTheSpecFpar))) {
      return;
    }
    Argument _get = newExpression.getArguments().get(indexOfTheSpecFpar);
    Expression _expression = null;
    if (_get!=null) {
      _expression=_get.getExpression();
    }
    boolean _not_2 = (!(_expression instanceof ObjectLiteral));
    if (_not_2) {
      return;
    }
    Argument _get_1 = newExpression.getArguments().get(indexOfTheSpecFpar);
    Expression _expression_1 = null;
    if (_get_1!=null) {
      _expression_1=_get_1.getExpression();
    }
    final ObjectLiteral objectLiteral = ((ObjectLiteral) _expression_1);
    Type _definedType = objectLiteral.getDefinedType();
    boolean _not_3 = (!(_definedType instanceof ContainerType<?>));
    if (_not_3) {
      return;
    }
    final TypingStrategyFilter strategy = new TypingStrategyFilter(TypingStrategy.STRUCTURAL);
    final Function1<TMember, Boolean> _function_1 = (TMember member) -> {
      return Boolean.valueOf(((member instanceof TField) || (member instanceof TSetter)));
    };
    final Function1<TMember, Boolean> _function_2 = (TMember it) -> {
      return Boolean.valueOf(strategy.apply(it));
    };
    final Function1<TMember, String> _function_3 = (TMember it) -> {
      return it.getName();
    };
    final Map<String, TMember> publicWritableFieldsAndSetters = IterableExtensions.<String, TMember>toMap(IterableExtensions.<TMember>filter(IterableExtensions.<TMember>filter(this.containerTypesHelper.fromContext(newExpression).allMembers(tclass), _function_1), _function_2), _function_3);
    final Consumer<TField> _function_4 = (TField it) -> {
      publicWritableFieldsAndSetters.put(it.getName(), it);
    };
    Iterables.<TField>filter(((ThisTypeRefStructural) typeRef).getStructuralMembers(), TField.class).forEach(_function_4);
    this.checkFieldInitializationOfImplementedInterface(publicWritableFieldsAndSetters, objectLiteral);
  }
  
  /**
   * Check if an object literal in {@code @Spec} constructor provide a property that comes from a built-in/provided by runtime interface.
   * IDE-2747.
   */
  private void checkFieldInitializationOfImplementedInterface(final Map<String, TMember> publicWritableFieldsMap, final ObjectLiteral objectLiteral) {
    Type _definedType = objectLiteral.getDefinedType();
    final EList<? extends TMember> properties = ((ContainerType<?>) _definedType).getOwnedMembers();
    final Consumer<TMember> _function = (TMember property) -> {
      final TMember field = publicWritableFieldsMap.get(property.getName());
      if ((field != null)) {
        final ContainerType<?> containingClassifier = field.getContainingType();
        if ((containingClassifier instanceof TInterface)) {
          boolean _builtInOrProvidedByRuntimeOrExternalWithoutN4JSAnnotation = N4JSLanguageUtils.builtInOrProvidedByRuntimeOrExternalWithoutN4JSAnnotation(((TInterface)containingClassifier));
          if (_builtInOrProvidedByRuntimeOrExternalWithoutN4JSAnnotation) {
            final String message = IssueCodes.getMessageForCLF_SPEC_BUILT_IN_OR_PROVIDED_BY_RUNTIME_OR_EXTENAL_WITHOUT_N4JS_ANNOTATION(field.getName(), ((TInterface)containingClassifier).getName());
            this.addIssue(message, property.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_SPEC_BUILT_IN_OR_PROVIDED_BY_RUNTIME_OR_EXTENAL_WITHOUT_N4JS_ANNOTATION);
          }
        }
      }
    };
    properties.forEach(_function);
  }
  
  @Check
  public void checkN4ClassDeclaration(final N4ClassDeclaration n4Class) {
    Type _definedType = n4Class.getDefinedType();
    boolean _not = (!(_definedType instanceof TClass));
    if (_not) {
      return;
    }
    boolean _holdsPolyfill = this.polyfillValidatorFragment.holdsPolyfill(this, n4Class);
    if (_holdsPolyfill) {
      Type _definedType_1 = n4Class.getDefinedType();
      final TClass tClass = ((TClass) _definedType_1);
      this.internalCheckAbstractFinal(tClass);
      boolean _holdsSuperClass = this.holdsSuperClass(n4Class);
      if (_holdsSuperClass) {
        this.holdsNoCyclicInheritance(n4Class);
      }
      StaticPolyfillValidatorExtension.internalCheckPolyFilledClassWithAdditionalInterface(n4Class, this);
      this.internalCheckImplementedInterfaces(n4Class);
      this.internalCheckSpecAnnotation(n4Class);
    }
  }
  
  /**
   * Overridden to be accessible from PolyfillValidatorFragment
   */
  @Override
  public void addIssue(final String message, final EObject source, final EStructuralFeature feature, final String issueCode, final String... issueData) {
    super.addIssue(message, source, feature, issueCode, issueData);
  }
  
  private void internalCheckAbstractFinal(final TClass tClass) {
    if ((tClass.isAbstract() && tClass.isFinal())) {
      final String message = IssueCodes.getMessageForCLF_ABSTRACT_FINAL("class");
      this.addIssue(message, tClass.getAstElement(), N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, IssueCodes.CLF_ABSTRACT_FINAL);
    }
  }
  
  private boolean holdsSuperClass(final N4ClassDeclaration n4Class) {
    ParameterizedTypeRef _superClassRef = n4Class.getSuperClassRef();
    Type _declaredType = null;
    if (_superClassRef!=null) {
      _declaredType=_superClassRef.getDeclaredType();
    }
    final Type superType = _declaredType;
    if (((superType != null) && (superType.getName() != null))) {
      if ((superType instanceof PrimitiveType)) {
        final String message = IssueCodes.getMessageForCLF_EXTENDS_PRIMITIVE_GENERIC_TYPE(((PrimitiveType)superType).getName());
        this.addIssue(message, n4Class.getSuperClassRef(), null, IssueCodes.CLF_EXTENDS_PRIMITIVE_GENERIC_TYPE);
      } else {
        if (((!(superType instanceof TClass)) && (!(superType instanceof TObjectPrototype)))) {
          if ((superType instanceof TInterface)) {
            final String message_1 = IssueCodes.getMessageForSYN_KW_EXTENDS_IMPLEMENTS_MIXED_UP(this.validatorMessageHelper.description(n4Class), "extend", 
              this.validatorMessageHelper.description(superType), "implements");
            this.addIssue(message_1, n4Class.getSuperClassRef(), null, IssueCodes.SYN_KW_EXTENDS_IMPLEMENTS_MIXED_UP);
          } else {
            final String message_2 = IssueCodes.getMessageForCLF_WRONG_META_TYPE(this.validatorMessageHelper.description(n4Class), "extend", this.validatorMessageHelper.description(superType));
            this.addIssue(message_2, n4Class.getSuperClassRef(), null, IssueCodes.CLF_WRONG_META_TYPE);
            return false;
          }
        } else {
          if ((superType instanceof TClass)) {
            boolean _isFinal = ((TClass)superType).isFinal();
            if (_isFinal) {
              final String message_3 = IssueCodes.getMessageForCLF_EXTEND_FINAL(((TClass)superType).getName());
              Object _eGet = ((TClass)superType).eGet(TypesPackage.eINSTANCE.getSyntaxRelatedTElement_AstElement(), 
                false);
              final EObject superTypeAstElement = ((EObject) _eGet);
              String _xifexpression = null;
              if ((superTypeAstElement != null)) {
                _xifexpression = EcoreUtil.getURI(superTypeAstElement).toString();
              }
              final String superClassUri = _xifexpression;
              if ((superClassUri != null)) {
                this.addIssue(message_3, n4Class.getSuperClassRef(), null, IssueCodes.CLF_EXTEND_FINAL, 
                  IssueUserDataKeys.CLF_EXTEND_FINAL.SUPER_TYPE_DECLARATION_URI, superClassUri);
              } else {
                this.addIssue(message_3, n4Class.getSuperClassRef(), null, IssueCodes.CLF_EXTEND_FINAL, 
                  IssueUserDataKeys.CLF_EXTEND_FINAL.SUPER_TYPE_DECLARATION_URI);
              }
              return false;
            }
            boolean _holdsCtorOfSuperTypeIsAccessible = this.holdsCtorOfSuperTypeIsAccessible(n4Class, ((TClassifier)superType));
            boolean _not = (!_holdsCtorOfSuperTypeIsAccessible);
            if (_not) {
              return false;
            }
            if ((((TClass)superType).isObservable() && (!((TClass) n4Class.getDefinedType()).isObservable()))) {
              final String message_4 = IssueCodes.getMessageForCLF_OBSERVABLE_MISSING(n4Class.getName(), ((TClass)superType).getName());
              this.addIssue(message_4, n4Class, N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, IssueCodes.CLF_OBSERVABLE_MISSING);
              return false;
            }
          } else {
            if ((superType instanceof TObjectPrototype)) {
              boolean _holdsCtorOfSuperTypeIsAccessible_1 = this.holdsCtorOfSuperTypeIsAccessible(n4Class, ((TClassifier)superType));
              boolean _not_1 = (!_holdsCtorOfSuperTypeIsAccessible_1);
              if (_not_1) {
                return false;
              }
            }
          }
        }
      }
    }
    return true;
  }
  
  private boolean holdsCtorOfSuperTypeIsAccessible(final N4ClassDeclaration n4Class, final TClassifier superType) {
    final ParameterizedTypeRef receiverTypeRef = TypeUtils.createTypeRef(n4Class.getDefinedType());
    final TMethod superCtor = this.containerTypesHelper.fromContext(n4Class).findConstructor(superType);
    if (((superCtor != null) && (!this.memberVisibilityChecker.isVisible(n4Class, receiverTypeRef, superCtor).visibility))) {
      final String message = IssueCodes.getMessageForCLF_EXTEND_NON_ACCESSIBLE_CTOR(
        this.n4jsElementKeywordProvider.keyword(superType), 
        superType.getName());
      this.addIssue(message, n4Class, N4JSPackage.Literals.N4_CLASS_DEFINITION__SUPER_CLASS_REF, IssueCodes.CLF_EXTEND_NON_ACCESSIBLE_CTOR);
      return false;
    }
    return true;
  }
  
  private void internalCheckImplementedInterfaces(final N4ClassDeclaration n4Class) {
    final Consumer<ParameterizedTypeRef> _function = (ParameterizedTypeRef it) -> {
      final Type consumedType = it.getDeclaredType();
      if (((consumedType != null) && (consumedType.getName() != null))) {
        if ((!(consumedType instanceof TInterface))) {
          if ((((consumedType instanceof TClass) || (consumedType instanceof TObjectPrototype)) && 
            (n4Class.getSuperClassRef() == null))) {
            final String message = IssueCodes.getMessageForSYN_KW_EXTENDS_IMPLEMENTS_MIXED_UP(this.validatorMessageHelper.description(n4Class), "implement", 
              this.validatorMessageHelper.description(consumedType), "extends");
            this.addIssue(message, it, null, IssueCodes.SYN_KW_EXTENDS_IMPLEMENTS_MIXED_UP);
          } else {
            final String message_1 = IssueCodes.getMessageForCLF_WRONG_META_TYPE(this.validatorMessageHelper.description(n4Class), "implement", 
              this.validatorMessageHelper.description(consumedType));
            this.addIssue(message_1, it, null, IssueCodes.CLF_WRONG_META_TYPE);
          }
        }
      }
    };
    n4Class.getImplementedInterfaceRefs().forEach(_function);
  }
  
  private void internalCheckSpecAnnotation(final N4ClassDeclaration n4ClassDeclaration) {
    final N4MethodDeclaration ctor = n4ClassDeclaration.getOwnedCtor();
    if ((ctor != null)) {
      ArrayList<Annotation> specAnnotations = CollectionLiterals.<Annotation>newArrayList();
      int i = 0;
      EList<FormalParameter> _fpars = ctor.getFpars();
      for (final FormalParameter currFPar : _fpars) {
        {
          final Annotation annSpec = AnnotationDefinition.SPEC.getAnnotation(currFPar);
          if ((annSpec != null)) {
            specAnnotations.add(annSpec);
            final boolean correctType = ((currFPar.getDeclaredTypeRef() instanceof ThisTypeRef) && (TypingStrategy.STRUCTURAL_FIELD_INITIALIZER == currFPar.getDeclaredTypeRef().getTypingStrategy()));
            if ((!correctType)) {
              final String message = IssueCodes.getMessageForCLF_SPEC_WRONG_TYPE();
              this.addIssue(message, annSpec, null, IssueCodes.CLF_SPEC_WRONG_TYPE);
            } else {
              this.holdsAdditionalSpecFieldMatchesOwnedFields(n4ClassDeclaration, ctor, i);
            }
          }
          i = (i + 1);
        }
      }
      int _size = specAnnotations.size();
      boolean _greaterEqualsThan = (_size >= 2);
      if (_greaterEqualsThan) {
        for (final Annotation currAnnSpec : specAnnotations) {
          {
            final String message = IssueCodes.getMessageForCLF_SPEC_MULTIPLE();
            this.addIssue(message, currAnnSpec, null, IssueCodes.CLF_SPEC_MULTIPLE);
          }
        }
      }
    }
  }
  
  public void holdsAdditionalSpecFieldMatchesOwnedFields(final N4ClassDeclaration n4ClassDeclaration, final N4MethodDeclaration ctor, final int parIndex) {
    Type _definedType = n4ClassDeclaration.getDefinedType();
    final TClass tclass = ((TClass) _definedType);
    Type _definedType_1 = ctor.getDefinedType();
    final EList<TFormalParameter> fpars = ((TFunction) _definedType_1).getFpars();
    int _size = fpars.size();
    boolean _greaterEqualsThan = (parIndex >= _size);
    if (_greaterEqualsThan) {
      return;
    }
    final TFormalParameter fpar = fpars.get(parIndex);
    final TypeRef fparType = fpar.getTypeRef();
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(n4ClassDeclaration);
    int memberIndex = 0;
    EList<TStructMember> _structuralMembers = fparType.getStructuralMembers();
    for (final TStructMember smember : _structuralMembers) {
      {
        final Function1<TMember, Boolean> _function = (TMember it) -> {
          String _name = it.getName();
          String _name_1 = smember.getName();
          return Boolean.valueOf(Objects.equal(_name, _name_1));
        };
        final TMember tfield = IterableExtensions.<TMember>findFirst(tclass.getOwnedMembers(), _function);
        if (((tfield != null) && (tfield.isField() || tfield.isSetter()))) {
          final TypeRef fieldType = this.ts.tau(tfield, TypeUtils.createTypeRef(tclass));
          final TypeRef smemberType = this.ts.tau(smember, TypeUtils.createTypeRef(tclass));
          final Result subtypeRes = this.ts.subtype(G, smemberType, fieldType);
          boolean _isFailure = subtypeRes.isFailure();
          if (_isFailure) {
            final String message = IssueCodes.getMessageForCLF_SPEC_WRONG_ADD_MEMBERTYPE(smember.getName(), this.validatorMessageHelper.description(tfield), 
              this.validatorMessageHelper.trimTypesystemMessage(subtypeRes));
            TypeRef _declaredTypeRef = ctor.getFpars().get(parIndex).getDeclaredTypeRef();
            final TStructMember errMember = ((StructuralTypeRef) _declaredTypeRef).getStructuralMembers().get(memberIndex);
            EObject _xifexpression = null;
            EObject _astElement = errMember.getAstElement();
            boolean _tripleNotEquals = (_astElement != null);
            if (_tripleNotEquals) {
              _xifexpression = errMember.getAstElement();
            } else {
              _xifexpression = errMember;
            }
            final EObject sourceObject = _xifexpression;
            this.addIssue(message, sourceObject, 
              IssueCodes.CLF_SPEC_WRONG_ADD_MEMBERTYPE);
          }
        }
        memberIndex = (memberIndex + 1);
      }
    }
  }
  
  private boolean holdsNoCyclicInheritance(final N4ClassDeclaration n4ClassDeclaration) {
    Type _definedType = n4ClassDeclaration.getDefinedType();
    final TClassifier cls = ((TClassifier) _definedType);
    final List<TClassifier> cycle = this.findCyclicInheritance(cls);
    if ((cycle != null)) {
      final Function1<TClassifier, String> _function = (TClassifier it) -> {
        return it.getName();
      };
      final String message = IssueCodes.getMessageForCLF_INHERITANCE_CYCLE(IterableExtensions.join(ListExtensions.<TClassifier, String>map(cycle, _function), ", "));
      this.addIssue(message, n4ClassDeclaration, N4JSPackage.Literals.N4_CLASS_DEFINITION__SUPER_CLASS_REF, 
        IssueCodes.CLF_INHERITANCE_CYCLE);
      return false;
    }
    return true;
  }
}
