/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.List;
import java.util.function.Predicate;
import org.eclipse.n4js.compare.ProjectCompareHelper;
import org.eclipse.n4js.compare.ProjectCompareResult;
import org.eclipse.n4js.compare.ProjectComparison;
import org.eclipse.n4js.compare.ProjectComparisonEntry;
import org.eclipse.n4js.naming.N4JSQualifiedNameConverter;
import org.eclipse.n4js.projectDescription.ProjectDescriptionFactory;
import org.eclipse.n4js.projectDescription.ProjectReference;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.junit.Assert;

/**
 * Helper methods for testing API / implementation compare functionality.
 */
@SuppressWarnings("all")
public class ApiImplCompareTestHelper {
  @Inject
  private ProjectCompareHelper projectCompareHelper;
  
  /**
   * Find entry for type with given name and check number, order and status/description of
   * its child entries (corresponding to the members of the type).
   * 
   * @param expectedChildrenNameStatusDescription
   *             the expected name, compare status, and compare description of each child in the following format:
   *             <pre>
   *             expected name of child  ->  expected comparison status  ->  expected comparison description
   *             </pre>
   */
  public void assertCorrectChildEntries(final ProjectComparison comparison, final String fqnOfModule, final String nameOfType, final Pair<Pair<String, ProjectCompareResult.Status>, String>... expectedChildrenNameStatusDescription) {
    final String fqnOfType = ((fqnOfModule + N4JSQualifiedNameConverter.DELIMITER) + nameOfType);
    final ProjectComparisonEntry entryForType = this.findEntryForType(comparison, fqnOfType);
    Assert.assertNotNull(("cannot find entry for type " + fqnOfType), entryForType);
    final ProjectComparisonEntry[] childEntriesOfType = entryForType.getChildren();
    final int expectedChildCount = ((List<Pair<Pair<String, ProjectCompareResult.Status>, String>>)Conversions.doWrapArray(expectedChildrenNameStatusDescription)).size();
    final Function1<Pair<Pair<String, ProjectCompareResult.Status>, String>, String> _function = (Pair<Pair<String, ProjectCompareResult.Status>, String> it) -> {
      return it.getKey().getKey();
    };
    final List<String> expectedChildNames = IterableExtensions.<String>toList(ListExtensions.<Pair<Pair<String, ProjectCompareResult.Status>, String>, String>map(((List<Pair<Pair<String, ProjectCompareResult.Status>, String>>)Conversions.doWrapArray(expectedChildrenNameStatusDescription)), _function));
    Assert.assertEquals(nameOfType, this.getElementNameForEntry(entryForType));
    Assert.assertEquals(
      ((("expected exactly " + Integer.valueOf(expectedChildCount)) + " child entries for ") + nameOfType), expectedChildCount, 
      ((List<ProjectComparisonEntry>)Conversions.doWrapArray(childEntriesOfType)).size());
    final Function1<ProjectComparisonEntry, String> _function_1 = (ProjectComparisonEntry it) -> {
      return this.getElementNameForEntry(it);
    };
    Assert.assertEquals(
      (("child entries of " + nameOfType) + " have wrong name(s) or incorrect order"), expectedChildNames, 
      IterableExtensions.<String>toList(ListExtensions.<ProjectComparisonEntry, String>map(((List<ProjectComparisonEntry>)Conversions.doWrapArray(childEntriesOfType)), _function_1)));
    for (int idx = 0; (idx < ((List<ProjectComparisonEntry>)Conversions.doWrapArray(childEntriesOfType)).size()); idx++) {
      {
        final ProjectComparisonEntry currChildEntry = childEntriesOfType[idx];
        final ProjectCompareResult.Status expectedStatus = expectedChildrenNameStatusDescription[idx].getKey().getValue();
        final String expectedDescription = expectedChildrenNameStatusDescription[idx].getValue();
        this.assertDiff(currChildEntry, expectedStatus, expectedDescription);
      }
    }
  }
  
  public void assertCorrectTypeEntry(final ProjectComparison comparision, final String fqnOfModule, final Pair<Pair<String, ProjectCompareResult.Status>, String>... expectedTypeStatusDescriptions) {
    for (int idx = 0; (idx < ((List<Pair<Pair<String, ProjectCompareResult.Status>, String>>)Conversions.doWrapArray(expectedTypeStatusDescriptions)).size()); idx++) {
      {
        final String elementName = expectedTypeStatusDescriptions[idx].getKey().getKey();
        final ProjectCompareResult.Status elementStatus = expectedTypeStatusDescriptions[idx].getKey().getValue();
        final String elementDescription = expectedTypeStatusDescriptions[idx].getValue();
        final String fqnOfType = ((fqnOfModule + N4JSQualifiedNameConverter.DELIMITER) + elementName);
        final ProjectComparisonEntry entryForType = this.findEntryForType(comparision, fqnOfType);
        this.assertDiff(entryForType, elementStatus, elementDescription);
      }
    }
  }
  
  private void assertDiff(final ProjectComparisonEntry entry, final ProjectCompareResult.Status status, final String description) {
    Assert.assertNotNull(entry);
    Assert.assertSame("wrong status", status, this.getStatusForFirstImplementation(entry));
    Assert.assertEquals("wrong description", description, this.getDescriptionForFirstImplementation(entry));
  }
  
  /**
   * Search the given comparison for an entry with an API- or implementation-side element
   * with the given fully qualified name.
   */
  public ProjectComparisonEntry findEntryForType(final ProjectComparison comparison, final String fqn) {
    final Predicate<ProjectComparisonEntry> _function = (ProjectComparisonEntry entry) -> {
      String _elementFqnForEntry = this.getElementFqnForEntry(entry);
      return Objects.equal(_elementFqnForEntry, fqn);
    };
    return comparison.getAllEntries().filter(_function).findFirst().orElse(null);
  }
  
  public String getElementNameForEntry(final ProjectComparisonEntry entry) {
    final Function1<IdentifiableElement, String> _function = (IdentifiableElement it) -> {
      return it.getName();
    };
    return IterableExtensions.<String>head(IterableExtensions.<IdentifiableElement, String>map(Iterables.<IdentifiableElement>filter(((Iterable<?>)Conversions.doWrapArray(entry.getAllElements())), IdentifiableElement.class), _function));
  }
  
  public String getElementFqnForEntry(final ProjectComparisonEntry entry) {
    final Function1<Type, String> _function = (Type it) -> {
      TModule _containingModule = it.getContainingModule();
      String _qualifiedName = null;
      if (_containingModule!=null) {
        _qualifiedName=_containingModule.getQualifiedName();
      }
      String _plus = (_qualifiedName + N4JSQualifiedNameConverter.DELIMITER);
      String _name = it.getName();
      return (_plus + _name);
    };
    return IterableExtensions.<String>head(IterableExtensions.<Type, String>map(Iterables.<Type>filter(((Iterable<?>)Conversions.doWrapArray(entry.getAllElements())), Type.class), _function));
  }
  
  private ProjectCompareResult.Status getStatusForFirstImplementation(final ProjectComparisonEntry entry) {
    return this.projectCompareHelper.compareApiImpl(entry, 0).status;
  }
  
  private String getDescriptionForFirstImplementation(final ProjectComparisonEntry entry) {
    return this.projectCompareHelper.compareApiImpl(entry, 0).description;
  }
  
  public ProjectReference createProjectReference(final String projectNameOfTargetProject) {
    final ProjectReference pref = ProjectDescriptionFactory.eINSTANCE.createProjectReference();
    pref.setProjectName(projectNameOfTargetProject);
    return pref;
  }
}
