/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.xpect.validation.suppression;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.xpect.validation.suppression.IssueCode;
import org.eclipse.n4js.xpect.validation.suppression.IssueConfiguration;
import org.eclipse.n4js.xpect.validation.suppression.SuppressIssuesSetupRoot;
import org.eclipse.xpect.XpectImport;
import org.eclipse.xpect.XpectReplace;
import org.eclipse.xpect.setup.ISetupInitializer;
import org.eclipse.xpect.setup.XpectSetupFactory;
import org.eclipse.xpect.state.Creates;
import org.eclipse.xpect.text.IRegion;
import org.eclipse.xpect.xtext.lib.setup.ThisResource;
import org.eclipse.xpect.xtext.lib.tests.ValidationTestModuleSetup;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.validation.CheckMode;
import org.eclipse.xtext.validation.IResourceValidator;
import org.eclipse.xtext.validation.Issue;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * This setup factory filters issues based on their issue code before
 * passing them to xpect methods.
 * 
 * For more detail on which issue codes are suppressed see {@link N4JSLanguageConstants#DEFAULT_SUPPRESSED_ISSUE_CODES_FOR_TESTS}.
 * 
 * To integrate this setup with an xpect runner, import this class via @XpectImport.
 * 
 * For further configuration you can use {@link IssueConfiguration} in the XPECTSETUP of specific files.
 * 
 * <p>
 * When sub-classing this setup class, the two following points must be fulfilled:
 * 
 * <ul>
 * 	<li>The subclass must be decorated with the same set of <code>@Xpect*</code> annotations as this class.</li>
 * 	<li>The subclass must implement a constructor with the same <code>@ThisResource</code> annotation on the
 * 		<code>resource</code> parameters as in this class.</li>
 * </ul>
 * </p>
 */
@XpectSetupFactory
@XpectReplace(ValidationTestModuleSetup.IssuesByLineProvider.class)
@XpectImport({ SuppressIssuesSetupRoot.class })
@SuppressWarnings("all")
public abstract class AbstractSuppressIssuesSetup extends ValidationTestModuleSetup.IssuesByLineProvider {
  private final Collection<String> suppressedIssueCodes = new ArrayList<String>(this.getDefaultSuppressedIssueCodes());
  
  private Multimap<IRegion, Issue> issuesByLine = null;
  
  /**
   * Instantiates a new SuppressingValidatorSetup.
   * 
   * @param resource
   * 		The XtextResource. Provided by Xpect.
   * @param setupInitializer
   * 		The setup initializer for setup specific settings. Provided by Xpect.
   * @param suppressedIssueCodes
   * 		The issue codes which should be suppressed while testing.
   */
  public AbstractSuppressIssuesSetup(@ThisResource final XtextResource resource, final ISetupInitializer<SuppressIssuesSetupRoot> setupInitializer) {
    super(resource);
    this.initialize(setupInitializer);
  }
  
  private void initialize(final ISetupInitializer<SuppressIssuesSetupRoot> setupInitializer) {
    final SuppressIssuesSetupRoot setup = new SuppressIssuesSetupRoot();
    setupInitializer.initialize(setup);
    IssueConfiguration _issueConfiguration = setup.getIssueConfiguration();
    boolean _tripleEquals = (_issueConfiguration == null);
    if (_tripleEquals) {
      return;
    }
    List<IssueCode> _issueCodes = setup.getIssueConfiguration().getIssueCodes();
    for (final IssueCode code : _issueCodes) {
      boolean _isEnabled = code.isEnabled();
      if (_isEnabled) {
        this.suppressedIssueCodes.remove(code.getName());
      } else {
        this.suppressedIssueCodes.add(code.getName());
      }
    }
  }
  
  /**
   * Returns the list of issue codes that are suppressed by default when
   * using this suppress issues setup.
   */
  protected abstract Collection<String> getDefaultSuppressedIssueCodes();
  
  /**
   * Override this method to remove suppressed issues from the
   * list of issues in the resource.
   */
  @Override
  protected boolean includeIssue(final Issue issue) {
    boolean _xifexpression = false;
    boolean _contains = this.suppressedIssueCodes.contains(issue.getCode());
    if (_contains) {
      _xifexpression = false;
    } else {
      _xifexpression = super.includeIssue(issue);
    }
    return _xifexpression;
  }
  
  /**
   * Override this method to additionally filter the issue by line mapping.
   * This is required, since many xpect methods obtain their list of issues by an IssuesByLine parameter.
   */
  @Creates(ValidationTestModuleSetup.IssuesByLine.class)
  @Override
  public Multimap<IRegion, Issue> collectIssuesByLine() {
    if ((this.issuesByLine == null)) {
      IResourceValidator _resourceValidator = this.getResource().getResourceServiceProvider().getResourceValidator();
      final ValidationTestModuleSetup.TestingResourceValidator validator = ((ValidationTestModuleSetup.TestingResourceValidator) _resourceValidator);
      this.issuesByLine = validator.validateDelegateAndMapByOffset(this.getResource(), CheckMode.ALL, CancelIndicator.NullImpl, null);
    }
    boolean _isEmpty = this.suppressedIssueCodes.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final HashMultimap<IRegion, Issue> filteredResults = HashMultimap.<IRegion, Issue>create(this.issuesByLine);
      final Function1<Map.Entry<IRegion, Issue>, Boolean> _function = (Map.Entry<IRegion, Issue> entry) -> {
        boolean _includeIssue = this.includeIssue(entry.getValue());
        return Boolean.valueOf((!_includeIssue));
      };
      final Consumer<Map.Entry<IRegion, Issue>> _function_1 = (Map.Entry<IRegion, Issue> entry) -> {
        filteredResults.remove(entry.getKey(), entry.getValue());
      };
      IterableExtensions.<Map.Entry<IRegion, Issue>>filter(this.issuesByLine.entries(), _function).forEach(_function_1);
      this.issuesByLine = filteredResults;
    }
    return this.issuesByLine;
  }
}
