/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.base.Joiner;
import com.google.common.collect.Iterators;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.organize.imports.ScriptDependencyResolver;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.ReferencingElement_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.transpiler.utils.TranspilerUtils;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

/**
 * Transformation to clean up imports:
 * <ul>
 * <li>add missing imports,
 * <li>remove unused import statements.
 * </ul>
 */
@SuppressWarnings("all")
public class SanitizeImportsTransformation extends Transformation {
  @Override
  public void analyze() {
  }
  
  @Override
  public void assertPreConditions() {
    this.assertTrue("requires a fully resolved script", this.getState().im.isFlaggedUsageMarkingFinished());
  }
  
  @Override
  public void assertPostConditions() {
    final Function1<ImportSpecifier, Boolean> _function = (ImportSpecifier it) -> {
      boolean _isUsed = this.isUsed(it);
      return Boolean.valueOf((!_isUsed));
    };
    final List<ImportSpecifier> unusedImports = IterableExtensions.<ImportSpecifier>toList(IterableExtensions.<ImportSpecifier>filter(EcoreUtil2.<ImportSpecifier>getAllContentsOfType(this.getState().im, ImportSpecifier.class), _function));
    String _join = Joiner.on(",").join(unusedImports);
    String _plus = (("There should not be any unused import." + " Unused=") + _join);
    int _size = unusedImports.size();
    boolean _tripleEquals = (_size == 0);
    this.assertTrue(_plus, _tripleEquals);
  }
  
  @Override
  public void transform() {
    this.addMissingImplicitImports();
    this.removeUnusedImports();
  }
  
  /**
   * Compute imports for globally defined elements in other resources on the projects dependencies and
   * add them iff missing.
   */
  private void addMissingImplicitImports() {
    final Function1<IdentifierRef_IM, SymbolTableEntry> _function = (IdentifierRef_IM it) -> {
      return it.getId_IM();
    };
    final Set<SymbolTableEntryOriginal> referencedSTEs = IteratorExtensions.<SymbolTableEntryOriginal>toSet(Iterators.<SymbolTableEntryOriginal>filter(IteratorExtensions.<IdentifierRef_IM, SymbolTableEntry>map(Iterators.<IdentifierRef_IM>filter(this.getState().im.eAllContents(), IdentifierRef_IM.class), _function), SymbolTableEntryOriginal.class));
    final TModule baseModule = this.getState().resource.getModule();
    final Function1<SymbolTableEntryOriginal, Boolean> _function_1 = (SymbolTableEntryOriginal it) -> {
      return Boolean.valueOf(ScriptDependencyResolver.shouldBeImported(baseModule, it.getOriginalTarget()));
    };
    final Iterable<SymbolTableEntryOriginal> requiresImport = IterableExtensions.<SymbolTableEntryOriginal>filter(referencedSTEs, _function_1);
    final Iterable<SymbolTableEntryOriginal> thingsToImportSTE = requiresImport;
    final Consumer<SymbolTableEntryOriginal> _function_2 = (SymbolTableEntryOriginal ste) -> {
      ImportSpecifier _importSpecifier = ste.getImportSpecifier();
      boolean _tripleEquals = (_importSpecifier == null);
      if (_tripleEquals) {
        final IdentifiableElement orig = ste.getOriginalTarget();
        boolean _isExported = N4JSLanguageUtils.isExported(orig);
        if (_isExported) {
          final TModule module = orig.getContainingModule();
          boolean _hasAnnotation = AnnotationDefinition.GLOBAL.hasAnnotation(module);
          if (_hasAnnotation) {
            this.addNamedImport(ste, null);
          }
        }
      }
    };
    thingsToImportSTE.forEach(_function_2);
  }
  
  private void removeUnusedImports() {
    final Function1<ImportSpecifier, Boolean> _function = (ImportSpecifier it) -> {
      boolean _isUsed = this.isUsed(it);
      return Boolean.valueOf((!_isUsed));
    };
    final List<ImportSpecifier> unusedImports = IterableExtensions.<ImportSpecifier>toList(IterableExtensions.<ImportSpecifier>filter(EcoreUtil2.<ImportSpecifier>getAllContentsOfType(this.getState().im, ImportSpecifier.class), _function));
    for (final Iterator<ImportSpecifier> iter = unusedImports.iterator(); iter.hasNext();) {
      {
        final ImportSpecifier is = iter.next();
        EObject _eContainer = is.eContainer();
        final ImportDeclaration container = ((ImportDeclaration) _eContainer);
        EObject _xifexpression = null;
        if (((container.getImportSpecifiers().size() == 1) && container.getImportSpecifiers().contains(is))) {
          _xifexpression = container;
        } else {
          _xifexpression = is;
        }
        final EObject toBeRemoved = _xifexpression;
        this.remove(toBeRemoved);
      }
    }
  }
  
  private boolean isUsed(final ImportSpecifier importSpec) {
    boolean _isFlaggedUsedInCode = importSpec.isFlaggedUsedInCode();
    boolean _tripleEquals = (Boolean.valueOf(_isFlaggedUsedInCode) == Boolean.valueOf(false));
    if (_tripleEquals) {
      return false;
    } else {
      SymbolTableEntryOriginal _xifexpression = null;
      if ((importSpec instanceof NamedImportSpecifier)) {
        _xifexpression = this.findSymbolTableEntryForNamedImport(((NamedImportSpecifier)importSpec));
      } else {
        SymbolTableEntryOriginal _xifexpression_1 = null;
        if ((importSpec instanceof NamespaceImportSpecifier)) {
          _xifexpression_1 = this.findSymbolTableEntryForNamespaceImport(((NamespaceImportSpecifier)importSpec));
        }
        _xifexpression = _xifexpression_1;
      }
      final SymbolTableEntryOriginal ste = _xifexpression;
      final Function1<ReferencingElement_IM, Boolean> _function = (ReferencingElement_IM it) -> {
        return Boolean.valueOf(TranspilerUtils.isIntermediateModelElement(it));
      };
      final boolean hasReference = IterableExtensions.<ReferencingElement_IM>exists(ste.getReferencingElements(), _function);
      if (hasReference) {
        final IdentifiableElement target = ste.getOriginalTarget();
        return ((target != null) && ScriptDependencyResolver.shouldBeImported(this.getState().resource.getModule(), target));
      }
      return false;
    }
  }
}
