/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.utils;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.transpiler.InformationRegistry;
import org.eclipse.n4js.transpiler.TranspilerState;
import org.eclipse.n4js.transpiler.im.ReferencingElement_IM;
import org.eclipse.n4js.transpiler.im.Script_IM;
import org.eclipse.n4js.transpiler.im.SymbolTable;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.utils.UtilN4;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

/**
 * Some utilities for transpiler debugging, mainly dumping a {@link TranspilerState} to {@code stdout}, etc.
 */
@SuppressWarnings("all")
public class TranspilerDebugUtils {
  /**
   * Perform some consistency checks on the transpiler state. For example, this asserts that no node in the
   * intermediate model has a direct cross-reference to the original AST or an original TModule element.
   */
  public void validateState(final TranspilerState state, final boolean allowDanglingSecondaryReferencesInSTEs) throws AssertionError {
    EClass _parameterizedPropertyAccessExpression = N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression();
    EClass _identifierRef = N4JSPackage.eINSTANCE.getIdentifierRef();
    EClass _parameterizedTypeRef = TypeRefsPackage.eINSTANCE.getParameterizedTypeRef();
    final List<EClass> replacedEClasses = Collections.<EClass>unmodifiableList(CollectionLiterals.<EClass>newArrayList(_parameterizedPropertyAccessExpression, _identifierRef, _parameterizedTypeRef));
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      return Boolean.valueOf(replacedEClasses.contains(it.eClass()));
    };
    final EObject badObject = IteratorExtensions.<EObject>findFirst(state.im.eAllContents(), _function);
    EClass _eClass = null;
    if (badObject!=null) {
      _eClass=badObject.eClass();
    }
    String _name = null;
    if (_eClass!=null) {
      _name=_eClass.getName();
    }
    String _plus = ("intermediate model should not contain objects of type " + _name);
    TranspilerDebugUtils.assertNull(_plus, badObject);
    final Function1<EObject, Boolean> _function_1 = (EObject it) -> {
      boolean _allowedCrossRefToOutside = this.allowedCrossRefToOutside(it, state.info);
      return Boolean.valueOf((!_allowedCrossRefToOutside));
    };
    final Function1<EObject, Boolean> _function_2 = (EObject it) -> {
      return Boolean.valueOf(TranspilerDebugUtils.hasCrossRefToOutsideOf(it, state));
    };
    TranspilerDebugUtils.assertFalse(
      ("intermediate model should not have a cross-reference to an element outside the intermediate model" + " (except for SymbolTableEntry)"), 
      IteratorExtensions.<EObject>exists(IteratorExtensions.<EObject>filter(state.im.eAllContents(), _function_1), _function_2));
    final SymbolTable st = state.im.getSymbolTable();
    TranspilerDebugUtils.assertNotNull("intermediate model should have a symbol table", st);
    EList<SymbolTableEntry> _entries = st.getEntries();
    for (final SymbolTableEntry ste : _entries) {
      {
        if ((ste instanceof SymbolTableEntryOriginal)) {
          TranspilerDebugUtils.assertNotNull("originalTarget should not be null", ((SymbolTableEntryOriginal)ste).getOriginalTarget());
          TranspilerDebugUtils.assertFalse("originalTarget should not be an element in the intermediate model", 
            TranspilerDebugUtils.isElementInIntermediateModelOf(((SymbolTableEntryOriginal)ste).getOriginalTarget(), state));
        }
        if (allowDanglingSecondaryReferencesInSTEs) {
        } else {
          final Function1<NamedElement, Boolean> _function_3 = (NamedElement it) -> {
            return Boolean.valueOf(TranspilerDebugUtils.isElementInIntermediateModelOf(it, state));
          };
          TranspilerDebugUtils.assertTrue("all elementsOfThisName should be elements in the intermediate model", 
            IterableExtensions.<NamedElement>forall(ste.getElementsOfThisName(), _function_3));
          final Function1<ReferencingElement_IM, Boolean> _function_4 = (ReferencingElement_IM it) -> {
            return Boolean.valueOf(TranspilerDebugUtils.isElementInIntermediateModelOf(it, state));
          };
          TranspilerDebugUtils.assertTrue("all referencingElements should be elements in the intermediate model", 
            IterableExtensions.<ReferencingElement_IM>forall(ste.getReferencingElements(), _function_4));
          if ((ste instanceof SymbolTableEntryOriginal)) {
            ImportSpecifier _importSpecifier = ((SymbolTableEntryOriginal)ste).getImportSpecifier();
            boolean _tripleNotEquals = (_importSpecifier != null);
            if (_tripleNotEquals) {
              TranspilerDebugUtils.assertTrue("importSpecifier should be an element in the intermediate model", 
                TranspilerDebugUtils.isElementInIntermediateModelOf(((SymbolTableEntryOriginal)ste).getImportSpecifier(), state));
            }
          }
        }
      }
    }
  }
  
  private boolean allowedCrossRefToOutside(final EObject eobj, final InformationRegistry info) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (eobj instanceof SymbolTableEntry) {
      _matched=true;
      _switchResult = true;
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  private static boolean hasCrossRefToOutsideOf(final EObject elementInIntermediateModel, final TranspilerState state) {
    final Function1<EObject, Boolean> _function = (EObject it) -> {
      boolean _isElementInIntermediateModelOf = TranspilerDebugUtils.isElementInIntermediateModelOf(it, state);
      boolean _not = (!_isElementInIntermediateModelOf);
      if (_not) {
        return Boolean.valueOf(true);
      }
      return Boolean.valueOf(false);
    };
    return IterableExtensions.<EObject>exists(elementInIntermediateModel.eCrossReferences(), _function);
  }
  
  private static boolean isElementInIntermediateModelOf(final EObject eobj, final TranspilerState state) {
    Script_IM _containerOfType = EcoreUtil2.<Script_IM>getContainerOfType(eobj, Script_IM.class);
    return (_containerOfType == state.im);
  }
  
  /**
   * Asserts {@code value} to be <code>true</code> and throws an {@link AssertionError} otherwise.
   */
  public static void assertTrue(final String message, final boolean value) throws AssertionError {
    if ((!value)) {
      TranspilerDebugUtils.assertionFailure(message);
    }
  }
  
  /**
   * Asserts {@code value} to be <code>false</code> and throws an {@link AssertionError} otherwise.
   */
  public static void assertFalse(final String message, final boolean value) throws AssertionError {
    if (value) {
      TranspilerDebugUtils.assertionFailure(message);
    }
  }
  
  /**
   * Asserts {@code value} to be <code>null</code> and throws an {@link AssertionError} otherwise.
   */
  public static void assertNull(final String message, final Object value) throws AssertionError {
    if ((value != null)) {
      TranspilerDebugUtils.assertionFailure(message);
    }
  }
  
  /**
   * Asserts {@code value} to be non-<code>null</code> and throws an {@link AssertionError} otherwise.
   */
  public static void assertNotNull(final String message, final Object value) throws AssertionError {
    if ((value == null)) {
      TranspilerDebugUtils.assertionFailure(message);
    }
  }
  
  private static void assertionFailure(final String message) throws AssertionError {
    final AssertionError ex = new AssertionError(message);
    ex.printStackTrace();
    throw ex;
  }
  
  /**
   * Dumps the transpiler state to the {@code ASTGraphView}.
   * <p>
   * Does nothing if running in headless mode, or if {@code ASTGraphView} is unavailable for some other reason. May be
   * invoked from non-UI threads. It is thus safe to call this method at all times and from everywhere.
   */
  public static void dumpGraph(final TranspilerState state, final String label) {
    UtilN4.takeSnapshotInGraphView(label, state.im);
  }
  
  public static void dump(final TranspilerState state) {
    InputOutput.<String>println(TranspilerDebugUtils.dumpToString(state));
  }
  
  public static String dumpToString(final TranspilerState state) {
    final StringWriter w = new StringWriter();
    TranspilerDebugUtils.dump(state, w);
    return w.toString();
  }
  
  /**
   * Dumps the transpiler state to the given writer.
   */
  public static void dump(final TranspilerState state, final Writer out) {
    final PrintWriter w = new PrintWriter(out);
    TranspilerDebugUtils.dump(w, state.im, 0);
  }
  
  private static void dump(final PrintWriter w, final EObject obj, final int indentLevel) {
    TranspilerDebugUtils.indent(w, indentLevel);
    TranspilerDebugUtils.printObj(w, obj, true, indentLevel);
    w.println();
    EList<EObject> _eContents = obj.eContents();
    for (final EObject child : _eContents) {
      TranspilerDebugUtils.dump(w, child, (indentLevel + 1));
    }
  }
  
  private static void printObj(final PrintWriter w, final EObject obj, final boolean includeCrossRefs, final int indentLevel) {
    w.print(obj.eClass().getName());
    if ((((((obj instanceof IdentifiableElement) || (obj instanceof NamedElement)) || (obj instanceof VariableDeclaration)) || (obj instanceof ImportSpecifier)) || (obj instanceof SymbolTableEntry))) {
      String _hexString = Integer.toHexString(obj.hashCode());
      String _plus = (" @" + _hexString);
      w.print(_plus);
    }
    final String objStr = obj.toString();
    final int idx = objStr.indexOf("(");
    if ((idx >= 0)) {
      String _substring = objStr.substring(idx);
      String _plus_1 = (" " + _substring);
      w.print(_plus_1);
    }
    if (includeCrossRefs) {
      final Function1<EReference, Boolean> _function = (EReference it) -> {
        boolean _isContainment = it.isContainment();
        return Boolean.valueOf((!_isContainment));
      };
      final Iterable<EReference> crossRefs = IterableExtensions.<EReference>filter(obj.eClass().getEAllReferences(), _function);
      boolean _isEmpty = IterableExtensions.isEmpty(crossRefs);
      boolean _not = (!_isEmpty);
      if (_not) {
        for (final EReference ref : crossRefs) {
          {
            w.println();
            TranspilerDebugUtils.indent(w, indentLevel);
            String _name = ref.getName();
            String _plus_2 = ("--" + _name);
            String _plus_3 = (_plus_2 + "--> ");
            w.print(_plus_3);
            boolean _isMany = ref.isMany();
            if (_isMany) {
              w.print("[");
              Object _eGet = obj.eGet(ref);
              final EList<? extends EObject> targets = ((EList<? extends EObject>) _eGet);
              final Iterator<? extends EObject> iter = targets.iterator();
              while (iter.hasNext()) {
                {
                  final EObject currTarget = iter.next();
                  if ((currTarget != null)) {
                    TranspilerDebugUtils.printObj(w, currTarget, false, indentLevel);
                  } else {
                    w.print("null");
                  }
                  boolean _hasNext = iter.hasNext();
                  if (_hasNext) {
                    w.print(", ");
                  }
                }
              }
              w.print("]");
            } else {
              Object _eGet_1 = obj.eGet(ref);
              final EObject target = ((EObject) _eGet_1);
              if ((target != null)) {
                TranspilerDebugUtils.printObj(w, target, false, indentLevel);
              } else {
                w.print("null");
              }
            }
          }
        }
      }
    }
  }
  
  private static void indent(final PrintWriter w, final int indentLevel) {
    ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, indentLevel, true);
    for (final Integer i : _doubleDotLessThan) {
      w.print("\t");
    }
  }
}
