/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling.helper;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.resource.N4JSCache;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.ts.types.TypeAccessModifier;
import org.eclipse.n4js.ui.internal.N4JSActivator;
import org.eclipse.n4js.ui.labeling.N4JSLabelProvider;
import org.eclipse.n4js.ui.labeling.helper.N4JSDecoratorRow;
import org.eclipse.n4js.ui.labeling.helper.N4JSImageDescriptionLibrary;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.resource.FileExtensionProvider;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.validation.IResourceValidator;
import org.eclipse.xtext.validation.Issue;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;

/**
 * Helper class to create and cache image descriptions as triggered by
 * {@link ImageCalculationHelper}.
 * <br /><br />
 * Be sure to use convertToImageDescriptor and convertToImage provided by
 * {@link N4JSLabelProvider} as they do already some caching and registering
 * image descriptions in a image description library. So never use createImage
 * directly.
 * <br /><br />
 * It creates a {@link N4JSDecoratorRow} whenever one or more decorators are needed.
 * Unlike {@code DecorationOverlayIcon}, a {@link N4JSDecoratorRow} doesn't hold references to {@code Image}.
 * <br /><br />
 * For common images like error markers you can use
 * PlatformUI.workbench.sharedImages.getImageDescriptor instead of providing
 * your own image file. Remember, never call dispose on these images.
 * <br /><br />
 * Validation issues calculated for an element are cached in this class, but
 * maybe it would make sense to centralize this, as also the compiler needs
 * validation issues to decide whether it is allowed to compile or not.
 * As validation should only be done for N4JS elements but the label provider
 * also visits all elements in the parse tree even the keywords, that are defined
 * in n4js.xtext, so there is a check that only resources with an expected
 * file extensions (as provided by the file extension provider) are validated.
 * <br /><br />
 * Image descriptions for overlays are fetched from {@link N4JSImageDescriptionLibrary}.
 */
@SuppressWarnings("all")
public class ImageDescriptionHelper {
  /**
   * Keeps track of errors and warnings not as issues but as their URI strings.
   * A fresh instance of this class is populated via calls to {@link #add}.
   * Afterwards, it's queried via {@link #getMaxSeverityAtOrBelow}.
   */
  public static final class IssueSummary {
    public final static ImageDescriptionHelper.IssueSummary EMPTY_SUMMARY = new ImageDescriptionHelper.IssueSummary(null, null);
    
    /**
     * For performance at most one issue representative is tracked for a method.
     */
    public static ImageDescriptionHelper.IssueSummary create(final List<Issue> issues) {
      if (((null == issues) || issues.isEmpty())) {
        return ImageDescriptionHelper.IssueSummary.EMPTY_SUMMARY;
      }
      final Set<String> errors = ImageDescriptionHelper.IssueSummary.deduplicated(issues, Severity.ERROR);
      final Set<String> warnings = ImageDescriptionHelper.IssueSummary.deduplicated(issues, Severity.WARNING);
      warnings.removeAll(errors);
      if ((errors.isEmpty() && warnings.isEmpty())) {
        return ImageDescriptionHelper.IssueSummary.EMPTY_SUMMARY;
      }
      String[] _asCondensedArray = ImageDescriptionHelper.IssueSummary.asCondensedArray(errors);
      String[] _asCondensedArray_1 = ImageDescriptionHelper.IssueSummary.asCondensedArray(warnings);
      return new ImageDescriptionHelper.IssueSummary(_asCondensedArray, _asCondensedArray_1);
    }
    
    /**
     * Once sub-method issues have been collapsed as string-URIs for the method,
     * there may be duplicate string-URIs. This method de-duplicates them.
     */
    private static Set<String> deduplicated(final List<Issue> issues, final Severity severityOfInterest) {
      final Function1<Issue, Boolean> _function = (Issue i) -> {
        return Boolean.valueOf(((i.getSeverity() == severityOfInterest) && (null != i.getUriToProblem())));
      };
      final Iterable<Issue> chosenIssues = IterableExtensions.<Issue>filter(issues, _function);
      final Function1<Issue, String> _function_1 = (Issue i) -> {
        return ImageDescriptionHelper.IssueSummary.minusBody(i.getUriToProblem().toString());
      };
      final Iterable<String> collapsed = IterableExtensions.<Issue, String>map(chosenIssues, _function_1);
      return IterableExtensions.<String>toSet(collapsed);
    }
    
    /**
     * For the purposes of displaying a warning or error decorator on the Outline view,
     * the finer granularity we're interested in is field or method (and not for example some expression within a method).
     */
    private static String minusBody(final String uri) {
      final int idx = uri.lastIndexOf("/@body");
      if ((idx != (-1))) {
        return uri.substring(0, idx);
      }
      return uri;
    }
    
    /**
     * A "condensed array" means null represents the empty array. As GC-friendly as it gets.
     */
    private static String[] asCondensedArray(final Set<String> uris) {
      boolean _isEmpty = uris.isEmpty();
      if (_isEmpty) {
        return null;
      }
      final String[] result = new String[uris.size()];
      final Procedure2<String, Integer> _function = (String uri, Integer idx) -> {
        result[(idx).intValue()] = uri;
      };
      IterableExtensions.<String>forEach(uris, _function);
      return result;
    }
    
    private final String[] errors;
    
    private final String[] warnings;
    
    public IssueSummary(final String[] errors, final String[] warnings) {
      this.errors = errors;
      this.warnings = warnings;
    }
    
    private boolean isEmpty() {
      return ((null == this.errors) && (null == this.warnings));
    }
    
    /**
     * Return the maximum severity (but limited to ERROR or WARNING) for the given AST element or any of its contained elements.
     */
    private Optional<Severity> getMaxSeverityAtOrBelow(final EObject eo) {
      boolean _isEmpty = this.isEmpty();
      if (_isEmpty) {
        return Optional.<Severity>empty();
      }
      EObject _xifexpression = null;
      if ((eo instanceof SyntaxRelatedTElement)) {
        _xifexpression = ((SyntaxRelatedTElement)eo).getAstElement();
      } else {
        _xifexpression = eo;
      }
      final URI uri = EcoreUtil2.getURI(_xifexpression);
      final String currentURI = uri.toString();
      boolean _enclosesAnyOf = this.enclosesAnyOf(currentURI, this.errors);
      if (_enclosesAnyOf) {
        return Optional.<Severity>of(Severity.ERROR);
      }
      boolean _enclosesAnyOf_1 = this.enclosesAnyOf(currentURI, this.warnings);
      if (_enclosesAnyOf_1) {
        return Optional.<Severity>of(Severity.WARNING);
      }
      return Optional.<Severity>empty();
    }
    
    private final static char FORWARD_SLASH = '/';
    
    /**
     * Is the candidate a prefix of some of the strings?
     * In terms of AST nodes: does the candidate contain a problematic AST node?
     */
    private boolean enclosesAnyOf(final String candidate, final String[] problems) {
      if ((null != problems)) {
        int idx = 0;
        while ((idx < problems.length)) {
          {
            final String problem = problems[idx];
            boolean _startsWith = problem.startsWith(candidate);
            if (_startsWith) {
              final int clen = candidate.length();
              int _length = problem.length();
              boolean _tripleEquals = (_length == clen);
              if (_tripleEquals) {
                return true;
              }
              char _charAt = problem.charAt(clen);
              boolean _tripleEquals_1 = (_charAt == ImageDescriptionHelper.IssueSummary.FORWARD_SLASH);
              if (_tripleEquals_1) {
                return true;
              }
            }
            idx++;
          }
        }
      }
      return false;
    }
  }
  
  private static String PLUGIN_ID = "org.eclipse.n4js.ui";
  
  @Inject
  private N4JSCache cache;
  
  /**
   * Injecting language specific instance of {@link FileExtensionProvider} is fine, assuming this class is always called in
   * context of the editor, which is language specific.
   */
  @Inject
  private FileExtensionProvider fileExtensionProvider;
  
  @Inject
  private IResourceValidator resourceValidator;
  
  @Inject
  @Extension
  private N4JSImageDescriptionLibrary n4jsImageDescriptionLibrary;
  
  private N4JSLabelProvider labelProvider;
  
  public void setLabelProvider(final N4JSLabelProvider provider) {
    this.labelProvider = provider;
    this.n4jsImageDescriptionLibrary.setImageDescriptionHelper(this);
  }
  
  public N4JSImageDescriptionLibrary getImageDescriptionLibrary() {
    return this.n4jsImageDescriptionLibrary;
  }
  
  /**
   * Creates an image descriptor for the given image file and adds validation decorators (ERROR or WARNING)
   * if the given EObject has validation issues.
   * <p>
   * IDE-1723 A cache is used behind the scenes but it's still expensive to scan all issues for the whole resource
   * when all we want to know is whether any exist for the AST element in question.
   */
  public ImageDescriptor createValidationAwareImageDescriptor(final EObject eo, final String imageFileName) {
    ImageDescriptor _xblockexpression = null;
    {
      final ImageDescriptor mainImageDescriptor = this.createSimpleImageDescriptor(imageFileName);
      CancelIndicator _xifexpression = null;
      CancelIndicator _cancelIndicator = null;
      if (this.labelProvider!=null) {
        _cancelIndicator=this.labelProvider.cancelIndicator;
      }
      boolean _tripleNotEquals = (null != _cancelIndicator);
      if (_tripleNotEquals) {
        _xifexpression = this.labelProvider.cancelIndicator;
      } else {
        _xifexpression = CancelIndicator.NullImpl;
      }
      final CancelIndicator mon = _xifexpression;
      final Optional<Severity> optSeverity = this.getMaxSeverityAtOrBelow(eo, mon);
      boolean _isPresent = optSeverity.isPresent();
      if (_isPresent) {
        Severity _get = optSeverity.get();
        boolean _equals = Objects.equal(_get, Severity.ERROR);
        if (_equals) {
          final ImageDescriptor decorator = PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_DEC_FIELD_ERROR);
          return this.createDecorationOverlayIcon(mainImageDescriptor, decorator, IDecoration.BOTTOM_LEFT);
        }
        Severity _get_1 = optSeverity.get();
        boolean _equals_1 = Objects.equal(_get_1, Severity.WARNING);
        if (_equals_1) {
          final ImageDescriptor decorator_1 = PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_DEC_FIELD_WARNING);
          return this.createDecorationOverlayIcon(mainImageDescriptor, decorator_1, IDecoration.BOTTOM_LEFT);
        }
      }
      _xblockexpression = mainImageDescriptor;
    }
    return _xblockexpression;
  }
  
  /**
   * Creates an decorated image with using the given main image and placing the given decorator in the
   * corner given by the quadrant. Valid values for quadrant are IDecoration.BOTTOM_LEFT, IDecoration.BOTTOM_RIGHT,
   * IDecoration.TOP_RIGHT, IDecoration.TOP_LEFT.
   * <b/>
   * This method should be used for a single decorator.
   * For two or more decorators see {@link #createDecorationComposite} is better suited.
   */
  public ImageDescriptor createDecorationOverlayIcon(final ImageDescriptor main, final ImageDescriptor decorator, final int quadrant) {
    return new N4JSDecoratorRow(main, quadrant, decorator);
  }
  
  /**
   * returns an ImageDescriptor to display the decorators on top-right.
   * <b/>
   * This method should be used for two or more decorators. For a single decorator {@link #createDecorationOverlayIcon} is better suited.
   */
  public ImageDescriptor createDecorationComposite(final ImageDescriptor main, final ImageDescriptor... decorators) {
    return new N4JSDecoratorRow(main, IDecoration.TOP_RIGHT, (List<ImageDescriptor>)Conversions.doWrapArray(decorators));
  }
  
  /**
   * Look up all issues for the resource of the AST element, cache them,
   * and return the maximum severity (but limited to ERROR or WARNING) for the AST element or any of its contained elements.
   * <p>
   * The above is achieved by lookup using the resource as key. On cache miss, the resource is validated as a whole and the issues cached.
   * Finally those issues are selected based on the AST element's URI and severity of interest.
   * <p>
   * The cache is invalidated upon semantic change in the resource.
   */
  private Optional<Severity> getMaxSeverityAtOrBelow(final EObject eo, final CancelIndicator cancelIndicator) {
    if (((!this.isN4Resource(eo.eResource())) || (!ImageDescriptionHelper.isCompletelyLoaded(((N4JSResource) eo.eResource()))))) {
      return Optional.<Severity>empty();
    }
    final ImageDescriptionHelper.IssueSummary summary = this.getOrElseUpdateSummary(eo, cancelIndicator);
    return summary.getMaxSeverityAtOrBelow(eo);
  }
  
  /**
   * The cache keeps two kinds of entries for each resource: list of issues (these are shared with the transpiler)
   * and summary of issues (of interest to the outline view only).
   * <p>
   * Given a list-of-issues, preparing its summary is fast, no need to check for cancelation in between.
   * The lookup of issue-summary is always preceded by a lookup of list-of-issues. Why? To leverage the cancelation-handling
   * mechanism in {@link N4JSCache#getOrElseUpdateIssues}. Depending on its result, an additional get-or-else-update
   * (this time for issue-summary) can proceed, this time without having to handle cancelations.
   * <p>
   * During an activation of this method the cache lock is held, ie the transpiler can't get issues in the meantime.
   * Same goes in the other direction.
   */
  private ImageDescriptionHelper.IssueSummary getOrElseUpdateSummary(final EObject eo, final CancelIndicator cancelIndicator) {
    final Resource res = eo.eResource();
    synchronized (this.cache) {
      final List<Issue> issues = this.cache.getOrElseUpdateIssues(this.resourceValidator, res, cancelIndicator);
      if (((null == issues) || cancelIndicator.isCanceled())) {
        return ImageDescriptionHelper.IssueSummary.EMPTY_SUMMARY;
      }
      final Provider<ImageDescriptionHelper.IssueSummary> _function = () -> {
        return ImageDescriptionHelper.IssueSummary.create(issues);
      };
      final ImageDescriptionHelper.IssueSummary summary = this.cache.<ImageDescriptionHelper.IssueSummary>get("ImageDescriptionHelper-IssueSummary", res, _function);
      return summary;
    }
  }
  
  /**
   * "N4JSResource completely loaded" means not just TModule loaded from Xtext index, but also the resource's script is available.
   */
  private static boolean isCompletelyLoaded(final N4JSResource res) {
    Script _script = null;
    if (res!=null) {
      _script=res.getScript();
    }
    final Script scr = _script;
    return ((scr != null) && (!scr.eIsProxy()));
  }
  
  private boolean isN4Resource(final Resource res) {
    Set<String> _fileExtensions = this.fileExtensionProvider.getFileExtensions();
    URI _uRI = null;
    if (res!=null) {
      _uRI=res.getURI();
    }
    String _fileExtension = null;
    if (_uRI!=null) {
      _fileExtension=_uRI.fileExtension();
    }
    boolean _contains = _fileExtensions.contains(_fileExtension);
    if (_contains) {
      return (res instanceof N4JSResource);
    }
    return false;
  }
  
  /**
   * returns either the original ImageDescriptor (in case of public access modifier) or an ImageDescriptor showing the access modifier on the bottom right.
   */
  public ImageDescriptor addAccessibiltyImageDecorator(final ImageDescriptor main, final TypeAccessModifier typeAccessModifier) {
    ImageDescriptor _switchResult = null;
    if (typeAccessModifier != null) {
      switch (typeAccessModifier) {
        case PUBLIC:
          _switchResult = main;
          break;
        default:
          ImageDescriptor _xblockexpression = null;
          {
            ImageDescriptor _switchResult_1 = null;
            if (typeAccessModifier != null) {
              switch (typeAccessModifier) {
                case PUBLIC_INTERNAL:
                  _switchResult_1 = this.n4jsImageDescriptionLibrary.createPublicInternalVisibleImageDecorator();
                  break;
                case PROJECT:
                  _switchResult_1 = this.n4jsImageDescriptionLibrary.createProjectVisibleImageDecorator();
                  break;
                case PRIVATE:
                  _switchResult_1 = this.n4jsImageDescriptionLibrary.createPrivateVisibleImageDecorator();
                  break;
                default:
                  _switchResult_1 = this.n4jsImageDescriptionLibrary.createProjectVisibleImageDecorator();
                  break;
              }
            } else {
              _switchResult_1 = this.n4jsImageDescriptionLibrary.createProjectVisibleImageDecorator();
            }
            final ImageDescriptor decorator = _switchResult_1;
            _xblockexpression = this.createDecorationOverlayIcon(main, decorator, IDecoration.BOTTOM_RIGHT);
          }
          _switchResult = _xblockexpression;
          break;
      }
    } else {
      ImageDescriptor _xblockexpression = null;
      {
        ImageDescriptor _switchResult_1 = null;
        if (typeAccessModifier != null) {
          switch (typeAccessModifier) {
            case PUBLIC_INTERNAL:
              _switchResult_1 = this.n4jsImageDescriptionLibrary.createPublicInternalVisibleImageDecorator();
              break;
            case PROJECT:
              _switchResult_1 = this.n4jsImageDescriptionLibrary.createProjectVisibleImageDecorator();
              break;
            case PRIVATE:
              _switchResult_1 = this.n4jsImageDescriptionLibrary.createPrivateVisibleImageDecorator();
              break;
            default:
              _switchResult_1 = this.n4jsImageDescriptionLibrary.createProjectVisibleImageDecorator();
              break;
          }
        } else {
          _switchResult_1 = this.n4jsImageDescriptionLibrary.createProjectVisibleImageDecorator();
        }
        final ImageDescriptor decorator = _switchResult_1;
        _xblockexpression = this.createDecorationOverlayIcon(main, decorator, IDecoration.BOTTOM_RIGHT);
      }
      _switchResult = _xblockexpression;
    }
    return _switchResult;
  }
  
  public ImageDescriptor createSimpleImageDescriptor(final String imageFileName) {
    ImageDescriptor _xifexpression = null;
    if ((imageFileName != null)) {
      ImageDescriptor _xblockexpression = null;
      {
        final ImageDescriptor existingImageDescriptor = this.labelProvider.asImageDescriptor(imageFileName);
        ImageDescriptor _elvis = null;
        if (existingImageDescriptor != null) {
          _elvis = existingImageDescriptor;
        } else {
          ImageDescriptor _imageDescriptorFromPlugin = N4JSActivator.imageDescriptorFromPlugin(ImageDescriptionHelper.PLUGIN_ID, ("icons/" + imageFileName));
          _elvis = _imageDescriptorFromPlugin;
        }
        _xblockexpression = _elvis;
      }
      _xifexpression = _xblockexpression;
    } else {
      _xifexpression = N4JSLabelProvider.getDefaultImageDescriptor();
    }
    return _xifexpression;
  }
}
