/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling.helper;

import com.google.common.base.Objects;
import java.util.Arrays;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.resource.N4JSResourceDescriptionStrategy;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TEnumLiteral;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.TypeAccessModifier;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ui.labeling.EObjectWithContext;
import org.eclipse.xtext.resource.IEObjectDescription;

/**
 * This helper class dispatches the selection of the main image file to use for
 * a given AST (resp. types) element. For classes, interfaces, enumerations, fields
 * and methods there are already different images available for different visibility
 * states so for their selection there is a switch here.
 * <br /><br />
 * As fallback for an unknown element null is returned, this is expected by the API
 * of the label provider.
 */
@SuppressWarnings("all")
public class ImageFileNameCalculationHelper {
  /**
   * calculation of image file name for AST -> delegates to types model, as information is easier to retrieve there
   */
  protected String _getImageFileName(final EObjectWithContext eObjectWithContext) {
    return this.getImageFileName(eObjectWithContext.obj);
  }
  
  protected String _getImageFileName(final Script script) {
    return this.getImageFileName(script.getModule());
  }
  
  protected String _getImageFileName(final N4ClassDeclaration n4ClassDeclaration) {
    return this.getImageFileName(n4ClassDeclaration.getDefinedType());
  }
  
  protected String _getImageFileName(final N4InterfaceDeclaration n4InterfaceDeclaration) {
    return this.getImageFileName(n4InterfaceDeclaration.getDefinedType());
  }
  
  protected String _getImageFileName(final FunctionDeclaration functionDeclaration) {
    return this.getImageFileName(functionDeclaration.getDefinedType());
  }
  
  protected String _getImageFileName(final N4EnumDeclaration n4EnumDeclaration) {
    return this.getImageFileName(n4EnumDeclaration.getDefinedTypeAsEnum());
  }
  
  protected String _getImageFileName(final N4MethodDeclaration n4MethodDeclaration) {
    return this.getImageFileName(n4MethodDeclaration.getDefinedTypeElement());
  }
  
  protected String _getImageFileName(final N4FieldDeclaration n4FieldDeclaration) {
    return this.getImageFileName(n4FieldDeclaration.getDefinedField());
  }
  
  protected String _getImageFileName(final ExportedVariableDeclaration vd) {
    return this.getImageFileName(vd.getDefinedVariable());
  }
  
  protected String _getImageFileName(final ExportedVariableStatement vs) {
    return "var_simple_aggr.gif";
  }
  
  protected String _getImageFileName(final ImportDeclaration importDelaration) {
    return "import_obj.gif";
  }
  
  protected String _getImageFileName(final NamedImportSpecifier namedImportSpecifier) {
    return "import.gif";
  }
  
  /**
   * calculation of image file name for types model
   */
  protected String _getImageFileName(final TModule tClass) {
    return "n4js_module_corr.png";
  }
  
  protected String _getImageFileName(final TClass tClass) {
    return this.getClassFileName(tClass.getTypeAccessModifier());
  }
  
  private String getClassFileName(final TypeAccessModifier modifier) {
    String _switchResult = null;
    if (modifier != null) {
      switch (modifier) {
        case PUBLIC:
          _switchResult = "innerclass_public_obj.png";
          break;
        case PUBLIC_INTERNAL:
          _switchResult = "innerclass_protected_obj.png";
          break;
        case PROJECT:
          _switchResult = "innerclass_default_obj.png";
          break;
        case PRIVATE:
          _switchResult = "innerclass_private_obj.png";
          break;
        default:
          _switchResult = "innerclass_default_obj.png";
          break;
      }
    } else {
      _switchResult = "innerclass_default_obj.png";
    }
    return _switchResult;
  }
  
  protected String _getImageFileName(final TInterface tInterface) {
    return this.getInterfaceFileName(tInterface.getTypeAccessModifier());
  }
  
  private String getInterfaceFileName(final TypeAccessModifier modifier) {
    String _switchResult = null;
    if (modifier != null) {
      switch (modifier) {
        case PUBLIC:
          _switchResult = "innerinterface_public_obj.png";
          break;
        case PUBLIC_INTERNAL:
          _switchResult = "innerinterface_protected_obj.png";
          break;
        case PROJECT:
          _switchResult = "innerinterface_default_obj.png";
          break;
        case PRIVATE:
          _switchResult = "innerinterface_private_obj.png";
          break;
        default:
          _switchResult = "innerinterface_default_obj.png";
          break;
      }
    } else {
      _switchResult = "innerinterface_default_obj.png";
    }
    return _switchResult;
  }
  
  protected String _getImageFileName(final TEnum tEnum) {
    return this.getEnumFileName(tEnum.getTypeAccessModifier());
  }
  
  private String getEnumFileName(final TypeAccessModifier modifier) {
    String _switchResult = null;
    if (modifier != null) {
      switch (modifier) {
        case PUBLIC:
          _switchResult = "enum_obj.png";
          break;
        case PUBLIC_INTERNAL:
          _switchResult = "enum_protected_obj.png";
          break;
        case PROJECT:
          _switchResult = "enum_default_obj.png";
          break;
        case PRIVATE:
          _switchResult = "enum_private_obj.png";
          break;
        default:
          _switchResult = "enum_obj.png";
          break;
      }
    } else {
      _switchResult = "enum_obj.png";
    }
    return _switchResult;
  }
  
  protected String _getImageFileName(final TFunction tFunction) {
    String _switchResult = null;
    TypeAccessModifier _typeAccessModifier = tFunction.getTypeAccessModifier();
    if (_typeAccessModifier != null) {
      switch (_typeAccessModifier) {
        case PUBLIC:
          _switchResult = "n4js_function.png";
          break;
        case PUBLIC_INTERNAL:
          _switchResult = "n4js_function.png";
          break;
        case PROJECT:
          _switchResult = "n4js_function.png";
          break;
        case PRIVATE:
          _switchResult = "n4js_function.png";
          break;
        default:
          _switchResult = "n4js_function.png";
          break;
      }
    } else {
      _switchResult = "n4js_function.png";
    }
    return _switchResult;
  }
  
  protected String _getImageFileName(final TMethod tMethod) {
    String _switchResult = null;
    MemberAccessModifier _memberAccessModifier = tMethod.getMemberAccessModifier();
    if (_memberAccessModifier != null) {
      switch (_memberAccessModifier) {
        case PUBLIC:
          _switchResult = "method_public_obj.gif";
          break;
        case PUBLIC_INTERNAL:
          _switchResult = "method_public_obj.gif";
          break;
        case PROTECTED:
          _switchResult = "method_protected_obj.gif";
          break;
        case PROJECT:
          _switchResult = "method.gif";
          break;
        case PRIVATE:
          _switchResult = "method_private_obj.gif";
          break;
        default:
          _switchResult = "method.gif";
          break;
      }
    } else {
      _switchResult = "method.gif";
    }
    return _switchResult;
  }
  
  protected String _getImageFileName(final FieldAccessor fieldAccessor) {
    String _switchResult = null;
    MemberAccessModifier _memberAccessModifier = fieldAccessor.getMemberAccessModifier();
    if (_memberAccessModifier != null) {
      switch (_memberAccessModifier) {
        case PUBLIC:
          _switchResult = "method_public_obj.gif";
          break;
        case PUBLIC_INTERNAL:
          _switchResult = "method_public_obj.gif";
          break;
        case PROTECTED:
          _switchResult = "method_protected_obj.gif";
          break;
        case PROJECT:
          _switchResult = "method.gif";
          break;
        case PRIVATE:
          _switchResult = "method_private_obj.gif";
          break;
        default:
          _switchResult = "method.gif";
          break;
      }
    } else {
      _switchResult = "method.gif";
    }
    return _switchResult;
  }
  
  protected String _getImageFileName(final TField tField) {
    String _switchResult = null;
    MemberAccessModifier _memberAccessModifier = tField.getMemberAccessModifier();
    if (_memberAccessModifier != null) {
      switch (_memberAccessModifier) {
        case PUBLIC:
          _switchResult = "field_public_obj.gif";
          break;
        case PUBLIC_INTERNAL:
          _switchResult = "field_public_obj.gif";
          break;
        case PROTECTED:
          _switchResult = "method_protected_obj.gif";
          break;
        case PROJECT:
          _switchResult = "field_default_obj.gif";
          break;
        case PRIVATE:
          _switchResult = "field_private_obj.gif";
          break;
        default:
          _switchResult = "field_default_obj.gif";
          break;
      }
    } else {
      _switchResult = "field_default_obj.gif";
    }
    return _switchResult;
  }
  
  protected String _getImageFileName(final TEnumLiteral tEnumLiteral) {
    return "field_public_obj.gif";
  }
  
  protected String _getImageFileName(final TVariable tVariable) {
    return "var_simple.gif";
  }
  
  protected String _getImageFileName(final IEObjectDescription description) {
    String _xblockexpression = null;
    {
      final TypeAccessModifier modifier = this.tryGetAccessModifier(description);
      String _switchResult = null;
      EClass _eClass = description.getEClass();
      boolean _matched = false;
      EClass _tClass = TypesPackage.eINSTANCE.getTClass();
      if (Objects.equal(_eClass, _tClass)) {
        _matched=true;
        _switchResult = this.getClassFileName(modifier);
      }
      if (!_matched) {
        EClass _tInterface = TypesPackage.eINSTANCE.getTInterface();
        if (Objects.equal(_eClass, _tInterface)) {
          _matched=true;
          _switchResult = this.getInterfaceFileName(modifier);
        }
      }
      if (!_matched) {
        EClass _tEnum = TypesPackage.eINSTANCE.getTEnum();
        if (Objects.equal(_eClass, _tEnum)) {
          _matched=true;
          _switchResult = this.getEnumFileName(modifier);
        }
      }
      if (!_matched) {
        _switchResult = null;
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  private TypeAccessModifier tryGetAccessModifier(final IEObjectDescription description) {
    return N4JSResourceDescriptionStrategy.getTypeAccessModifier(description);
  }
  
  protected String _getImageFileName(final Object object) {
    return null;
  }
  
  public String getImageFileName(final Object tClass) {
    if (tClass instanceof TClass) {
      return _getImageFileName((TClass)tClass);
    } else if (tClass instanceof TInterface) {
      return _getImageFileName((TInterface)tClass);
    } else if (tClass instanceof TMethod) {
      return _getImageFileName((TMethod)tClass);
    } else if (tClass instanceof N4ClassDeclaration) {
      return _getImageFileName((N4ClassDeclaration)tClass);
    } else if (tClass instanceof N4InterfaceDeclaration) {
      return _getImageFileName((N4InterfaceDeclaration)tClass);
    } else if (tClass instanceof TEnum) {
      return _getImageFileName((TEnum)tClass);
    } else if (tClass instanceof TFunction) {
      return _getImageFileName((TFunction)tClass);
    } else if (tClass instanceof ExportedVariableDeclaration) {
      return _getImageFileName((ExportedVariableDeclaration)tClass);
    } else if (tClass instanceof N4EnumDeclaration) {
      return _getImageFileName((N4EnumDeclaration)tClass);
    } else if (tClass instanceof N4MethodDeclaration) {
      return _getImageFileName((N4MethodDeclaration)tClass);
    } else if (tClass instanceof FieldAccessor) {
      return _getImageFileName((FieldAccessor)tClass);
    } else if (tClass instanceof TField) {
      return _getImageFileName((TField)tClass);
    } else if (tClass instanceof ExportedVariableStatement) {
      return _getImageFileName((ExportedVariableStatement)tClass);
    } else if (tClass instanceof FunctionDeclaration) {
      return _getImageFileName((FunctionDeclaration)tClass);
    } else if (tClass instanceof N4FieldDeclaration) {
      return _getImageFileName((N4FieldDeclaration)tClass);
    } else if (tClass instanceof TVariable) {
      return _getImageFileName((TVariable)tClass);
    } else if (tClass instanceof ImportDeclaration) {
      return _getImageFileName((ImportDeclaration)tClass);
    } else if (tClass instanceof TEnumLiteral) {
      return _getImageFileName((TEnumLiteral)tClass);
    } else if (tClass instanceof NamedImportSpecifier) {
      return _getImageFileName((NamedImportSpecifier)tClass);
    } else if (tClass instanceof Script) {
      return _getImageFileName((Script)tClass);
    } else if (tClass instanceof TModule) {
      return _getImageFileName((TModule)tClass);
    } else if (tClass instanceof EObjectWithContext) {
      return _getImageFileName((EObjectWithContext)tClass);
    } else if (tClass instanceof IEObjectDescription) {
      return _getImageFileName((IEObjectDescription)tClass);
    } else if (tClass != null) {
      return _getImageFileName(tClass);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(tClass).toString());
    }
  }
}
