/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.labeling.helper;

import com.google.common.base.Objects;
import java.util.Arrays;
import org.eclipse.n4js.n4JS.ExportedVariableStatement;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4idl.N4IDLGlobals;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.ui.labeling.EObjectWithContext;
import org.eclipse.n4js.ui.labeling.N4JSLabelProvider;
import org.eclipse.n4js.ui.outline.N4JSOutlineTreeProvider;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.ui.label.AbstractLabelProvider;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This helper class serves as replacement for the polymorphic dispatch done
 * by {@link AbstractLabelProvider} in favor of
 * Xtend dispatch methods. Here the dispatch of labels to be shown e.g. in
 * the outline view is done. It is called in {@link N4JSLabelProvider#doGetText}.
 * <br /><br />
 * General pattern is to delegate from an AST element to the types element
 * as in the types model the required information to calculate the name
 * is better provided.
 * <br /><br />
 * In general always the name is used, so most cases should be handled by
 * NamedElement (AST) resp. IdentifiableElement (types).
 * <br /><br />
 * This label calculation assumes a certain structure produced by the
 * {@link N4JSOutlineTreeProvider}, e.g. a single element import and
 * wild card imports will produce only one outline node while multiple
 * imports in one import declaration will produce a sub tree with the root
 * node stating the import from module and sub nodes stating each import with
 * name and alias if used.
 * <br /><br />
 * For all unexpected elements the label "\<unknow\>" is created.
 * <br /><br />
 * Please note, that if you want to created styled labels, you have to
 * use {@link StyledTextCalculationHelper}. As styled text application
 * kicks in after the label calculation has been done you can here create
 * an initial label and then append the styled parts in the styled text
 * calculation.
 */
@SuppressWarnings("all")
public class LabelCalculationHelper {
  /**
   * dispatchDoGetText AST -> delegates to types model, as information is easier to retrieve there
   */
  protected String _dispatchDoGetText(final Void _null) {
    return "*null*";
  }
  
  protected String _dispatchDoGetText(final EObjectWithContext objectWithContext) {
    String label = this.dispatchDoGetText(objectWithContext.obj);
    TMember _xifexpression = null;
    if ((objectWithContext.obj instanceof N4MemberDeclaration)) {
      _xifexpression = ((N4MemberDeclaration) objectWithContext.obj).getDefinedTypeElement();
    } else {
      TMember _xifexpression_1 = null;
      if ((objectWithContext.obj instanceof TMember)) {
        _xifexpression_1 = ((TMember) objectWithContext.obj);
      } else {
        _xifexpression_1 = null;
      }
      _xifexpression = _xifexpression_1;
    }
    final TMember member = _xifexpression;
    if ((((member != null) && (member.getContainingType() != null)) && (!Objects.equal(member.getContainingType(), objectWithContext.context)))) {
      String _label = label;
      String _dispatchDoGetText = this.dispatchDoGetText(member.getContainingType());
      String _plus = (" from " + _dispatchDoGetText);
      label = (_label + _plus);
    }
    return label;
  }
  
  protected String _dispatchDoGetText(final Script script) {
    return this.dispatchDoGetText(script.getModule());
  }
  
  protected String _dispatchDoGetText(final ImportDeclaration importDelaration) {
    String _moduleSpecifier = this.getModuleSpecifier(importDelaration.getModule());
    String _plus = ("imports from " + _moduleSpecifier);
    String _xifexpression = null;
    if (((importDelaration.getImportSpecifiers().size() == 1) && 
      (!(IterableExtensions.<ImportSpecifier>head(importDelaration.getImportSpecifiers()) instanceof NamespaceImportSpecifier)))) {
      String _dispatchDoGetText = this.dispatchDoGetText(IterableExtensions.<ImportSpecifier>head(importDelaration.getImportSpecifiers()));
      _xifexpression = (": " + _dispatchDoGetText);
    } else {
      _xifexpression = "";
    }
    return (_plus + _xifexpression);
  }
  
  protected String _dispatchDoGetText(final NamedImportSpecifier namedImportSpecifier) {
    TExportableElement _importedElement = namedImportSpecifier.getImportedElement();
    String _name = null;
    if (_importedElement!=null) {
      _name=_importedElement.getName();
    }
    String _xifexpression = null;
    String _alias = namedImportSpecifier.getAlias();
    boolean _tripleNotEquals = (_alias != null);
    if (_tripleNotEquals) {
      String _alias_1 = namedImportSpecifier.getAlias();
      _xifexpression = (" as " + _alias_1);
    } else {
      _xifexpression = "";
    }
    return (_name + _xifexpression);
  }
  
  protected String _dispatchDoGetText(final N4ClassifierDeclaration it) {
    String _xifexpression = null;
    Type _definedType = it.getDefinedType();
    boolean _tripleEquals = (null == _definedType);
    if (_tripleEquals) {
      _xifexpression = "<unknown>";
    } else {
      _xifexpression = this.dispatchDoGetText(it.getDefinedType());
    }
    return _xifexpression;
  }
  
  protected String _dispatchDoGetText(final N4GetterDeclaration n4GetterDeclaration) {
    return this.dispatchDoGetText(n4GetterDeclaration.getDefinedGetter());
  }
  
  protected String _dispatchDoGetText(final N4SetterDeclaration n4SetterDeclaration) {
    return this.dispatchDoGetText(n4SetterDeclaration.getDefinedSetter());
  }
  
  /**
   * comma separated list of all contained variable names
   */
  protected String _dispatchDoGetText(final ExportedVariableStatement vs) {
    final Function1<VariableDeclaration, String> _function = (VariableDeclaration it) -> {
      return it.getName();
    };
    return IterableExtensions.join(ListExtensions.<VariableDeclaration, String>map(vs.getVarDecl(), _function), ", ");
  }
  
  protected String _dispatchDoGetText(final NamedElement namedElement) {
    return namedElement.getName();
  }
  
  /**
   * dispatchDoGetText types model
   * the fully qualified module specifier, e.g. mypack/MyFile
   */
  protected String _dispatchDoGetText(final TModule tModule) {
    return this.getModuleSpecifier(tModule);
  }
  
  /**
   * name + optional type variables, e.g. A<T, U>
   */
  protected String _dispatchDoGetText(final TClassifier tClassifier) {
    String _name = tClassifier.getName();
    String _typeVersionDescription = this.getTypeVersionDescription(tClassifier);
    String _plus = (_name + _typeVersionDescription);
    String _typeVarDescriptions = this.getTypeVarDescriptions(tClassifier);
    return (_plus + _typeVarDescriptions);
  }
  
  private String getTypeVarDescriptions(final TClassifier tClassifier) {
    String _xifexpression = null;
    int _size = tClassifier.getTypeVars().size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      final Function1<TypeVariable, String> _function = (TypeVariable it) -> {
        return it.getName();
      };
      String _join = IterableExtensions.join(ListExtensions.<TypeVariable, String>map(tClassifier.getTypeVars(), _function), ", ");
      String _plus = ("<" + _join);
      _xifexpression = (_plus + ">");
    } else {
      _xifexpression = "";
    }
    return _xifexpression;
  }
  
  private String getTypeVersionDescription(final TClassifier tClassifier) {
    int _declaredVersion = tClassifier.getDeclaredVersion();
    boolean _notEquals = (_declaredVersion != 0);
    if (_notEquals) {
      String _string = Integer.toString(tClassifier.getDeclaredVersion());
      return (N4IDLGlobals.VERSION_SEPARATOR + _string);
    }
    return "";
  }
  
  protected String _dispatchDoGetText(final TGetter tGetter) {
    return tGetter.getName();
  }
  
  protected String _dispatchDoGetText(final TSetter tSetter) {
    return tSetter.getName();
  }
  
  protected String _dispatchDoGetText(final IdentifiableElement identifiableElement) {
    return identifiableElement.getName();
  }
  
  protected String _dispatchDoGetText(final IEObjectDescription d) {
    if ((TypesPackage.eINSTANCE.getTN4Classifier().isSuperTypeOf(d.getEClass()) || 
      TypesPackage.eINSTANCE.getTEnum().isSuperTypeOf(d.getEClass()))) {
      return d.getQualifiedName().getLastSegment();
    } else {
      return "<unknown>";
    }
  }
  
  protected String _dispatchDoGetText(final Object object) {
    return "<unknown>";
  }
  
  private String getModuleSpecifier(final TModule tModule) {
    String _xifexpression = null;
    String _qualifiedName = null;
    if (tModule!=null) {
      _qualifiedName=tModule.getQualifiedName();
    }
    boolean _tripleNotEquals = (_qualifiedName != null);
    if (_tripleNotEquals) {
      _xifexpression = tModule.getModuleSpecifier();
    } else {
      _xifexpression = "<unknown>";
    }
    return _xifexpression;
  }
  
  public String dispatchDoGetText(final Object tClassifier) {
    if (tClassifier instanceof TClassifier) {
      return _dispatchDoGetText((TClassifier)tClassifier);
    } else if (tClassifier instanceof TGetter) {
      return _dispatchDoGetText((TGetter)tClassifier);
    } else if (tClassifier instanceof TSetter) {
      return _dispatchDoGetText((TSetter)tClassifier);
    } else if (tClassifier instanceof N4ClassifierDeclaration) {
      return _dispatchDoGetText((N4ClassifierDeclaration)tClassifier);
    } else if (tClassifier instanceof N4GetterDeclaration) {
      return _dispatchDoGetText((N4GetterDeclaration)tClassifier);
    } else if (tClassifier instanceof N4SetterDeclaration) {
      return _dispatchDoGetText((N4SetterDeclaration)tClassifier);
    } else if (tClassifier instanceof ExportedVariableStatement) {
      return _dispatchDoGetText((ExportedVariableStatement)tClassifier);
    } else if (tClassifier instanceof ImportDeclaration) {
      return _dispatchDoGetText((ImportDeclaration)tClassifier);
    } else if (tClassifier instanceof NamedImportSpecifier) {
      return _dispatchDoGetText((NamedImportSpecifier)tClassifier);
    } else if (tClassifier instanceof Script) {
      return _dispatchDoGetText((Script)tClassifier);
    } else if (tClassifier instanceof IdentifiableElement) {
      return _dispatchDoGetText((IdentifiableElement)tClassifier);
    } else if (tClassifier instanceof TModule) {
      return _dispatchDoGetText((TModule)tClassifier);
    } else if (tClassifier instanceof NamedElement) {
      return _dispatchDoGetText((NamedElement)tClassifier);
    } else if (tClassifier == null) {
      return _dispatchDoGetText((Void)null);
    } else if (tClassifier instanceof EObjectWithContext) {
      return _dispatchDoGetText((EObjectWithContext)tClassifier);
    } else if (tClassifier instanceof IEObjectDescription) {
      return _dispatchDoGetText((IEObjectDescription)tClassifier);
    } else if (tClassifier != null) {
      return _dispatchDoGetText(tClassifier);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(tClassifier).toString());
    }
  }
}
