/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.utils;

import com.google.inject.Inject;
import java.util.LinkedHashSet;
import java.util.List;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.n4js.ui.utils.IssueUtilN4;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.ui.editor.contentassist.ContentAssistContext;
import org.eclipse.xtext.ui.editor.contentassist.FQNPrefixMatcher;
import org.eclipse.xtext.ui.editor.contentassist.ICompletionProposalAcceptor;
import org.eclipse.xtext.ui.editor.contentassist.IContentProposalProvider;
import org.eclipse.xtext.ui.editor.contentassist.PrefixMatcher;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.util.concurrent.IUnitOfWork;
import org.eclipse.xtext.validation.Issue;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Utility class for adding imports to a document and other import-related modifications.
 * <p>
 * Currently, this simply delegates to the corresponding functionality in Content Assist to allow reuse
 * of this functionality from other parts of the UI implementation (e.g. from a quick fix). This class
 * should be removed when reusable parts (esp. handling of imports) have been factored out of content
 * assist code.
 */
@SuppressWarnings("all")
public class ImportUtil {
  /**
   * An acceptor that simply collects the accepted proposals.
   */
  private static class CollectingAcceptor implements ICompletionProposalAcceptor {
    private final LinkedHashSet<ICompletionProposal> proposals = CollectionLiterals.<ICompletionProposal>newLinkedHashSet();
    
    public LinkedHashSet<ICompletionProposal> get() {
      return this.proposals;
    }
    
    @Override
    public void accept(final ICompletionProposal proposal) {
      this.proposals.add(proposal);
    }
    
    @Override
    public boolean canAcceptMoreProposals() {
      return true;
    }
  }
  
  /**
   * A matcher that makes sure we have an exact match, i.e. name and prefix have same length.
   */
  private static class ExactMatcher extends PrefixMatcher {
    private final PrefixMatcher delegate;
    
    public ExactMatcher(final PrefixMatcher delegate) {
      this.delegate = delegate;
    }
    
    @Override
    public boolean isCandidateMatchingPrefix(final String name, final String prefix) {
      boolean _isCandidateMatchingPrefix = this.delegate.isCandidateMatchingPrefix(name, prefix);
      boolean _not = (!_isCandidateMatchingPrefix);
      if (_not) {
        return false;
      }
      String lastSegment = name;
      if ((this.delegate instanceof FQNPrefixMatcher)) {
        final FQNPrefixMatcher.LastSegmentFinder lsf = ((FQNPrefixMatcher)this.delegate).getLastSegmentFinder();
        final char delimiter = ((FQNPrefixMatcher)this.delegate).getDelimiter();
        int _indexOf = lastSegment.indexOf(delimiter);
        boolean _greaterEqualsThan = (_indexOf >= 0);
        if (_greaterEqualsThan) {
          lastSegment = lsf.getLastSegment(name, delimiter);
          if ((lastSegment == null)) {
            return false;
          }
        }
      }
      int _length = lastSegment.length();
      int _length_1 = prefix.length();
      return (_length == _length_1);
    }
  }
  
  @Inject
  private ContentAssistContext.Factory contentAssistContextFactory;
  
  @Inject
  private IContentProposalProvider contentProposalProvider;
  
  @Inject
  @Extension
  private IssueUtilN4 _issueUtilN4;
  
  /**
   * Collect all possible import candidates for the given issue. It is assumed that the issue spans a name
   * and an import is to be added such that the name can be resolved (as is the case with 'Couldn't resolve
   * reference to ...' issues created by the Xtext linker).
   * <p>
   * Parameter 'allowCompletion' is explained {@link #findImportCandidates(ISourceViewer,IXtextDocument,int,boolean) here}.
   * 
   * @see #findImportCandidates(ISourceViewer,IXtextDocument,int,boolean)
   */
  public List<ICompletionProposal> findImportCandidates(final Issue issue, final boolean allowCompletion) {
    ISourceViewer _viewer = this._issueUtilN4.getViewer(issue);
    IXtextDocument _document = this._issueUtilN4.getDocument(issue);
    Integer _offset = issue.getOffset();
    Integer _length = issue.getLength();
    int _plus = ((_offset).intValue() + (_length).intValue());
    return this.findImportCandidates(_viewer, _document, _plus, allowCompletion);
  }
  
  /**
   * Collect all possible import candidates for the given offset. If 'allowCompletion' is <code>true</code>,
   * this is very similar to pressing CTRL+Space at the given offset.
   * 
   * @param allowCompletion if <code>true</code>, collects imports for all names <em>starting</em> with the
   *                        character sequence before the given offset; if <code>false</code>, collects imports
   *                        only for exact matches.
   */
  public List<ICompletionProposal> findImportCandidates(final ISourceViewer viewer, final IXtextDocument doc, final int offset, final boolean allowCompletion) {
    final IUnitOfWork<List<ICompletionProposal>, XtextResource> _function = (XtextResource resource) -> {
      List<ICompletionProposal> _xblockexpression = null;
      {
        final ImportUtil.CollectingAcceptor collector = new ImportUtil.CollectingAcceptor();
        final ContentAssistContext[] contentAssistContexts = this.createContentAssistContexts(viewer, offset, resource, allowCompletion);
        for (final ContentAssistContext contentAssistContext : contentAssistContexts) {
          this.contentProposalProvider.createProposals(contentAssistContext, collector);
        }
        _xblockexpression = IterableExtensions.<ICompletionProposal>toList(collector.get());
      }
      return _xblockexpression;
    };
    return doc.<List<ICompletionProposal>>readOnly(_function);
  }
  
  private ContentAssistContext[] createContentAssistContexts(final ISourceViewer viewer, final int offset, final XtextResource resource, final boolean allowCompletion) {
    try {
      final ContentAssistContext[] result = this.contentAssistContextFactory.create(viewer, offset, resource);
      if (allowCompletion) {
        return result;
      } else {
        final Function1<ContentAssistContext, ContentAssistContext> _function = (ContentAssistContext it) -> {
          return this.disallowCompletion(it);
        };
        return ((ContentAssistContext[])Conversions.unwrapArray(ListExtensions.<ContentAssistContext, ContentAssistContext>map(((List<ContentAssistContext>)Conversions.doWrapArray(result)), _function), ContentAssistContext.class));
      }
    } catch (final Throwable _t) {
      if (_t instanceof BadLocationException) {
        final BadLocationException e = (BadLocationException)_t;
        throw new RuntimeException(e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * Disallow completion in the given ContentAssistContext by wrapping the contained matcher with an ExactMatcher.
   * Disallowing completion means that in all matching proposals the name must have the same length as the prefix.
   */
  private ContentAssistContext disallowCompletion(final ContentAssistContext context) {
    ContentAssistContext.Builder _copy = context.copy();
    PrefixMatcher _matcher = context.getMatcher();
    ImportUtil.ExactMatcher _exactMatcher = new ImportUtil.ExactMatcher(_matcher);
    return _copy.setMatcher(_exactMatcher).toContext();
  }
}
