/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.ui.wizard.generator;

import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.scoping.N4JSScopeProvider;
import org.eclipse.n4js.scoping.imports.PlainAccessOfAliasedImportDescription;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ui.changes.IAtomicChange;
import org.eclipse.n4js.ui.changes.Replacement;
import org.eclipse.n4js.ui.organize.imports.ImportsRegionHelper;
import org.eclipse.n4js.ui.wizard.generator.ImportAnalysis;
import org.eclipse.n4js.ui.wizard.generator.ImportRequirement;
import org.eclipse.n4js.ui.wizard.generator.WizardGeneratorHelper;
import org.eclipse.n4js.ui.wizard.model.ClassifierReference;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * The N4JSImportRequirementResolver provides functionality to resolve import name conflicts and requirements.
 * 
 * Using N4JSImportRequirementResolver one can compute the remaining imports to be made when inserting code
 * into an existing module. One can also computer alias naming conflicts and how to resolve them.
 */
@SuppressWarnings("all")
public class N4JSImportRequirementResolver {
  @Inject
  private N4JSScopeProvider scopeProvider;
  
  @Inject
  private ImportsRegionHelper hImportsRegion;
  
  /**
   * Analyzes a list of demanded import requirements and a resource in which the requirements should be available.
   * 
   * This includes matching required and existing imports and alias bindings if required imports are aliased
   * in the given resource.
   * 
   * @param demandedImports A list of all import requirements demanded to be included in the resource
   * @param resource The resource to work on.
   * 
   * @return An {@link ImportAnalysis}
   */
  public ImportAnalysis analyzeImportRequirements(final List<ImportRequirement> demandedImports, final XtextResource resource) {
    ArrayList<ImportRequirement> importRequirements = new ArrayList<ImportRequirement>(demandedImports);
    final Map<URI, String> aliasBindings = new HashMap<URI, String>();
    final Script script = N4JSImportRequirementResolver.getScript(resource);
    final List<ImportDeclaration> resourceImportStatements = IteratorExtensions.<ImportDeclaration>toList(Iterators.<ImportDeclaration>filter(script.eAllContents(), ImportDeclaration.class));
    final IScope typeScope = this.scopeProvider.getScope(script, TypeRefsPackage.Literals.PARAMETERIZED_TYPE_REF__DECLARED_TYPE);
    final IScope scopeIdRef = this.scopeProvider.getScope(script, N4JSPackage.Literals.IDENTIFIER_REF__ID);
    for (int i = (importRequirements.size() - 1); (i >= 0); i--) {
      {
        final ImportRequirement dependency = importRequirements.get(i);
        boolean fullfilled = false;
        final Object typeScopeElement = typeScope.getSingleElement(QualifiedName.create(dependency.typeName));
        if ((typeScopeElement instanceof IEObjectDescription)) {
          if ((((IEObjectDescription)typeScopeElement).getEObjectURI().equals(dependency.typeUri) && (!(typeScopeElement instanceof PlainAccessOfAliasedImportDescription)))) {
            importRequirements.remove(i);
            fullfilled = true;
          }
        }
        if ((!fullfilled)) {
          for (int j = (resourceImportStatements.size() - 1); ((j >= 0) && (!fullfilled)); j--) {
            {
              final ImportDeclaration declaration = resourceImportStatements.get(j);
              final ImportSpecifier fulfillingSpecifier = this.findFulfillingImportSpecifier(declaration, dependency);
              if ((fulfillingSpecifier != null)) {
                if ((fulfillingSpecifier instanceof NamedImportSpecifier)) {
                  if (((((NamedImportSpecifier)fulfillingSpecifier).getAlias() != null) && (!((NamedImportSpecifier)fulfillingSpecifier).getAlias().isEmpty()))) {
                    aliasBindings.put(dependency.typeUri, ((NamedImportSpecifier)fulfillingSpecifier).getAlias());
                  }
                  importRequirements.remove(i);
                  fullfilled = true;
                }
              }
            }
          }
        }
      }
    }
    final Function1<String, Boolean> _function = (String name) -> {
      final QualifiedName qualifiedName = QualifiedName.create(name);
      final Object typeScopeElement = typeScope.getSingleElement(qualifiedName);
      if (((typeScopeElement != null) && (typeScopeElement instanceof PlainAccessOfAliasedImportDescription))) {
        return false;
      }
      final Object idScopeElement = scopeIdRef.getSingleElement(qualifiedName);
      return (((typeScopeElement != null) || 
        (idScopeElement != null)) || 
        aliasBindings.containsValue(name));
    };
    final Map<URI, String> nameResolvingAliasBindings = this.resolveImportNameConflicts(importRequirements, _function);
    aliasBindings.putAll(nameResolvingAliasBindings);
    return new ImportAnalysis(importRequirements, aliasBindings);
  }
  
  /**
   * Resolves name conflicts inside the import requirements list, which are name conflicts between elements of the list.
   * Additional conditions are checked through the isUsedNamePredicate to allow for example advanced scope checking.
   * 
   * The method returns a map of alias bindings which is a possible solution to resolve the naming conflicts.
   * 
   * <p>Note: If isUsedNamePredicate is null only inner list name conflicts are resolved</p>
   * 
   * @param importRequirements A list of import requirements
   * @param isUsedNamePredicate Predicate for advanced name checking. May be null.
   * 
   * @return The alias bindings.
   */
  public Map<URI, String> resolveImportNameConflicts(final Collection<ImportRequirement> importRequirements, final Function1<? super String, ? extends Boolean> isUsedNamePredicate) {
    final HashMap<String, Boolean> usedNames = new HashMap<String, Boolean>();
    HashMap<URI, String> aliasBindings = new HashMap<URI, String>();
    for (final ImportRequirement requirement : importRequirements) {
      boolean _isEmpty = requirement.alias.isEmpty();
      if (_isEmpty) {
        String alias = requirement.typeName;
        while (((((isUsedNamePredicate != null) && (isUsedNamePredicate.apply(alias)).booleanValue()) || 
          usedNames.containsKey(alias)) || 
          aliasBindings.containsValue(alias))) {
          alias = ("Alias" + alias);
        }
        boolean _equals = requirement.typeName.equals(alias);
        boolean _not = (!_equals);
        if (_not) {
          requirement.alias = alias;
          aliasBindings.put(requirement.typeUri, alias);
          usedNames.put(alias, Boolean.valueOf(true));
        } else {
          usedNames.put(requirement.typeName, Boolean.valueOf(true));
        }
      }
    }
    return aliasBindings;
  }
  
  /**
   * Return the fulfilling import specifier in the given ImportDeclaration or null if the declaration doesn't fulfill the requirement.
   * 
   * @param declaration The import declaration to search in
   * @param requirement The ImportRequirement to fulfill
   * 
   * @return The fulfilling specifier or null
   */
  private ImportSpecifier findFulfillingImportSpecifier(final ImportDeclaration declaration, final ImportRequirement requirement) {
    Object _xblockexpression = null;
    {
      final URI declarationModuleURI = this.toContainingModuleURI(EcoreUtil.getURI(declaration.getModule()));
      boolean _equals = declarationModuleURI.equals(this.toContainingModuleURI(requirement.typeUri));
      boolean _not = (!_equals);
      if (_not) {
        return null;
      }
      EList<ImportSpecifier> _importSpecifiers = declaration.getImportSpecifiers();
      for (final ImportSpecifier specifier : _importSpecifiers) {
        if ((specifier instanceof NamedImportSpecifier)) {
          TExportableElement importedElement = ((NamedImportSpecifier)specifier).getImportedElement();
          if ((importedElement instanceof TClassifier)) {
            boolean _equals_1 = requirement.typeUri.equals(this.uriOfEObject(importedElement));
            if (_equals_1) {
              return specifier;
            }
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ImportSpecifier)_xblockexpression);
  }
  
  /**
   * Returns an absolute URI which consists of the resource part and the
   * path inside of the resource separated with a '#' character.
   */
  private URI uriOfEObject(final EObject object) {
    return object.eResource().getURI().appendFragment(object.eResource().getURIFragment(object));
  }
  
  /**
   * Returns the URI of the module containing the element with the given URI.
   */
  private URI toContainingModuleURI(final URI uri) {
    return URI.createURI(uri.toString().split("#")[0]);
  }
  
  /**
   * Return the {@link IAtomicChange} to insert the import statement for requirements into the the resource.
   * 
   * @param resource The resource to modify
   * @param requirements The import requirements
   * 
   * @return The change to perform
   */
  public IAtomicChange getImportStatementChanges(final XtextResource resource, final List<ImportRequirement> requirements) {
    Replacement _xblockexpression = null;
    {
      String importStatements = this.generateImportStatements(requirements);
      boolean _isEmpty = importStatements.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        String _importStatements = importStatements;
        importStatements = (_importStatements + WizardGeneratorHelper.LINEBREAK);
      }
      URI _uRI = resource.getURI();
      int _importStatementOffset = this.getImportStatementOffset(resource);
      _xblockexpression = new Replacement(_uRI, _importStatementOffset, 
        0, importStatements);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the offset of import statements in the given resource.
   */
  public int getImportStatementOffset(final XtextResource resource) {
    return this.hImportsRegion.getImportOffset(resource);
  }
  
  /**
   * Generate the code for the given importRequirements
   */
  public String generateImportStatements(final List<ImportRequirement> importRequirements) {
    final Function1<ImportRequirement, String> _function = (ImportRequirement dep) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("import {");
      _builder.append(dep.typeName);
      {
        boolean _isEmpty = dep.alias.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          _builder.append(" as ");
          _builder.append(dep.alias);
        }
      }
      _builder.append("} from \"");
      _builder.append(dep.moduleSpecifier);
      _builder.append("\"");
      return _builder.toString();
    };
    return IterableExtensions.join(ListExtensions.<ImportRequirement, String>map(importRequirements, _function), WizardGeneratorHelper.LINEBREAK);
  }
  
  /**
   * @param xtextResource
   *            the resource to process.
   * @return Script instance or null
   */
  private static Script getScript(final XtextResource xtextResource) {
    boolean _isEmpty = xtextResource.getContents().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final EObject eo = xtextResource.getContents().get(0);
      if ((eo instanceof Script)) {
        return ((Script)eo);
      }
    }
    return null;
  }
  
  /**
   * Convert {@link ClassifierReference}s to import requirements without alias
   */
  public static List<ImportRequirement> classifierReferencesToImportRequirements(final List<ClassifierReference> refs) {
    final Function1<ClassifierReference, ImportRequirement> _function = (ClassifierReference ref) -> {
      return N4JSImportRequirementResolver.classifierReferenceToImportRequirement(ref);
    };
    return ListExtensions.<ClassifierReference, ImportRequirement>map(refs, _function);
  }
  
  /**
   * Convert a {@link ClassifierReference} to an import requirement without alias
   */
  public static ImportRequirement classifierReferenceToImportRequirement(final ClassifierReference reference) {
    return new ImportRequirement(reference.classifierName, "", reference.classifierModuleSpecifier, reference.uri);
  }
}
