/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils.collections;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Stream;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Utility functions for collections
 */
@SuppressWarnings("all")
public class Collections2 {
  /**
   * Private constructor to prevent instantiation.
   */
  private Collections2() {
  }
  
  /**
   * Creates a new linked list containing the given elements. This just delegates to
   * {@link CollectionLiterals#newLinkedList}.
   * 
   * @param initial the initial elements of the list
   * 
   * @return the newly created linked list
   */
  public static <T extends Object> LinkedList<T> newLinkedList(final T... initial) {
    return CollectionLiterals.<T>newLinkedList(initial);
  }
  
  /**
   * Creates a new array-backed list containing the given elements. This just delegates to
   * {@link CollectionLiterals#newArrayList}.
   * 
   * @param initial the initial elements of the list
   * 
   * @return the newly created array-backed list
   */
  public static <T extends Object> ArrayList<T> newArrayList(final T... initial) {
    return CollectionLiterals.<T>newArrayList(initial);
  }
  
  /**
   * Creates a new hash set containing the given elements. This just delegates to
   * {@link CollectionLiterals#newHashSet}.
   * 
   * @param initial the initial elements of the set
   * 
   * @return the newly created hash set
   */
  public static <T extends Object> HashSet<T> newHashSet(final T... initial) {
    return CollectionLiterals.<T>newHashSet(initial);
  }
  
  /**
   * Creates a new linked hash set containing the given elements. This just delegates to
   * {@link CollectionLiterals#newLinkedHashSet}.
   * 
   * @param initial the initial elements of the set
   * 
   * @return the newly created linked hash set
   */
  public static <T extends Object> LinkedHashSet<T> newLinkedHashSet(final T... initial) {
    return CollectionLiterals.<T>newLinkedHashSet(initial);
  }
  
  /**
   * Creates a new tree set containing the given elements. This just delegates to
   * {@link CollectionLiterals#newTreeSet}.
   * 
   * @param comparator the comparator to use
   * @param initial the initial elements of the set
   * 
   * @return the newly created tree set
   */
  public static <T extends Object> TreeSet<T> newTreeSet(final Comparator<? super T> comparator, final T... initial) {
    return CollectionLiterals.<T>newTreeSet(comparator, initial);
  }
  
  /**
   * Creates a new hash set containing the given elements. This just delegates to
   * {@link CollectionLiterals#newHashMap}.
   * 
   * @param initial the initial elements of the map
   * 
   * @return the newly created hash map
   */
  public static <K extends Object, V extends Object> HashMap<K, V> newHashMap(final Pair<? extends K, ? extends V>... initial) {
    return CollectionLiterals.<K, V>newHashMap(initial);
  }
  
  /**
   * Creates a new linked hash set containing the given elements. This just delegates to
   * {@link CollectionLiterals#newLinkedHashMasp}.
   * 
   * @param initial the initial elements of the map
   * 
   * @return the newly created linked hash map
   */
  public static <K extends Object, V extends Object> LinkedHashMap<K, V> newLinkedHashMap(final Pair<? extends K, ? extends V>... initial) {
    return CollectionLiterals.<K, V>newLinkedHashMap(initial);
  }
  
  /**
   * Creates a new linked hash set containing the given elements. This just delegates to
   * {@link CollectionLiterals#newTreeMap}.
   * 
   * @param initial the initial elements of the map
   * 
   * @return the newly created linked hash map
   */
  public static <K extends Object, V extends Object> TreeMap<K, V> newTreeMap(final Comparator<? super K> comparator, final Pair<? extends K, ? extends V>... initial) {
    return CollectionLiterals.<K, V>newTreeMap(comparator, initial);
  }
  
  /**
   * Concatenate the given lists while omitting duplicates.
   * 
   * @param listA
   *            the first list, must not be <code>null</code>
   * @param listB
   *            the second list, must not be <code>null</code>
   * @return the unique concatenation of the given lists
   */
  public static <T extends Object> List<T> concatUnique(final List<T> listA, final List<T> listB) {
    Objects.<List<T>>requireNonNull(listA);
    Objects.<List<T>>requireNonNull(listB);
    final Set<T> result = Collections2.<T>newLinkedHashSet(((T[])Conversions.unwrapArray(listA, Object.class)));
    result.addAll(listB);
    return Collections2.<T>newLinkedList(((T[])Conversions.unwrapArray(result, Object.class)));
  }
  
  /**
   * Returns a stream of all pairs that can be formed with the elements of the given collection.
   * 
   * Example: <code> [1,2,3] -> (1,1), (1,2), (1,3), (2,1), (2,2), (2,3), (3,1), (3,2), (3,3) </code>
   */
  public static <T extends Object> Stream<Pair<T, T>> pairs(final Collection<T> collection) {
    final Function<T, Stream<? extends Pair<T, T>>> _function = (T e1) -> {
      final Function<T, Pair<T, T>> _function_1 = (T e2) -> {
        return Pair.<T, T>of(e1, e2);
      };
      return collection.stream().<Pair<T, T>>map(_function_1);
    };
    return collection.stream().<Pair<T, T>>flatMap(_function);
  }
}
