/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.versioning;

import com.google.common.base.Optional;
import com.google.inject.Inject;
import java.util.Collections;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4idl.scoping.VersionScopeProvider;
import org.eclipse.n4js.n4idl.versioning.VersionUtils;
import org.eclipse.n4js.ts.typeRefs.VersionedReference;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.TVersionable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.versions.VersionableUtils;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Contains helper methods to determine version related information of objects as well as to find a specific version
 * of a versionable element or members.
 */
@SuppressWarnings("all")
public class VersionHelper {
  @Inject
  private VersionScopeProvider versionScopeProvider;
  
  @Inject
  private IQualifiedNameConverter nameConverter;
  
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  /**
   * Computes the maximum version for which the given object applies. The following cases are considered.
   * 
   * <ul>
   * <li>If the given object is <code>null</code> or has no version, then {@link Integer#MAX_VALUE} is returned.</li>
   * <li>If the given object is the maximum available version, then {@link Integer#MAX_VALUE} is returned.</li>
   * <li>Otherwise, the no of the next higher version of the given object minus one is returned.</li>
   * </ul>
   * 
   * <p>
   * As an example, assume that there is a class <code>A</code> with versions 1 and 4, then calling this method with
   * <code>A#1</code> as parameter, this method will return 3. If called with <code>A#4</code>, then this method will
   * return {@link Integer#MAX_VALUE}.
   * </p>
   * 
   * @param object
   *            the object for which to compute the maximum version
   * @return the maximum version of the given object or {@link Optional.absent()} if no context version can be determined for the given object.
   */
  public Optional<Integer> computeMaximumVersion(final EObject object) {
    if ((object == null)) {
      return Optional.<Integer>absent();
    }
    boolean _matched = false;
    if (object instanceof VersionedReference) {
      boolean _hasRequestedVersion = ((VersionedReference)object).hasRequestedVersion();
      if (_hasRequestedVersion) {
        _matched=true;
        return Optional.<Integer>of(Integer.valueOf(((VersionedReference)object).getRequestedVersionOrZero()));
      }
    }
    if (!_matched) {
      if (object instanceof N4ClassDeclaration) {
        boolean _isVersioned = VersionUtils.isVersioned(object);
        if (_isVersioned) {
          _matched=true;
          Type _definedType = ((N4ClassDeclaration)object).getDefinedType();
          return this.computeMaximumVersionForVersionable(((TClass) _definedType));
        }
      }
    }
    if (!_matched) {
      if (object instanceof N4InterfaceDeclaration) {
        boolean _isVersioned = VersionUtils.isVersioned(object);
        if (_isVersioned) {
          _matched=true;
          Type _definedType = ((N4InterfaceDeclaration)object).getDefinedType();
          return this.computeMaximumVersionForVersionable(((TInterface) _definedType));
        }
      }
    }
    if (!_matched) {
      if (object instanceof N4EnumDeclaration) {
        boolean _isVersioned = VersionUtils.isVersioned(object);
        if (_isVersioned) {
          _matched=true;
          Type _definedType = ((N4EnumDeclaration)object).getDefinedType();
          return this.computeMaximumVersionForVersionable(((TEnum) _definedType));
        }
      }
    }
    if (!_matched) {
      if (object instanceof FunctionDeclaration) {
        boolean _isVersioned = VersionUtils.isVersioned(object);
        if (_isVersioned) {
          _matched=true;
          Type _definedType = ((FunctionDeclaration)object).getDefinedType();
          return this.computeMaximumVersionForVersionable(((TFunction) _definedType));
        }
      }
    }
    if (!_matched) {
      if (object instanceof TClassifier) {
        _matched=true;
        return this.computeMaximumVersionForVersionable(((TVersionable)object));
      }
    }
    return this.computeMaximumVersion(object.eContainer());
  }
  
  /**
   * Computes the version range of the given versionable type. The lower limit of the version range is determined by the
   * declared version of the given classifier while the upper limit is computed via a call to {@link #computeUpperLimit(TClassifier, int)}.
   */
  private Optional<Integer> computeMaximumVersionForVersionable(final TVersionable versionableType) {
    if ((versionableType == null)) {
      return Optional.<Integer>absent();
    }
    final int lowerLimit = versionableType.getVersion();
    return this.computeUpperLimit(versionableType, lowerLimit);
  }
  
  /**
   * Computes the upper limit of the version range of the given (versionable) type using the given lower limit.
   * 
   * <p>
   * The upper limit is computed by looking at all versions of the given type that are visible in the module
   * containing the given classifier and selecting the version with the lowest version number that is larger than the
   * given lower limit.
   * </p>
   * 
   * <p>
   * If no such version of the given type can be found, then {@link Integer#MAX_VALUE} is returned.
   * </p>
   */
  private Optional<Integer> computeUpperLimit(final TVersionable versionable, final int lowerLimit) {
    final IScope scope = this.versionScopeProvider.getVersionScope(versionable);
    final QualifiedName name = this.nameConverter.toQualifiedName(versionable.getName());
    final Iterable<IEObjectDescription> elements = scope.getElements(name);
    final Function1<IEObjectDescription, Boolean> _function = (IEObjectDescription it) -> {
      return Boolean.valueOf(((it.getEObjectOrProxy() instanceof Type) && (it.getEObjectOrProxy() instanceof TVersionable)));
    };
    final Function1<IEObjectDescription, Type> _function_1 = (IEObjectDescription it) -> {
      EObject _eObjectOrProxy = it.getEObjectOrProxy();
      return ((Type) _eObjectOrProxy);
    };
    final Function1<Type, Boolean> _function_2 = (Type it) -> {
      int _version = it.getVersion();
      return Boolean.valueOf((_version > lowerLimit));
    };
    final Function2<Integer, Type, Integer> _function_3 = (Integer u, Type c) -> {
      int _version = c.getVersion();
      int _minus = (_version - 1);
      return Integer.valueOf(Integer.min((u).intValue(), _minus));
    };
    return Optional.<Integer>of(
      IterableExtensions.<Type, Integer>fold(IterableExtensions.<Type>filter(IterableExtensions.<IEObjectDescription, Type>map(IterableExtensions.<IEObjectDescription>filter(elements, _function), _function_1), _function_2), Integer.valueOf(Integer.MAX_VALUE), _function_3));
  }
  
  /**
   * Finds the {@link IEObjectDescription} of all versions of the given TClassifier.
   */
  public <T extends Type> Iterable<? extends T> findTypeVersions(final T type) {
    if (((type instanceof TObjectPrototype) || 
      (!VersionableUtils.isTVersionable(type)))) {
      return Collections.<T>singleton(type);
    }
    final IScope scope = this.versionScopeProvider.getVersionScope(type);
    final QualifiedName name = this.nameConverter.toQualifiedName(type.getName());
    final Iterable<IEObjectDescription> elements = scope.getElements(name);
    final Function1<IEObjectDescription, EObject> _function = (IEObjectDescription d) -> {
      return d.getEObjectOrProxy();
    };
    final Function1<EObject, Boolean> _function_1 = (EObject o) -> {
      return Boolean.valueOf(type.getClass().isInstance(o));
    };
    final Function1<EObject, T> _function_2 = (EObject t) -> {
      Type _cast = type.getClass().cast(t);
      return ((T) _cast);
    };
    return IterableExtensions.<EObject, T>map(IterableExtensions.<EObject>filter(IterableExtensions.<IEObjectDescription, EObject>map(elements, _function), _function_1), _function_2);
  }
  
  /**
   * Finds the latest version of the given classifier that is not greater than the given version.
   * 
   * @param classifier
   *            the classifier to search for
   * @param version
   *            the maximum version to return
   * @return the requested version of the given classifier, or <code>null</code> if no such version exists
   */
  public <T extends Type> T findTypeWithVersion(final T type, final int version) {
    if ((type instanceof TObjectPrototype)) {
      return type;
    }
    final Iterable<? extends T> elements = this.<T>findTypeVersions(type);
    final Function1<T, Boolean> _function = (T it) -> {
      int _version = it.getVersion();
      return Boolean.valueOf((_version <= version));
    };
    final Function2<T, T, T> _function_1 = (T l, T c) -> {
      T _xifexpression = null;
      int _version = l.getVersion();
      int _version_1 = c.getVersion();
      boolean _greaterThan = (_version > _version_1);
      if (_greaterThan) {
        _xifexpression = l;
      } else {
        _xifexpression = c;
      }
      return _xifexpression;
    };
    return IterableExtensions.<T>reduce(IterableExtensions.filter(elements, _function), _function_1);
  }
  
  /**
   * Finds the latest version of the given member that is not greater than the given version. Since members are not
   * versioned individually, this is equivalent to finding the container of the given member in the requested version,
   * and returning the member with the same name and type from that container version.
   * 
   * @param member
   *            the member to search for
   * @param version
   *            the maximum version to return
   * @return the requested version of the given member, or <code>null</code> if no such version exists
   */
  public <T extends TMember> T findMemberWithVersion(final T member, final int version) {
    final ContainerType<?> container = member.getContainingType();
    if ((container instanceof TClassifier)) {
      final TClassifier containerInRequestedVersion = this.<TClassifier>findTypeWithVersion(((TClassifier)container), version);
      final ContainerTypesHelper.MemberCollector memberCollector = this.containerTypesHelper.fromContext(containerInRequestedVersion);
      final TMember result = memberCollector.findMember(containerInRequestedVersion, member.getName(), false, member.isStatic());
      return ((T) result);
    }
    return member;
  }
}
