/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.scoping.utils;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.Scopes;
import org.eclipse.xtext.scoping.impl.MapBasedScope;
import org.eclipse.xtext.scoping.impl.MultimapBasedScope;
import org.eclipse.xtext.scoping.impl.SimpleScope;
import org.eclipse.xtext.util.SimpleAttributeResolver;

/**
 * Some utility methods, similar to xtext's {@link Scopes}.
 */
@SuppressWarnings("all")
public class ScopesHelper {
  @Inject
  private JavaScriptVariantHelper javaScriptVariantHelper;
  
  /**
   * Creates a map based scope for the given iterable of descriptions.
   * 
   * @param context The context of the scope
   * @param descriptions The descriptions
   */
  public IScope mapBasedScopeFor(final EObject context, final Iterable<IEObjectDescription> descriptions) {
    return this.mapBasedScopeFor(context, IScope.NULLSCOPE, descriptions);
  }
  
  /**
   * Creates a map based scope for the given iterable of descriptions.
   * 
   * @param context The context of the scope
   * @param parent The parent scope
   * @param descriptions The descriptions
   */
  public IScope mapBasedScopeFor(final EObject context, final IScope parent, final Iterable<IEObjectDescription> descriptions) {
    boolean _isMultiQNScope = this.javaScriptVariantHelper.isMultiQNScope(context);
    if (_isMultiQNScope) {
      return MultimapBasedScope.createScope(parent, descriptions, false);
    } else {
      return MapBasedScope.createScope(parent, descriptions);
    }
  }
  
  /**
   * Convenience method for {@link #scopeFor(Iterable,Function,Function,IScope)}.
   */
  public IScope scopeFor(final Iterable<? extends EObject> elements, final Function<IEObjectDescription, ? extends IEObjectDescription> wrapper) {
    return this.scopeFor(elements, wrapper, IScope.NULLSCOPE);
  }
  
  /**
   * Convenience method for {@link #scopeFor(Iterable,Function,Function,IScope)}.
   */
  public IScope scopeFor(final Iterable<? extends EObject> elements, final Function<IEObjectDescription, ? extends IEObjectDescription> wrapper, final IScope outer) {
    return this.<EObject>scopeFor(elements, QualifiedName.<EObject>wrapper(SimpleAttributeResolver.NAME_RESOLVER), wrapper, outer);
  }
  
  /**
   * Similar to {@link Scopes#scopeFor(Iterable,Function,IScope)} but supports custom wrapping
   * of the IEObjectDescriptions, for example to wrap them with error message providing subclasses
   * such as {@link WrongStaticAccessDescription}. The wrapper can return the object description
   * unchanged, create and return a new one or may return null to remove the corresponding object
   * from the scope.
   */
  public <T extends EObject> IScope scopeFor(final Iterable<? extends T> elements, final Function<T, QualifiedName> nameComputation, final Function<IEObjectDescription, ? extends IEObjectDescription> wrapper, final IScope outer) {
    Iterable<IEObjectDescription> _transform = Iterables.<IEObjectDescription, IEObjectDescription>transform(Scopes.<T>scopedElementsFor(elements, nameComputation), wrapper);
    return new SimpleScope(outer, _transform);
  }
}
