/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem.utils;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelperStrategy;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Type System Helper Strategy computing the meet of a given collection of types.
 * <p>
 * Definition from [Pierce02a, pp. 218]:
 * <pre>
 * Meet M = S ^ T, if
 *     M <: S, M <: T,                    common sub type
 *     forall L: L<:T and L<:S => L<:M    least
 * </pre>
 */
@SuppressWarnings("all")
public class MeetComputer extends TypeSystemHelperStrategy {
  @Inject
  private N4JSTypeSystem ts;
  
  /**
   * Returns the meet (first common sub type) of the given types, maybe an intersection type, if types have no declared meet.
   * This may be an intersection type, but not a union type. This method considers generics, but it does not
   * compute lower or upper bounds of them.
   * See class description for details.
   * @return The meet which might be an intersection type.
   * This method only returns null if all input type refs are null.
   * Note that the returned type ref is a newly created type ref which may be inserted into any container.
   */
  public TypeRef meet(final RuleEnvironment G, final Iterable<? extends TypeRef> typeRefs) {
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(typeRefs);
    if (_isNullOrEmpty) {
      return null;
    }
    final Function1<TypeRef, Iterable<? extends TypeRef>> _function = (TypeRef it) -> {
      return this.flattenTypeRefs(it);
    };
    final Iterable<TypeRef> flatTypeRefs = Iterables.<TypeRef>concat(IterableExtensions.map(typeRefs, _function));
    TypeRef meet = null;
    for (final TypeRef tr : flatTypeRefs) {
      if ((meet == null)) {
        meet = TypeUtils.<TypeRef>copyIfContained(tr);
      } else {
        boolean _subtypeSucceeded = this.ts.subtypeSucceeded(G, tr, meet);
        if (_subtypeSucceeded) {
          meet = TypeUtils.<TypeRef>copyIfContained(tr);
        } else {
          boolean _subtypeSucceeded_1 = this.ts.subtypeSucceeded(G, meet, tr);
          boolean _not = (!_subtypeSucceeded_1);
          if (_not) {
            meet = this.intersectRelaxed(G, meet, tr);
          }
        }
      }
    }
    return meet;
  }
  
  private Iterable<? extends TypeRef> _flattenTypeRefs(final TypeRef ref) {
    return Collections.<TypeRef>singleton(ref);
  }
  
  private Iterable<? extends TypeRef> _flattenTypeRefs(final ComposedTypeRef ref) {
    return ref.getTypeRefs();
  }
  
  /**
   * Creates the intersection according to [N4JS, 4.13 Intersection Type], but does not check for uniqueness
   * of class in the typerefs of the intersection.
   * The given type ref instances are only copied when added to the intersection
   * if they have a container, otherwise the method assumes that the caller has newly created the typerefs which can be added directly.
   * If one of the given type refs is any, then this any ref is returned; if the intersection contains only a single type, then this single type ios returned.
   */
  @VisibleForTesting
  public TypeRef intersectRelaxed(final RuleEnvironment G, final TypeRef... typeRefs) {
    final LinkedList<TypeRef> intersectTRs = new LinkedList<TypeRef>();
    final Function1<TypeRef, Iterable<TypeRef>> _function = (TypeRef it) -> {
      return this.flattenIntersectionTypes(it);
    };
    final Iterable<TypeRef> flattenedTypeRefs = Iterables.<TypeRef>concat(ListExtensions.<TypeRef, Iterable<TypeRef>>map(((List<TypeRef>)Conversions.doWrapArray(typeRefs)), _function));
    final Function1<TypeRef, Boolean> _function_1 = (TypeRef it) -> {
      return Boolean.valueOf(((it != null) && it.isTopType()));
    };
    final TypeRef containedAny = IterableExtensions.<TypeRef>findFirst(flattenedTypeRefs, _function_1);
    if ((containedAny != null)) {
      intersectTRs.add(containedAny);
    } else {
      intersectTRs.addAll(this._typeSystemHelper.getSubtypesOnly(G, ((TypeRef[])Conversions.unwrapArray(flattenedTypeRefs, TypeRef.class))));
    }
    int _size = intersectTRs.size();
    boolean _equals = (_size == 1);
    if (_equals) {
      final TypeRef tR = intersectTRs.get(0);
      return TypeUtils.<TypeRef>copyIfContained(tR);
    }
    final IntersectionTypeExpression intersection = TypeRefsFactory.eINSTANCE.createIntersectionTypeExpression();
    for (final TypeRef s : intersectTRs) {
      intersection.getTypeRefs().add(TypeUtils.<TypeRef>copyIfContained(s));
    }
    return intersection;
  }
  
  private Iterable<TypeRef> flattenIntersectionTypes(final TypeRef typeRef) {
    Iterable<TypeRef> _switchResult = null;
    boolean _matched = false;
    if (typeRef instanceof IntersectionTypeExpression) {
      _matched=true;
      final Function1<TypeRef, Iterable<TypeRef>> _function = (TypeRef it) -> {
        return this.flattenIntersectionTypes(it);
      };
      _switchResult = Iterables.<TypeRef>concat(ListExtensions.<TypeRef, Iterable<TypeRef>>map(((IntersectionTypeExpression)typeRef).getTypeRefs(), _function));
    }
    if (!_matched) {
      _switchResult = Collections.<TypeRef>singleton(typeRef);
    }
    return _switchResult;
  }
  
  private Iterable<? extends TypeRef> flattenTypeRefs(final TypeRef ref) {
    if (ref instanceof ComposedTypeRef) {
      return _flattenTypeRefs((ComposedTypeRef)ref);
    } else if (ref != null) {
      return _flattenTypeRefs(ref);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(ref).toString());
    }
  }
}
