/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.utils.validation.PostValidation;
import org.eclipse.n4js.validation.AbstractMessageAdjustingN4JSValidator;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Creates issues for unnecessary IDEBUG annotations.
 * <p>
 * The list of of used IDEBUG annotations
 * is managed by {@link AbstractN4JSDeclarativeValidator}.
 * 
 * @see AbstractN4JSDeclarativeValidator
 */
@SuppressWarnings("all")
public class IDEBUGValidator extends AbstractMessageAdjustingN4JSValidator {
  public final static String DEFINED_IDEBUGS_KEY = (AnnotationDefinition.IDEBUG.name + "_defined");
  
  public final static String USED_IDEBUGS_KEY = (AnnotationDefinition.IDEBUG.name + "_used");
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  public static Collection<Annotation> getDefinedAnnotations(final Map<Object, Object> context) {
    Object _get = context.get(IDEBUGValidator.DEFINED_IDEBUGS_KEY);
    Collection<Annotation> annotations = ((Collection<Annotation>) _get);
    if ((annotations == null)) {
      HashSet<Annotation> _hashSet = new HashSet<Annotation>();
      annotations = _hashSet;
      context.put(IDEBUGValidator.DEFINED_IDEBUGS_KEY, annotations);
    }
    return annotations;
  }
  
  public static Collection<Annotation> getUsedAnnotations(final Map<Object, Object> context) {
    Object _get = context.get(IDEBUGValidator.USED_IDEBUGS_KEY);
    Collection<Annotation> annotations = ((Collection<Annotation>) _get);
    if ((annotations == null)) {
      HashSet<Annotation> _hashSet = new HashSet<Annotation>();
      annotations = _hashSet;
      context.put(IDEBUGValidator.USED_IDEBUGS_KEY, annotations);
    }
    return annotations;
  }
  
  @Check
  public void checkDefinedIDEBUGAnnotation(final Annotation annotation) {
    boolean _equals = AnnotationDefinition.IDEBUG.name.equals(annotation.getName());
    if (_equals) {
      IDEBUGValidator.getDefinedAnnotations(this.getContext()).add(annotation);
    }
  }
  
  @Check
  public void checkUnusedIDEBUGAnnotation(final PostValidation postValidation) {
    final Collection<Annotation> definedAnnotations = IDEBUGValidator.getDefinedAnnotations(this.getContext());
    final Collection<Annotation> usedAnnotations = IDEBUGValidator.getUsedAnnotations(this.getContext());
    final Function1<Annotation, Boolean> _function = (Annotation it) -> {
      boolean _contains = usedAnnotations.contains(it);
      return Boolean.valueOf((!_contains));
    };
    Iterable<Annotation> _filter = IterableExtensions.<Annotation>filter(definedAnnotations, _function);
    for (final Annotation a : _filter) {
      boolean _isEmpty = a.getArgs().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final String bugID = a.getArgs().get(0).getValueAsString();
        this.addIssue(IssueCodes.getMessageForANN_UNUSED_IDEBUG(bugID), a, IssueCodes.ANN_UNUSED_IDEBUG);
      }
    }
  }
}
