/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators.packagejson;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Strings;
import com.google.common.base.Supplier;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.Multimap;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.N4JSGlobals;
import org.eclipse.n4js.external.ExternalIndexSynchronizer;
import org.eclipse.n4js.external.ExternalLibraryWorkspace;
import org.eclipse.n4js.external.ShadowingInfoHelper;
import org.eclipse.n4js.json.JSON.JSONArray;
import org.eclipse.n4js.json.JSON.JSONDocument;
import org.eclipse.n4js.json.JSON.JSONObject;
import org.eclipse.n4js.json.JSON.JSONPackage;
import org.eclipse.n4js.json.JSON.JSONStringLiteral;
import org.eclipse.n4js.json.JSON.JSONValue;
import org.eclipse.n4js.json.JSON.NameValuePair;
import org.eclipse.n4js.json.model.utils.JSONModelUtils;
import org.eclipse.n4js.packagejson.PackageJsonProperties;
import org.eclipse.n4js.packagejson.PackageJsonUtils;
import org.eclipse.n4js.projectDescription.ModuleFilterSpecifier;
import org.eclipse.n4js.projectDescription.ProjectDependency;
import org.eclipse.n4js.projectDescription.ProjectDescription;
import org.eclipse.n4js.projectDescription.ProjectReference;
import org.eclipse.n4js.projectDescription.ProjectType;
import org.eclipse.n4js.projectDescription.SourceContainerDescription;
import org.eclipse.n4js.projectDescription.SourceContainerType;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSProject;
import org.eclipse.n4js.projectModel.IN4JSSourceContainer;
import org.eclipse.n4js.resource.N4JSResourceDescriptionStrategy;
import org.eclipse.n4js.resource.XpectAwareFileExtensionCalculator;
import org.eclipse.n4js.semver.Semver.NPMVersionRequirement;
import org.eclipse.n4js.semver.Semver.VersionNumber;
import org.eclipse.n4js.semver.SemverHelper;
import org.eclipse.n4js.semver.SemverMatcher;
import org.eclipse.n4js.semver.model.SemverSerializer;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.utils.DependencyCycle;
import org.eclipse.n4js.utils.DependencyTraverser;
import org.eclipse.n4js.utils.ProjectDescriptionLoader;
import org.eclipse.n4js.utils.WildcardPathFilterHelper;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.n4js.validation.helper.SourceContainerAwareDependencyProvider;
import org.eclipse.n4js.validation.validators.packagejson.ASTTraceable;
import org.eclipse.n4js.validation.validators.packagejson.AbstractJSONValidatorExtension;
import org.eclipse.n4js.validation.validators.packagejson.CheckProperty;
import org.eclipse.n4js.validation.validators.packagejson.PolyFilledProvision;
import org.eclipse.n4js.validation.validators.packagejson.ProjectTypePredicate;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.resource.IContainer;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.resource.impl.ResourceDescriptionsProvider;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * A JSON validator extension that validates {@code package.json} resources in the context
 * of higher-level concepts such as project references, the general project setup and feature restrictions.
 * 
 * Generally, this validator includes constraints that are implemented based on the converted {@link ProjectDescription}
 * as it can be obtained from the {@link ProjectDescriptionLoader}. This especially includes non-local validation
 * such as the resolution of referenced projects.
 * 
 * For lower-level, structural and local validations with regard to {@code package.json}
 * files , see {@link PackageJsonValidatorExtension}.
 */
@Singleton
@SuppressWarnings("all")
public class N4JSProjectSetupJsonValidatorExtension extends AbstractJSONValidatorExtension {
  public static class HasTestDependencyVisitor implements DependencyTraverser.DependencyVisitor<IN4JSProject> {
    private boolean hasTestDependencies = false;
    
    @Override
    public void accept(final IN4JSProject project) {
      boolean _hasTestDependency = this.hasTestDependency(project);
      if (_hasTestDependency) {
        this.hasTestDependencies = true;
      }
    }
    
    private boolean hasTestDependency(final IN4JSProject p) {
      ImmutableList<? extends IN4JSProject> _dependencies = p.getDependencies();
      for (final IN4JSProject pDep : _dependencies) {
        if (((N4JSGlobals.VENDOR_ID.equals(pDep.getVendorID()) && N4JSGlobals.MANGELHAFT.equals(pDep.getProjectName())) || N4JSGlobals.MANGELHAFT_ASSERT.equals(pDep.getProjectName()))) {
          return true;
        }
      }
      return false;
    }
  }
  
  private static class ModuleSpecifierFileVisitor extends SimpleFileVisitor<Path> {
    private final N4JSProjectSetupJsonValidatorExtension setupValidator;
    
    private final IN4JSProject project;
    
    private final Map<ASTTraceable<ModuleFilterSpecifier>, Boolean> filterSpecifiers;
    
    public ModuleSpecifierFileVisitor(final N4JSProjectSetupJsonValidatorExtension validatorExtension, final IN4JSProject project, final Map<ASTTraceable<ModuleFilterSpecifier>, Boolean> filterSpecifiers) {
      this.setupValidator = validatorExtension;
      this.project = project;
      this.filterSpecifiers = filterSpecifiers;
    }
    
    @Override
    public FileVisitResult visitFile(final Path path, final BasicFileAttributes attrs) throws IOException {
      for (final Iterator<Map.Entry<ASTTraceable<ModuleFilterSpecifier>, Boolean>> iter = this.filterSpecifiers.entrySet().iterator(); iter.hasNext();) {
        {
          final Map.Entry<ASTTraceable<ModuleFilterSpecifier>, Boolean> entry = iter.next();
          final ASTTraceable<ModuleFilterSpecifier> filterSpecifierTraceable = entry.getKey();
          ModuleFilterSpecifier _element = filterSpecifierTraceable.element;
          String _moduleSpecifierWithWildcard = null;
          if (_element!=null) {
            _moduleSpecifierWithWildcard=_element.getModuleSpecifierWithWildcard();
          }
          final String specifier = _moduleSpecifierWithWildcard;
          final boolean checkForMatches = ((((specifier != null) && 
            this.isN4JSFile(specifier)) && path.toFile().isFile()) || (!this.isN4JSFile(specifier)));
          final URI location = this.getFileInSources(this.project, filterSpecifierTraceable.element, path);
          if ((checkForMatches && (location != null))) {
            final boolean matchesFile = this.setupValidator.wildcardHelper.isPathContainedByFilter(location, filterSpecifierTraceable.element);
            final boolean matchesN4JSFile = (matchesFile && this.isN4JSFile(path.toString()));
            if (matchesFile) {
              entry.setValue(Boolean.valueOf(true));
            }
            if (matchesN4JSFile) {
              this.setupValidator.addNoValidationForN4JSFilesIssue(filterSpecifierTraceable);
            }
            if ((matchesFile && matchesN4JSFile)) {
              iter.remove();
            }
          }
        }
      }
      boolean _isEmpty = this.filterSpecifiers.isEmpty();
      if (_isEmpty) {
        return FileVisitResult.TERMINATE;
      } else {
        return FileVisitResult.CONTINUE;
      }
    }
    
    private URI getFileInSources(final IN4JSProject project, final ModuleFilterSpecifier filterSpecifier, final Path filePath) {
      final Path lPath = project.getLocationPath();
      final Path filePathString = lPath.relativize(filePath);
      final URI projectRelativeURI = URI.createURI(filePathString.toString());
      return project.getLocation().appendSegments(projectRelativeURI.segments());
    }
    
    private boolean isN4JSFile(final String fileSpecifier) {
      return ((fileSpecifier.endsWith(("." + N4JSGlobals.N4JS_FILE_EXTENSION)) || 
        fileSpecifier.endsWith(("." + N4JSGlobals.N4JSX_FILE_EXTENSION))) || 
        fileSpecifier.endsWith(("." + N4JSGlobals.N4JSD_FILE_EXTENSION)));
    }
  }
  
  /**
   * Intermediate validation-only representation of a project reference.
   * 
   * This may or may not include a {@link #versionConstraint}.
   * 
   * Holds a trace link {@link #astRepresentation} to its original AST element, so that
   * check methods can add issues to the actual elements.
   */
  @Data
  private static class ValidationProjectReference {
    private final String referencedProjectName;
    
    private final NPMVersionRequirement npmVersion;
    
    private final EObject astRepresentation;
    
    public ValidationProjectReference(final String referencedProjectName, final NPMVersionRequirement npmVersion, final EObject astRepresentation) {
      super();
      this.referencedProjectName = referencedProjectName;
      this.npmVersion = npmVersion;
      this.astRepresentation = astRepresentation;
    }
    
    @Override
    @Pure
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((this.referencedProjectName== null) ? 0 : this.referencedProjectName.hashCode());
      result = prime * result + ((this.npmVersion== null) ? 0 : this.npmVersion.hashCode());
      return prime * result + ((this.astRepresentation== null) ? 0 : this.astRepresentation.hashCode());
    }
    
    @Override
    @Pure
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference other = (N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference) obj;
      if (this.referencedProjectName == null) {
        if (other.referencedProjectName != null)
          return false;
      } else if (!this.referencedProjectName.equals(other.referencedProjectName))
        return false;
      if (this.npmVersion == null) {
        if (other.npmVersion != null)
          return false;
      } else if (!this.npmVersion.equals(other.npmVersion))
        return false;
      if (this.astRepresentation == null) {
        if (other.astRepresentation != null)
          return false;
      } else if (!this.astRepresentation.equals(other.astRepresentation))
        return false;
      return true;
    }
    
    @Override
    @Pure
    public String toString() {
      ToStringBuilder b = new ToStringBuilder(this);
      b.add("referencedProjectName", this.referencedProjectName);
      b.add("npmVersion", this.npmVersion);
      b.add("astRepresentation", this.astRepresentation);
      return b.toString();
    }
    
    @Pure
    public String getReferencedProjectName() {
      return this.referencedProjectName;
    }
    
    @Pure
    public NPMVersionRequirement getNpmVersion() {
      return this.npmVersion;
    }
    
    @Pure
    public EObject getAstRepresentation() {
      return this.astRepresentation;
    }
  }
  
  private final static Logger LOGGER = Logger.getLogger(N4JSProjectSetupJsonValidatorExtension.class);
  
  private final static ProjectTypePredicate API_TYPE = ProjectTypePredicate.anyOf(ProjectType.API);
  
  private final static ProjectTypePredicate RE_TYPE = ProjectTypePredicate.anyOf(ProjectType.RUNTIME_ENVIRONMENT);
  
  private final static ProjectTypePredicate RL_TYPE = ProjectTypePredicate.anyOf(ProjectType.RUNTIME_LIBRARY);
  
  private final static ProjectTypePredicate TEST_TYPE = ProjectTypePredicate.anyOf(ProjectType.TEST);
  
  private final static ProjectTypePredicate RE_OR_RL_TYPE = ProjectTypePredicate.anyOf(ProjectType.RUNTIME_ENVIRONMENT, ProjectType.RUNTIME_LIBRARY);
  
  private final static ProjectTypePredicate PLAINJS_TYPE = ProjectTypePredicate.anyOf(ProjectType.PLAINJS);
  
  /**
   * Key to store a converted ProjectDescription instance in the validation context for re-use across different check-methods
   * @See {@link #getProjectDescription()}
   */
  private final static String PROJECT_DESCRIPTION_CACHE = "PROJECT_DESCRIPTION_CACHE";
  
  /**
   * Key to store a map of all available projects in the validation context for re-use across different check-methods.
   * @See {@link #getAllExistingProjectNames()}
   */
  private static String ALL_EXISTING_PROJECT_CACHE = "ALL_EXISTING_PROJECT_CACHE";
  
  /**
   * Key to store a map of all declared project dependencies in the validation context for re-use across different check-methods.
   * @See {@link #getDeclaredProjectDependencies()}
   */
  private static String DECLARED_DEPENDENCIES_CACHE = "DECLARED_DEPENDENCIES_CACHE";
  
  @Inject
  @Extension
  private IN4JSCore _iN4JSCore;
  
  @Inject
  private IContainer.Manager containerManager;
  
  @Inject
  private ResourceDescriptionsProvider resourceDescriptionsProvider;
  
  @Inject
  private XpectAwareFileExtensionCalculator fileExtensionCalculator;
  
  @Inject
  private ProjectDescriptionLoader projectDescriptionLoader;
  
  @Inject
  private WildcardPathFilterHelper wildcardHelper;
  
  @Inject
  protected N4JSElementKeywordProvider keywordProvider;
  
  @Inject
  protected ShadowingInfoHelper shadowingInfoHelper;
  
  @Inject
  protected ExternalIndexSynchronizer indexSynchronizer;
  
  @Inject
  protected ExternalLibraryWorkspace extWS;
  
  @Inject
  protected SemverHelper semverHelper;
  
  @Override
  public boolean isResponsible(final Map<Object, Object> context, final EObject eObject) {
    return this.fileExtensionCalculator.getFilenameWithoutXpectExtension(eObject.eResource().getURI()).equals(IN4JSProject.PACKAGE_JSON);
  }
  
  /**
   * According to IDESpec §§12.04 Polyfills at most one Polyfill can be provided for a class.
   * Here the consistency according to the given combination of runtime-environment and runtime-libraries of the
   * project definition will be checked.
   */
  @Check
  public void checkConsistentPolyfills(final JSONDocument document) {
    final Map<String, JSONStringLiteral> mQName2rtDep = CollectionLiterals.<String, JSONStringLiteral>newHashMap();
    final ProjectDescription description = this.getProjectDescription();
    final String projectName = description.getProjectName();
    if ((projectName == null)) {
      return;
    }
    final JSONArray requiredRuntimeLibrariesValue = this.<JSONArray>getSingleDocumentValue(PackageJsonProperties.REQUIRED_RUNTIME_LIBRARIES, JSONArray.class);
    if ((requiredRuntimeLibrariesValue == null)) {
      return;
    }
    Iterable<? extends JSONStringLiteral> rteAndRtl = Iterables.<JSONStringLiteral>filter(requiredRuntimeLibrariesValue.getElements(), 
      JSONStringLiteral.class);
    final JSONStringLiteral selfProject = JSONModelUtils.createStringLiteral(projectName);
    final Optional<? extends IN4JSProject> optOwnProject = this._iN4JSCore.findProject(document.eResource().getURI());
    boolean _isPresent = optOwnProject.isPresent();
    if (_isPresent) {
      rteAndRtl = Iterables.<JSONStringLiteral>concat(rteAndRtl, Collections.<JSONStringLiteral>unmodifiableSet(CollectionLiterals.<JSONStringLiteral>newHashSet(selfProject)));
    }
    for (final JSONStringLiteral libraryLiteral : rteAndRtl) {
      if ((null != libraryLiteral)) {
        final String libPPqname = libraryLiteral.getValue();
        mQName2rtDep.put(libPPqname, libraryLiteral);
      }
    }
    final List<IEObjectDescription> allPolyFillTypes = this.getAllNonStaticPolyfills(document.eResource());
    final LinkedListMultimap<String, PolyFilledProvision> exportedPolyfills_QN_to_PolyProvision = LinkedListMultimap.<String, PolyFilledProvision>create();
    for (final IEObjectDescription ieoT : allPolyFillTypes) {
      {
        final Optional<? extends IN4JSSourceContainer> optSrcContainer = this._iN4JSCore.findN4JSSourceContainer(ieoT.getEObjectURI());
        boolean _isPresent_1 = optSrcContainer.isPresent();
        if (_isPresent_1) {
          final IN4JSSourceContainer srcCont = optSrcContainer.get();
          final String depQName = srcCont.getProject().getProjectName();
          final JSONStringLiteral dependency = mQName2rtDep.get(depQName);
          if ((dependency == null)) {
          } else {
            if ((dependency != selfProject)) {
              String _string = ieoT.getQualifiedName().toString();
              PolyFilledProvision _polyFilledProvision = new PolyFilledProvision(depQName, dependency, ieoT);
              exportedPolyfills_QN_to_PolyProvision.put(_string, _polyFilledProvision);
            }
          }
        } else {
          QualifiedName _qualifiedName = ieoT.getQualifiedName();
          String _plus = ("No container library found for " + _qualifiedName);
          throw new IllegalStateException(_plus);
        }
      }
    }
    final Multimap<Set<JSONStringLiteral>, String> markerMapLibs2FilledName = LinkedListMultimap.<Set<JSONStringLiteral>, String>create();
    Set<String> _keySet = exportedPolyfills_QN_to_PolyProvision.keySet();
    for (final String polyExport_QN : _keySet) {
      {
        final List<PolyFilledProvision> polyProvisions = exportedPolyfills_QN_to_PolyProvision.get(polyExport_QN);
        int _size = polyProvisions.size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          final LinkedListMultimap<String, PolyFilledProvision> m = LinkedListMultimap.<String, PolyFilledProvision>create();
          for (final PolyFilledProvision prov : polyProvisions) {
            {
              EObject eoPolyFiller = prov.ieoDescrOfPolyfill.getEObjectOrProxy();
              if ((eoPolyFiller instanceof TClassifier)) {
                EObject _resolve = EcoreUtil.resolve(eoPolyFiller, document.eResource());
                final TClassifier resolvedEoPolyFiller = ((TClassifier) _resolve);
                boolean _isPolyfill = resolvedEoPolyFiller.isPolyfill();
                boolean _not = (!_isPolyfill);
                if (_not) {
                  String _name = resolvedEoPolyFiller.getName();
                  String _plus = ("Expected a polyfill, but wasn\'t: " + _name);
                  throw new IllegalStateException(_plus);
                } else {
                  EList<TMember> _ownedMembers = resolvedEoPolyFiller.getOwnedMembers();
                  for (final TMember member : _ownedMembers) {
                    m.put(member.getName(), prov);
                  }
                }
              }
            }
          }
          Set<String> _keySet_1 = m.keySet();
          for (final String filledInMemberName : _keySet_1) {
            {
              final List<PolyFilledProvision> providers = m.get(filledInMemberName);
              int _size_1 = providers.size();
              boolean _greaterThan_1 = (_size_1 > 1);
              if (_greaterThan_1) {
                final HashSet<JSONStringLiteral> keySet = CollectionLiterals.<JSONStringLiteral>newHashSet();
                final Consumer<PolyFilledProvision> _function = (PolyFilledProvision it) -> {
                  keySet.add(it.libraryProjectReferenceLiteral);
                };
                providers.forEach(_function);
                final String filledTypeFQN = IterableExtensions.<PolyFilledProvision>head(providers).descriptionStandard;
                final String message = ((filledTypeFQN + "#") + filledInMemberName);
                markerMapLibs2FilledName.put(keySet, message);
              }
            }
          }
        }
      }
    }
    Set<Set<JSONStringLiteral>> _keySet_1 = markerMapLibs2FilledName.keySet();
    for (final Set<JSONStringLiteral> keyS : _keySet_1) {
      {
        final Collection<String> polyFilledMemberAsStrings = markerMapLibs2FilledName.get(keyS);
        final Function1<JSONStringLiteral, String> _function = (JSONStringLiteral it) -> {
          return it.getValue();
        };
        final String libsString = IterableExtensions.join(IterableExtensions.<String>sort(ListExtensions.<JSONStringLiteral, String>map(IterableExtensions.<JSONStringLiteral>toList(keyS), _function)), ", ");
        final Function1<String, String> _function_1 = (String it) -> {
          return (("\"" + it) + "\"");
        };
        final String userPresentablePolyFills = IterableExtensions.join(IterableExtensions.<String>sort(ListExtensions.<String, String>map(IterableExtensions.<String>toList(polyFilledMemberAsStrings), _function_1)), ", ");
        int _size = keyS.size();
        boolean _greaterThan = (_size > 1);
        if (_greaterThan) {
          String _xifexpression = null;
          int _size_1 = polyFilledMemberAsStrings.size();
          boolean _equals = (_size_1 == 1);
          if (_equals) {
            _xifexpression = IssueCodes.getMessageForPOLY_CLASH_IN_RUNTIMEDEPENDENCY(libsString, userPresentablePolyFills);
          } else {
            _xifexpression = IssueCodes.getMessageForPOLY_CLASH_IN_RUNTIMEDEPENDENCY_MULTI(libsString, userPresentablePolyFills);
          }
          final String issMsg = _xifexpression;
          final Consumer<JSONStringLiteral> _function_2 = (JSONStringLiteral it) -> {
            this.addIssue(issMsg, it, IssueCodes.POLY_CLASH_IN_RUNTIMEDEPENDENCY);
          };
          keyS.forEach(_function_2);
        } else {
          final String issMsg_1 = IssueCodes.getMessageForPOLY_ERROR_IN_RUNTIMEDEPENDENCY(libsString, userPresentablePolyFills);
          this.addIssue(issMsg_1, IterableExtensions.<JSONStringLiteral>head(keyS), IssueCodes.POLY_ERROR_IN_RUNTIMEDEPENDENCY);
        }
      }
    }
  }
  
  /**
   * Get a list of all polyfill types & IEObjecdescriptions accessible in the whole Project.
   * @param manifestResourceUsedAsContext just a resource to build the scope
   * @return List of Type->IEObjectdescription
   */
  private List<IEObjectDescription> getAllNonStaticPolyfills(final Resource projectDescriptionResourceUsedAsContext) {
    final XtextResource asXtextRes = ((XtextResource) projectDescriptionResourceUsedAsContext);
    final IResourceDescription resDescr = asXtextRes.getResourceServiceProvider().getResourceDescriptionManager().getResourceDescription(asXtextRes);
    final List<IContainer> visibleContainers = this.containerManager.getVisibleContainers(resDescr, 
      this.resourceDescriptionsProvider.getResourceDescriptions(projectDescriptionResourceUsedAsContext));
    final ArrayList<IEObjectDescription> types = CollectionLiterals.<IEObjectDescription>newArrayList();
    final Function1<IContainer, Iterable<IEObjectDescription>> _function = (IContainer it) -> {
      return it.getExportedObjectsByType(TypesPackage.Literals.TYPE);
    };
    Iterable<IEObjectDescription> _flatten = Iterables.<IEObjectDescription>concat(ListExtensions.<IContainer, Iterable<IEObjectDescription>>map(visibleContainers, _function));
    for (final IEObjectDescription descr : _flatten) {
      {
        final boolean isPolyFill = N4JSResourceDescriptionStrategy.getPolyfill(descr);
        final boolean isStaticPolyFill = N4JSResourceDescriptionStrategy.getStaticPolyfill(descr);
        if ((isPolyFill && (!isStaticPolyFill))) {
          types.add(descr);
        }
      }
    }
    return types;
  }
  
  /**
   * IDEBUG-266 issue error warning on cyclic dependencies.
   */
  @Check
  public void checkCyclicDependencies(final JSONDocument document) {
    final IN4JSProject project = this._iN4JSCore.findProject(document.eResource().getURI()).orNull();
    if ((null != project)) {
      final SourceContainerAwareDependencyProvider dependencyProvider = new SourceContainerAwareDependencyProvider(true);
      final DependencyTraverser<IN4JSProject> traverser = new DependencyTraverser<IN4JSProject>(project, dependencyProvider, true);
      final DependencyCycle<IN4JSProject> traversalResult = traverser.findCycle();
      boolean _hasCycle = traversalResult.hasCycle();
      if (_hasCycle) {
        final JSONValue nameValue = this.getSingleDocumentValue(PackageJsonProperties.NAME);
        final Function<IN4JSProject, String> _function = (IN4JSProject it) -> {
          return this.calculateName(it);
        };
        final String message = IssueCodes.getMessageForPROJECT_DEPENDENCY_CYCLE(traversalResult.prettyPrint(_function));
        this.addIssuePreferred(Collections.<EObject>unmodifiableList(CollectionLiterals.<EObject>newArrayList(nameValue)), message, IssueCodes.PROJECT_DEPENDENCY_CYCLE);
      } else {
        this.holdsProjectWithTestFragmentDependsOnTestLibrary(project);
      }
    }
  }
  
  /**
   * Checks if a project containing {@link SourceContainerType#TEST} depends
   * (directly or transitively) on a {@link ProjectType#RUNTIME_LIBRARY} test runtime library.
   */
  private void holdsProjectWithTestFragmentDependsOnTestLibrary(final IN4JSProject project) {
    final JSONValue sourcesSection = this.<JSONValue>getSingleDocumentValue(PackageJsonProperties.SOURCES, JSONValue.class);
    final List<SourceContainerDescription> sourceContainers = PackageJsonUtils.asSourceContainerDescriptionsOrEmpty(sourcesSection);
    if ((sourceContainers == null)) {
      return;
    }
    final Function1<SourceContainerDescription, Boolean> _function = (SourceContainerDescription sf) -> {
      return Boolean.valueOf(SourceContainerType.TEST.equals(sf.getSourceContainerType()));
    };
    SourceContainerDescription _findFirst = IterableExtensions.<SourceContainerDescription>findFirst(sourceContainers, _function);
    final boolean hasTestFragment = (_findFirst != null);
    if ((!hasTestFragment)) {
      return;
    }
    boolean _anyDependsOnTestLibrary = this.anyDependsOnTestLibrary(Collections.<IN4JSProject>unmodifiableList(CollectionLiterals.<IN4JSProject>newArrayList(project)));
    boolean _not = (!_anyDependsOnTestLibrary);
    if (_not) {
      this.addIssuePreferred(Collections.<EObject>unmodifiableList(CollectionLiterals.<EObject>newArrayList()), IssueCodes.getMessageForSRCTEST_NO_TESTLIB_DEP(N4JSGlobals.MANGELHAFT), IssueCodes.SRCTEST_NO_TESTLIB_DEP);
    }
  }
  
  /**
   * check if any project in the list has dependency on test library, if so return true.
   * Otherwise invoke recursively in dependencies list of each project in initial list.
   * 
   * @returns true if any of the projects in the provided list depends (transitively) on the test library.
   */
  private boolean anyDependsOnTestLibrary(final List<? extends IN4JSProject> projects) {
    final SourceContainerAwareDependencyProvider dependencyProvider = new SourceContainerAwareDependencyProvider(true);
    final N4JSProjectSetupJsonValidatorExtension.HasTestDependencyVisitor hasTestDependencyVisitor = new N4JSProjectSetupJsonValidatorExtension.HasTestDependencyVisitor();
    for (final IN4JSProject project : projects) {
      {
        final DependencyTraverser<IN4JSProject> dependencyTraverser = new DependencyTraverser<IN4JSProject>(project, hasTestDependencyVisitor, dependencyProvider, true);
        dependencyTraverser.traverse();
      }
    }
    return hasTestDependencyVisitor.hasTestDependencies;
  }
  
  private String calculateName(final IN4JSProject it) {
    return it.getProjectName();
  }
  
  /**
   * Checks whether a test project either tests APIs or libraries. Raises a validation issue, if the test project
   * tests both APIs and libraries. Does nothing if the project description of the validated project is NOT a test
   * project.
   */
  @CheckProperty(property = PackageJsonProperties.TESTED_PROJECTS)
  public void checkTestedProjectsType(final JSONValue testedProjectsValue) {
    final ProjectDescription description = this.getProjectDescription();
    ProjectType _projectType = description.getProjectType();
    boolean _equals = Objects.equal(ProjectType.TEST, _projectType);
    if (_equals) {
      final EList<ProjectReference> projects = description.getTestedProjects();
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(projects);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        final Map<String, IN4JSProject> allProjects = this.getAllProjectsByName();
        final ProjectReference head = IterableExtensions.<ProjectReference>head(projects);
        IN4JSProject _get = allProjects.get(head.getProjectName());
        ProjectType _projectType_1 = null;
        if (_get!=null) {
          _projectType_1=_get.getProjectType();
        }
        final ProjectType refProjectType = _projectType_1;
        final Function1<ProjectReference, Boolean> _function = (ProjectReference testedProject) -> {
          boolean _and = false;
          boolean _containsKey = allProjects.containsKey(testedProject.getProjectName());
          if (!_containsKey) {
            _and = false;
          } else {
            IN4JSProject _get_1 = allProjects.get(testedProject.getProjectName());
            ProjectType _projectType_2 = null;
            if (_get_1!=null) {
              _projectType_2=_get_1.getProjectType();
            }
            boolean _notEquals = (!Objects.equal(refProjectType, _projectType_2));
            _and = _notEquals;
          }
          return Boolean.valueOf(_and);
        };
        boolean _exists = IterableExtensions.<ProjectReference>exists(projects, _function);
        if (_exists) {
          this.addIssue(
            IssueCodes.getMessageForMISMATCHING_TESTED_PROJECT_TYPES(), testedProjectsValue, 
            IssueCodes.MISMATCHING_TESTED_PROJECT_TYPES);
        }
      }
    }
  }
  
  /**
   * Checks whether a library project, that belongs to a specific implementation (has defined implementation ID) does not
   * depend on any other libraries that belong to any other implementation. In such cases, raises validation issue.
   */
  @CheckProperty(property = PackageJsonProperties.DEPENDENCIES)
  public void checkHasConsistentImplementationIdChain(final JSONValue dependenciesValue) {
    if ((!(dependenciesValue instanceof JSONObject))) {
      return;
    }
    final EList<NameValuePair> dependencyPairs = ((JSONObject) dependenciesValue).getNameValuePairs();
    final ProjectDescription description = this.getProjectDescription();
    if ((Objects.equal(ProjectType.LIBRARY, description.getProjectType()) && (!StringExtensions.isNullOrEmpty(description.getImplementationId())))) {
      final String expectedImplementationId = description.getImplementationId();
      final Map<String, IN4JSProject> allProjects = this.getAllProjectsByName();
      final Consumer<NameValuePair> _function = (NameValuePair pair) -> {
        final String dependencyProjectName = pair.getName();
        IN4JSProject _get = allProjects.get(dependencyProjectName);
        Optional<String> _implementationId = null;
        if (_get!=null) {
          _implementationId=_get.getImplementationId();
        }
        String _orNull = null;
        if (_implementationId!=null) {
          _orNull=_implementationId.orNull();
        }
        final String actualImplementationId = _orNull;
        if (((!StringExtensions.isNullOrEmpty(actualImplementationId)) && (!Objects.equal(actualImplementationId, expectedImplementationId)))) {
          final String message = IssueCodes.getMessageForMISMATCHING_IMPLEMENTATION_ID(expectedImplementationId, dependencyProjectName, actualImplementationId);
          this.addIssue(message, pair, IssueCodes.MISMATCHING_IMPLEMENTATION_ID);
        }
      };
      IterableExtensions.<NameValuePair>filterNull(dependencyPairs).forEach(_function);
    }
  }
  
  /**
   * Checks if any transitive external dependency of a workspace project references to a workspace
   * project. If so, raises a validation warning.
   */
  @CheckProperty(property = PackageJsonProperties.DEPENDENCIES)
  public void checkExternalProjectDoesNotReferenceWorkspaceProject(final JSONValue dependenciesValue) {
    final Map<String, IN4JSProject> allProjects = this.getAllProjectsByName();
    final ProjectDescription description = this.getProjectDescription();
    String _projectName = description.getProjectName();
    boolean _tripleEquals = (_projectName == null);
    if (_tripleEquals) {
      return;
    }
    final IN4JSProject currentProject = allProjects.get(description.getProjectName());
    if ((((null == currentProject) || (!currentProject.exists())) || currentProject.isExternal())) {
      return;
    }
    final HashSet<String> visitedProjectNames = CollectionLiterals.<String>newHashSet();
    final Stack<IN4JSProject> stack = new Stack<IN4JSProject>();
    final Function1<IN4JSProject, Boolean> _function = (IN4JSProject it) -> {
      return Boolean.valueOf(it.isExternal());
    };
    Iterables.<IN4JSProject>addAll(stack, IterableExtensions.<IN4JSProject>filter(Iterables.<IN4JSProject>filter(currentProject.getAllDirectDependencies(), IN4JSProject.class), _function));
    while ((!stack.isEmpty())) {
      {
        final IN4JSProject actual = stack.pop();
        final String actualId = actual.getProjectName();
        boolean _isExternal = actual.isExternal();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Implementation error. Only external projects are expected: ");
        _builder.append(actual);
        _builder.append(".");
        Preconditions.checkState(_isExternal, _builder);
        boolean _add = visitedProjectNames.add(actualId);
        boolean _not = (!_add);
        if (_not) {
          return;
        }
        final Iterable<IN4JSProject> actualDirectDependencies = Iterables.<IN4JSProject>filter(actual.getAllDirectDependencies(), IN4JSProject.class);
        final Function1<IN4JSProject, Boolean> _function_1 = (IN4JSProject it) -> {
          boolean _isExternal_1 = it.isExternal();
          return Boolean.valueOf((!_isExternal_1));
        };
        final IN4JSProject workspaceDependency = IterableExtensions.<IN4JSProject>findFirst(actualDirectDependencies, _function_1);
        if ((null != workspaceDependency)) {
          final String workspaceDependencyId = workspaceDependency.getProjectName();
          final String message = IssueCodes.getMessageForEXTERNAL_PROJECT_REFERENCES_WORKSPACE_PROJECT(actualId, workspaceDependencyId);
          this.addIssue(message, dependenciesValue, 
            IssueCodes.EXTERNAL_PROJECT_REFERENCES_WORKSPACE_PROJECT);
          return;
        }
        final Function1<IN4JSProject, Boolean> _function_2 = (IN4JSProject it) -> {
          return Boolean.valueOf(it.isExternal());
        };
        Iterables.<IN4JSProject>addAll(stack, IterableExtensions.<IN4JSProject>filter(actualDirectDependencies, _function_2));
      }
    }
  }
  
  @Check
  public void checkDependenciesAndDevDependencies(final JSONDocument document) {
    final Optional<? extends IN4JSProject> project = this._iN4JSCore.findProject(document.eResource().getURI());
    final boolean isExternal = (project.isPresent() && project.get().isExternal());
    final Iterable<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references = this.getDependencies((!isExternal));
    boolean _isEmpty = IterableExtensions.isEmpty(references);
    boolean _not = (!_isEmpty);
    if (_not) {
      this.checkReferencedProjects(references, this.createDependenciesPredicate(), "dependencies or devDependencies", false, false);
    }
    ProjectType _projectType = this.getProjectDescription().getProjectType();
    boolean _equals = Objects.equal(_projectType, ProjectType.API);
    if (_equals) {
      this.internalValidateAPIProjectReferences(references);
    }
  }
  
  /**
   * Returns a representation of all declared runtime dependencies of
   * the currently validate document (cf. {@link #getDocument()}).
   * 
   * @param includeDevDependencies
   * 					Specifies whether the returned iterable should also include devDependencies.
   */
  private Iterable<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> getDependencies(final boolean includeDevDependencies) {
    final ArrayList<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references = CollectionLiterals.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>newArrayList();
    Collection<JSONValue> _documentValues = this.getDocumentValues(PackageJsonProperties.DEPENDENCIES);
    for (final JSONValue value : _documentValues) {
      List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> _referencesFromDependenciesObject = this.getReferencesFromDependenciesObject(value);
      Iterables.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>addAll(references, _referencesFromDependenciesObject);
    }
    if (includeDevDependencies) {
      Collection<JSONValue> _documentValues_1 = this.getDocumentValues(PackageJsonProperties.DEV_DEPENDENCIES);
      for (final JSONValue value_1 : _documentValues_1) {
        List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> _referencesFromDependenciesObject_1 = this.getReferencesFromDependenciesObject(value_1);
        Iterables.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>addAll(references, _referencesFromDependenciesObject_1);
      }
    }
    return references;
  }
  
  /**
   * Checks that an API project does not declare any dependencies on implementation
   * projects (library projects with implementation ID).
   */
  public void internalValidateAPIProjectReferences(final Iterable<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references) {
    final Function1<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, Pair<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject>> _function = (N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference ref) -> {
      return Pair.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject>of(ref, this.getAllProjectsByName().get(ref.referencedProjectName));
    };
    final Function1<Pair<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject>, Boolean> _function_1 = (Pair<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject> pair) -> {
      return Boolean.valueOf((Objects.equal(pair.getValue().getProjectType(), ProjectType.LIBRARY) && pair.getValue().getImplementationId().isPresent()));
    };
    final Iterable<Pair<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject>> libraryDependenciesWithImplId = IterableExtensions.<Pair<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject>>filter(IterableExtensions.<Pair<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject>>filterNull(IterableExtensions.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, Pair<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject>>map(references, _function)), _function_1);
    for (final Pair<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, IN4JSProject> projectPair : libraryDependenciesWithImplId) {
      {
        final N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference reference = projectPair.getKey();
        this.addIssue(IssueCodes.getMessageForINVALID_API_PROJECT_DEPENDENCY(reference.referencedProjectName), reference.astRepresentation, 
          IssueCodes.INVALID_API_PROJECT_DEPENDENCY);
      }
    }
  }
  
  /**
   * Checks the 'n4js.extendedRuntimeEnvironment' section.
   */
  @CheckProperty(property = PackageJsonProperties.EXTENDED_RUNTIME_ENVIRONMENT)
  public void checkExtendedRuntimeEnvironment(final JSONValue extendedRuntimeEnvironmentValue) {
    boolean _checkFeatureRestrictions = this.checkFeatureRestrictions("extended runtime environment", extendedRuntimeEnvironmentValue, N4JSProjectSetupJsonValidatorExtension.RE_TYPE);
    boolean _not = (!_checkFeatureRestrictions);
    if (_not) {
      return;
    }
    final List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references = this.getReferencesFromJSONStringLiteral(extendedRuntimeEnvironmentValue);
    this.checkReferencedProjects(references, this.forN4jsProjects(N4JSProjectSetupJsonValidatorExtension.RE_TYPE), "extended runtime environment", false, false);
  }
  
  /**
   * Checks the 'n4js.requiredRuntimeLibraries' section.
   */
  @CheckProperty(property = PackageJsonProperties.REQUIRED_RUNTIME_LIBRARIES)
  public void checkRequiredRuntimeLibraries(final JSONValue requiredRuntimeLibrariesValue) {
    boolean _checkFeatureRestrictions = this.checkFeatureRestrictions("required runtime libraries", requiredRuntimeLibrariesValue, ProjectTypePredicate.not(N4JSProjectSetupJsonValidatorExtension.RE_TYPE));
    boolean _not = (!_checkFeatureRestrictions);
    if (_not) {
      return;
    }
    final List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references = this.getReferencesFromJSONStringArray(requiredRuntimeLibrariesValue);
    this.checkReferencedProjects(references, this.forN4jsProjects(N4JSProjectSetupJsonValidatorExtension.RL_TYPE), "required runtime libraries", true, false);
  }
  
  /**
   * Checks the 'n4js.providedRuntimeLibraries' section.
   */
  @CheckProperty(property = PackageJsonProperties.PROVIDED_RUNTIME_LIBRARIES)
  public void checkProvidedRuntimeLibraries(final JSONValue providedRuntimeLibraries) {
    boolean _checkFeatureRestrictions = this.checkFeatureRestrictions("provided runtime libraries", providedRuntimeLibraries, N4JSProjectSetupJsonValidatorExtension.RE_TYPE);
    boolean _not = (!_checkFeatureRestrictions);
    if (_not) {
      return;
    }
    final List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references = this.getReferencesFromJSONStringArray(providedRuntimeLibraries);
    this.checkReferencedProjects(references, this.forN4jsProjects(N4JSProjectSetupJsonValidatorExtension.RL_TYPE), "provided runtime libraries", false, false);
  }
  
  /**
   * Checks the 'n4js.testedProjects' section.
   */
  @CheckProperty(property = PackageJsonProperties.TESTED_PROJECTS)
  public void checkTestedProjects(final JSONValue testedProjectsValue) {
    boolean _checkFeatureRestrictions = this.checkFeatureRestrictions("tested projects", testedProjectsValue, N4JSProjectSetupJsonValidatorExtension.TEST_TYPE);
    boolean _not = (!_checkFeatureRestrictions);
    if (_not) {
      return;
    }
    final List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references = this.getReferencesFromJSONStringArray(testedProjectsValue);
    this.checkReferencedProjects(references, this.forN4jsProjects(ProjectTypePredicate.not(N4JSProjectSetupJsonValidatorExtension.TEST_TYPE)), "tested projects", true, false);
  }
  
  /**
   * Checks the 'n4js.initModules' section.
   */
  @CheckProperty(property = PackageJsonProperties.INIT_MODULES)
  public void checkInitModules(final JSONValue initModulesValue) {
    boolean _checkFeatureRestrictions = this.checkFeatureRestrictions(PackageJsonProperties.INIT_MODULES.name, initModulesValue, N4JSProjectSetupJsonValidatorExtension.RE_OR_RL_TYPE);
    if (_checkFeatureRestrictions) {
      if ((initModulesValue instanceof JSONArray)) {
        final Function1<JSONStringLiteral, Boolean> _function = (JSONStringLiteral l) -> {
          return Boolean.valueOf(l.getValue().isEmpty());
        };
        final Consumer<JSONStringLiteral> _function_1 = (JSONStringLiteral l) -> {
          this.addIssue(IssueCodes.getMessageForPKGJ_EMPTY_INIT_MODULE(), l, 
            IssueCodes.PKGJ_EMPTY_INIT_MODULE);
        };
        IterableExtensions.<JSONStringLiteral>filter(Iterables.<JSONStringLiteral>filter(((JSONArray)initModulesValue).getElements(), JSONStringLiteral.class), _function).forEach(_function_1);
      }
    }
  }
  
  /**
   * Checks the 'n4js.execModule' section.
   */
  @CheckProperty(property = PackageJsonProperties.EXEC_MODULE)
  public boolean checkExecModule(final JSONValue execModuleValue) {
    boolean _xifexpression = false;
    boolean _checkFeatureRestrictions = this.checkFeatureRestrictions(PackageJsonProperties.EXEC_MODULE.name, execModuleValue, N4JSProjectSetupJsonValidatorExtension.RE_OR_RL_TYPE);
    if (_checkFeatureRestrictions) {
      boolean _xifexpression_1 = false;
      if ((execModuleValue instanceof JSONStringLiteral)) {
        _xifexpression_1 = this.checkIsNonEmptyString(((JSONStringLiteral)execModuleValue), PackageJsonProperties.EXEC_MODULE);
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  /**
   * Checks the 'n4js.implementationId' section.
   */
  @CheckProperty(property = PackageJsonProperties.IMPLEMENTATION_ID)
  public boolean checkImplementationId(final JSONValue implementationIdValue) {
    return this.checkFeatureRestrictions(PackageJsonProperties.IMPLEMENTATION_ID.name, implementationIdValue, 
      ProjectTypePredicate.not(ProjectTypePredicate.or(N4JSProjectSetupJsonValidatorExtension.RE_OR_RL_TYPE, N4JSProjectSetupJsonValidatorExtension.TEST_TYPE)));
  }
  
  /**
   * Checks the 'n4js.implementedProjects' section.
   */
  @CheckProperty(property = PackageJsonProperties.IMPLEMENTED_PROJECTS)
  public void checkImplementedProjects(final JSONValue implementedProjectsValue) {
    boolean _checkFeatureRestrictions = this.checkFeatureRestrictions(PackageJsonProperties.IMPLEMENTED_PROJECTS.name, implementedProjectsValue, 
      ProjectTypePredicate.not(ProjectTypePredicate.or(N4JSProjectSetupJsonValidatorExtension.RE_OR_RL_TYPE, N4JSProjectSetupJsonValidatorExtension.TEST_TYPE)));
    if (_checkFeatureRestrictions) {
      final List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references = this.getReferencesFromJSONStringArray(implementedProjectsValue);
      this.checkReferencedProjects(references, this.forN4jsProjects(N4JSProjectSetupJsonValidatorExtension.API_TYPE), "implemented projects", false, true);
      final JSONValue implementationIdValue = this.getSingleDocumentValue(PackageJsonProperties.IMPLEMENTATION_ID);
      if (((!references.isEmpty()) && (implementationIdValue == null))) {
        this.addIssue(IssueCodes.getMessageForPKGJ_APIIMPL_MISSING_IMPL_ID(), implementedProjectsValue.eContainer(), 
          JSONPackage.Literals.NAME_VALUE_PAIR__NAME, IssueCodes.PKGJ_APIIMPL_MISSING_IMPL_ID);
      }
    }
  }
  
  /**
   * Validates the declared module filters.
   * 
   * For structural and duplicate validation see {@link PackageJsonValidatorExtension#checkModuleFilters}.
   * 
   * This includes limited syntactical validation of the wildcards as well as a check whether the filters
   * actually filter any resources or whether they are obsolete.
   */
  @CheckProperty(property = PackageJsonProperties.MODULE_FILTERS)
  public void checkModuleFilters(final JSONValue moduleFiltersValue) {
    final IN4JSProject project = this._iN4JSCore.findProject(moduleFiltersValue.eResource().getURI()).get();
    if ((!(moduleFiltersValue instanceof JSONObject))) {
      return;
    }
    final Multimap<String, JSONValue> nameValuePairs = this.collectObjectValues(((JSONObject) moduleFiltersValue));
    final Function1<JSONArray, EList<JSONValue>> _function = (JSONArray it) -> {
      return it.getElements();
    };
    final Function1<JSONValue, ASTTraceable<ModuleFilterSpecifier>> _function_1 = (JSONValue filter) -> {
      return ASTTraceable.<ModuleFilterSpecifier>of(filter, PackageJsonUtils.asModuleFilterSpecifierOrNull(filter));
    };
    final Iterable<ASTTraceable<ModuleFilterSpecifier>> filterSpecifierTraceables = IterableExtensions.<JSONValue, ASTTraceable<ModuleFilterSpecifier>>map(IterableExtensions.<JSONValue>filterNull(IterableExtensions.<JSONArray, JSONValue>flatMap(Iterables.<JSONArray>filter(nameValuePairs.values(), JSONArray.class), _function)), _function_1);
    this.holdsValidModuleSpecifiers(filterSpecifierTraceables, project);
  }
  
  private void holdsValidModuleSpecifiers(final Iterable<ASTTraceable<ModuleFilterSpecifier>> moduleFilterSpecifiers, final IN4JSProject project) {
    final ArrayList<ASTTraceable<ModuleFilterSpecifier>> validFilterSpecifier = new ArrayList<ASTTraceable<ModuleFilterSpecifier>>();
    for (final ASTTraceable<ModuleFilterSpecifier> filterSpecifier : moduleFilterSpecifiers) {
      {
        final boolean valid = this.holdsValidModuleFilterSpecifier(filterSpecifier);
        if (valid) {
          validFilterSpecifier.add(filterSpecifier);
        }
      }
    }
    this.internalCheckModuleSpecifierHasFile(project, validFilterSpecifier);
  }
  
  private boolean holdsValidModuleFilterSpecifier(final ASTTraceable<ModuleFilterSpecifier> filterSpecifierTraceable) {
    final String wrongWildcardPattern = "***";
    ModuleFilterSpecifier _element = null;
    if (filterSpecifierTraceable!=null) {
      _element=filterSpecifierTraceable.element;
    }
    final ModuleFilterSpecifier filterSpecifier = _element;
    String _moduleSpecifierWithWildcard = null;
    if (filterSpecifier!=null) {
      _moduleSpecifierWithWildcard=filterSpecifier.getModuleSpecifierWithWildcard();
    }
    boolean _tripleNotEquals = (_moduleSpecifierWithWildcard != null);
    if (_tripleNotEquals) {
      boolean _contains = filterSpecifier.getModuleSpecifierWithWildcard().contains(wrongWildcardPattern);
      if (_contains) {
        this.addIssue(
          IssueCodes.getMessageForPKGJ_INVALID_WILDCARD(wrongWildcardPattern), 
          filterSpecifierTraceable.astElement, 
          IssueCodes.PKGJ_INVALID_WILDCARD);
        return false;
      }
      final String wrongRelativeNavigation = "../";
      boolean _contains_1 = filterSpecifier.getModuleSpecifierWithWildcard().contains(wrongRelativeNavigation);
      if (_contains_1) {
        this.addIssue(
          IssueCodes.getMessageForPKGJ_NO_RELATIVE_NAVIGATION(), 
          filterSpecifierTraceable.astElement, 
          IssueCodes.PKGJ_NO_RELATIVE_NAVIGATION);
        return false;
      }
    }
    final JSONValue astElement = ((JSONValue) filterSpecifierTraceable.astElement);
    String _switchResult = null;
    boolean _matched = false;
    if (astElement instanceof JSONStringLiteral) {
      _matched=true;
      _switchResult = ((JSONStringLiteral)astElement).getValue();
    }
    if (!_matched) {
      if (astElement instanceof JSONObject) {
        _matched=true;
        _switchResult = JSONModelUtils.getPropertyAsStringOrNull(((JSONObject)astElement), PackageJsonProperties.NV_MODULE.name);
      }
    }
    final String moduleSpecifierWithWildcardFromAST = _switchResult;
    String _switchResult_1 = null;
    boolean _matched_1 = false;
    if (astElement instanceof JSONObject) {
      _matched_1=true;
      _switchResult_1 = JSONModelUtils.getPropertyAsStringOrNull(((JSONObject)astElement), PackageJsonProperties.NV_SOURCE_CONTAINER.name);
    }
    final String sourceContainerFromAST = _switchResult_1;
    if ((((moduleSpecifierWithWildcardFromAST != null) && moduleSpecifierWithWildcardFromAST.isEmpty()) || ((sourceContainerFromAST != null) && sourceContainerFromAST.isEmpty()))) {
      this.addIssue(IssueCodes.getMessageForPKGJ_INVALID_MODULE_FILTER_SPECIFIER_EMPTY(), 
        filterSpecifierTraceable.astElement, IssueCodes.PKGJ_INVALID_MODULE_FILTER_SPECIFIER_EMPTY);
      return false;
    }
    return true;
  }
  
  private void internalCheckModuleSpecifierHasFile(final IN4JSProject project, final List<ASTTraceable<ModuleFilterSpecifier>> filterSpecifiers) {
    final HashMap<ASTTraceable<ModuleFilterSpecifier>, Boolean> checkedFilterSpecifiers = new HashMap<ASTTraceable<ModuleFilterSpecifier>, Boolean>();
    final Function1<ASTTraceable<ModuleFilterSpecifier>, Boolean> _function = (ASTTraceable<ModuleFilterSpecifier> it) -> {
      return Boolean.valueOf((it.element != null));
    };
    final Function1<ASTTraceable<ModuleFilterSpecifier>, ASTTraceable<ModuleFilterSpecifier>> _function_1 = (ASTTraceable<ModuleFilterSpecifier> p) -> {
      return p;
    };
    final Function1<ASTTraceable<ModuleFilterSpecifier>, Boolean> _function_2 = (ASTTraceable<ModuleFilterSpecifier> it) -> {
      return Boolean.valueOf(false);
    };
    checkedFilterSpecifiers.putAll(IterableExtensions.<ASTTraceable<ModuleFilterSpecifier>, ASTTraceable<ModuleFilterSpecifier>, Boolean>toMap(IterableExtensions.<ASTTraceable<ModuleFilterSpecifier>>filter(filterSpecifiers, _function), _function_1, _function_2));
    try {
      final N4JSProjectSetupJsonValidatorExtension.ModuleSpecifierFileVisitor treeWalker = new N4JSProjectSetupJsonValidatorExtension.ModuleSpecifierFileVisitor(this, project, checkedFilterSpecifiers);
      Files.walkFileTree(project.getLocationPath(), treeWalker);
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        URI _uRI = this.getDocument().eResource().getURI();
        String _plus = ("Failed to check module filter section of package.json file " + _uRI);
        String _plus_1 = (_plus + ".");
        N4JSProjectSetupJsonValidatorExtension.LOGGER.error(_plus_1);
        e.printStackTrace();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    final Function1<Map.Entry<ASTTraceable<ModuleFilterSpecifier>, Boolean>, Boolean> _function_3 = (Map.Entry<ASTTraceable<ModuleFilterSpecifier>, Boolean> e_1) -> {
      Boolean _value = e_1.getValue();
      return Boolean.valueOf(((_value).booleanValue() == false));
    };
    final Function1<Map.Entry<ASTTraceable<ModuleFilterSpecifier>, Boolean>, ASTTraceable<ModuleFilterSpecifier>> _function_4 = (Map.Entry<ASTTraceable<ModuleFilterSpecifier>, Boolean> e_1) -> {
      return e_1.getKey();
    };
    final Iterable<ASTTraceable<ModuleFilterSpecifier>> unmatchedSpecifiers = IterableExtensions.<Map.Entry<ASTTraceable<ModuleFilterSpecifier>, Boolean>, ASTTraceable<ModuleFilterSpecifier>>map(IterableExtensions.<Map.Entry<ASTTraceable<ModuleFilterSpecifier>, Boolean>>filter(checkedFilterSpecifiers.entrySet(), _function_3), _function_4);
    for (final ASTTraceable<ModuleFilterSpecifier> filterSpecifier : unmatchedSpecifiers) {
      {
        final String msg = IssueCodes.getMessageForPKGJ_MODULE_FILTER_DOES_NOT_MATCH(filterSpecifier.element.getModuleSpecifierWithWildcard());
        this.addIssue(msg, filterSpecifier.astElement, IssueCodes.PKGJ_MODULE_FILTER_DOES_NOT_MATCH);
      }
    }
  }
  
  private void addNoValidationForN4JSFilesIssue(final ASTTraceable<ModuleFilterSpecifier> filterSpecifier) {
    EObject _eContainer = filterSpecifier.astElement.eContainer().eContainer();
    final String moduleFilterType = ((NameValuePair) _eContainer).getName();
    this.addIssue(IssueCodes.getMessageForPKGJ_FILTER_NO_N4JS_MATCH(moduleFilterType), filterSpecifier.astElement, 
      IssueCodes.PKGJ_FILTER_NO_N4JS_MATCH);
  }
  
  /**
   * Checks whether a given {@code feature} can be used with the declared project ({@link #getProjectDescription()}),
   * using the criteria defined by {@code supportedTypesPredicate}.
   * 
   * Adds INVALID_FEATURE_FOR_PROJECT_TYPE to {@code pair} otherwise.
   * 
   * @param featureDescription A textual user-facing description of the checked feature.
   * @param value The JSONValue that has been declared for the given feature.
   * @param supportedTypesPredicate A predicate which indicates whether the feature may be used for a given project type.
   */
  public boolean checkFeatureRestrictions(final String featureDescription, final JSONValue value, final Predicate<ProjectType> supportedTypesPredicate) {
    ProjectDescription _projectDescription = this.getProjectDescription();
    ProjectType _projectType = null;
    if (_projectDescription!=null) {
      _projectType=_projectDescription.getProjectType();
    }
    final ProjectType type = _projectType;
    if ((type == null)) {
      return false;
    }
    boolean _isEmptyValue = this.isEmptyValue(value);
    if (_isEmptyValue) {
      return true;
    }
    EObject _xifexpression = null;
    EObject _eContainer = value.eContainer();
    if ((_eContainer instanceof NameValuePair)) {
      _xifexpression = value.eContainer();
    } else {
      _xifexpression = value;
    }
    final EObject issueTarget = _xifexpression;
    boolean _apply = supportedTypesPredicate.apply(type);
    boolean _not = (!_apply);
    if (_not) {
      this.addIssue(IssueCodes.getMessageForINVALID_FEATURE_FOR_PROJECT_TYPE(StringExtensions.toFirstUpper(featureDescription), this.getLabel(type)), issueTarget, IssueCodes.INVALID_FEATURE_FOR_PROJECT_TYPE);
      return false;
    }
    return true;
  }
  
  /**
   * Return {@code true} iff the given value is considered empty (empty array or empty object).
   */
  private boolean isEmptyValue(final JSONValue value) {
    return (((value instanceof JSONArray) && ((JSONArray) value).getElements().isEmpty()) || ((value instanceof JSONObject) && ((JSONObject) value).getNameValuePairs().isEmpty()));
  }
  
  /**
   * Returns with a new predicate instance that provides {@code true} only if the given N4JS project
   * may be declared a dependency of a project of type {@link ProjectType#API}.
   * 
   * More specifically the given project must fulfill one of the following requirements:
   * <ul>
   * <li>The project type is API.</li>
   * <li>The project type is library.</li>
   * <li>The project type is validation.</li>
   * <li>The project type is plainjs.</li>
   * <li>The project type is runtime library.</li>
   * </ul>
   * Otherwise the predicate provides {@code false} value.
   */
  private Predicate<IN4JSProject> createAPIDependenciesPredicate() {
    final Predicate<IN4JSProject> _function = (IN4JSProject it) -> {
      return ProjectTypePredicate.anyOf(ProjectType.LIBRARY, ProjectType.VALIDATION, ProjectType.RUNTIME_LIBRARY, ProjectType.PLAINJS).apply(it.getProjectType());
    };
    return Predicates.<IN4JSProject>or(this.forN4jsProjects(N4JSProjectSetupJsonValidatorExtension.API_TYPE), _function);
  }
  
  /**
   * Returns with a new predicate instance that specifies the type of projects
   * that may be declared as dependency to the currently validated project description.
   */
  private Predicate<IN4JSProject> createDependenciesPredicate() {
    Predicate<IN4JSProject> _switchResult = null;
    ProjectType _projectType = this.getProjectDescription().getProjectType();
    if (_projectType != null) {
      switch (_projectType) {
        case API:
          _switchResult = this.createAPIDependenciesPredicate();
          break;
        case RUNTIME_LIBRARY:
          _switchResult = this.forN4jsProjects(N4JSProjectSetupJsonValidatorExtension.RL_TYPE);
          break;
        case DEFINITION:
          _switchResult = this.forN4jsProjects(ProjectTypePredicate.not(N4JSProjectSetupJsonValidatorExtension.PLAINJS_TYPE));
          break;
        default:
          _switchResult = Predicates.<IN4JSProject>alwaysTrue();
          break;
      }
    } else {
      _switchResult = Predicates.<IN4JSProject>alwaysTrue();
    }
    return _switchResult;
  }
  
  /**
   * Returns a list of {@link ValidationProjectReference}s that can be extracted from the given {@code value},
   * assuming it represents a valid {@code package.json} array of project IDs.
   * 
   * The returned references do not include version constraints.
   * 
   * Fails silently and returns an empty list, in case {@code value} is malformed.
   */
  private List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> getReferencesFromJSONStringArray(final JSONValue value) {
    if ((!(value instanceof JSONArray))) {
      return CollectionLiterals.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>emptyList();
    }
    final Function1<JSONStringLiteral, List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>> _function = (JSONStringLiteral literal) -> {
      return this.getReferencesFromJSONStringLiteral(literal);
    };
    return IterableExtensions.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>toList(IterableExtensions.<JSONStringLiteral, N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>flatMap(Iterables.<JSONStringLiteral>filter(((JSONArray) value).getElements(), JSONStringLiteral.class), _function));
  }
  
  /**
   * Returns a list of {@link ValidationProjectReference}s that can be extracted from the given {@code value},
   * assuming it represents a valid {@code package.json} dependencies object.
   * 
   * The returned references include version constraints.
   * 
   * Fails silently and returns an empty list, in case {@code value} is malformed.
   */
  private List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> getReferencesFromDependenciesObject(final JSONValue value) {
    if ((!(value instanceof JSONObject))) {
      return CollectionLiterals.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>emptyList();
    }
    final JSONObject jsonObj = ((JSONObject) value);
    final ArrayList<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> vprs = new ArrayList<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>();
    EList<NameValuePair> _nameValuePairs = jsonObj.getNameValuePairs();
    for (final NameValuePair pair : _nameValuePairs) {
      JSONValue _value = pair.getValue();
      if ((_value instanceof JSONStringLiteral)) {
        JSONValue _value_1 = pair.getValue();
        final JSONStringLiteral stringLit = ((JSONStringLiteral) _value_1);
        final String prjID = pair.getName();
        final NPMVersionRequirement npmVersion = this.semverHelper.parse(stringLit.getValue());
        final N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference vpr = new N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference(prjID, npmVersion, pair);
        vprs.add(vpr);
      }
    }
    return vprs;
  }
  
  /**
   * Returns a singleton list of the {@link ValidationProjectReference} that can be created from {@code literal}
   * assuming is of type {@link JSONStringLiteral}.
   * 
   * Returns an empty list, if this does not hold true.
   */
  private List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> getReferencesFromJSONStringLiteral(final JSONValue value) {
    List<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> _xifexpression = null;
    if ((value instanceof JSONStringLiteral)) {
      String _value = ((JSONStringLiteral)value).getValue();
      N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference _validationProjectReference = new N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference(_value, null, value);
      return Collections.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>unmodifiableList(CollectionLiterals.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>newArrayList(_validationProjectReference));
    } else {
      _xifexpression = CollectionLiterals.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>emptyList();
    }
    return _xifexpression;
  }
  
  /**
   * Checks the given iterable of referenced projects.
   * 
   * This includes checking whether the referenced project can be found and validating the given
   * {@link ValidationProjectReference#versionConstraint} if specified.
   * 
   * @param references
   * 				The list of project references to validate.
   * @param allProjects
   * 				A map of all projects that are accessible in the workspace.
   * @param sectionLabel
   * 				A user-facing description of which section of project references is currently validated (e.g. "tested projects").
   * @param enforceDependency
   * 				Additionally enforces that all given references are also listed as explicit project dependencies
   * @param allowReflexive
   * 				Specifies whether reflexive (self) references are allowed.
   */
  private void checkReferencedProjects(final Iterable<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references, final Predicate<IN4JSProject> projectPredicate, final String sectionLabel, final boolean enforceDependency, final boolean allowReflexive) {
    final ProjectDescription description = this.getProjectDescription();
    final Map<String, IN4JSProject> allProjects = this.getAllProjectsByName();
    final HashMultimap<String, N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> existentIds = HashMultimap.<String, N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>create();
    final URI projectDescriptionFileURI = this.getDocument().eResource().getURI();
    final IN4JSProject currentProject = this._iN4JSCore.findProject(projectDescriptionFileURI).orNull();
    final Function1<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, String> _function = (N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference it) -> {
      return it.referencedProjectName;
    };
    final Set<String> allReferencedProjectNames = IterableExtensions.<String>toSet(IterableExtensions.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, String>map(references, _function));
    for (final N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference ref : references) {
      {
        final String id = ref.referencedProjectName;
        if ((null != id)) {
          this.checkReference(ref, allProjects, description, currentProject, allReferencedProjectNames, existentIds, allowReflexive, projectPredicate, sectionLabel);
        }
      }
    }
    this.checkForDuplicateProjectReferences(existentIds);
    if (enforceDependency) {
      this.checkDeclaredDependencies(existentIds.values(), sectionLabel);
    }
  }
  
  private void checkReference(final N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference ref, final Map<String, IN4JSProject> allProjects, final ProjectDescription description, final IN4JSProject currentProject, final Set<String> allReferencedProjectNames, final HashMultimap<String, N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> existentIds, final boolean allowReflexive, final Predicate<IN4JSProject> projectPredicate, final String sectionLabel) {
    final String id = ref.referencedProjectName;
    final String currentProjectName = description.getProjectName();
    boolean _isEmpty = id.isEmpty();
    if (_isEmpty) {
      this.addIssue(IssueCodes.getMessageForPKGJ_EMPTY_PROJECT_REFERENCE(), ref.astRepresentation, 
        IssueCodes.PKGJ_EMPTY_PROJECT_REFERENCE);
      return;
    }
    final IN4JSProject project = allProjects.get(id);
    if (((null == project) || (null == project.getProjectType()))) {
      boolean _isExternal = currentProject.isExternal();
      boolean _not = (!_isExternal);
      if (_not) {
        final String msg = IssueCodes.getMessageForNON_EXISTING_PROJECT(id);
        String _xifexpression = null;
        if ((ref.npmVersion == null)) {
          _xifexpression = "";
        } else {
          _xifexpression = ref.npmVersion.toString();
        }
        final String packageVersion = _xifexpression;
        this.addIssue(msg, ref.astRepresentation, null, IssueCodes.NON_EXISTING_PROJECT, id, packageVersion);
      }
      return;
    } else {
      existentIds.put(id, ref);
    }
    if ((((!currentProject.isExternal()) && project.isExternal()) && (!this.indexSynchronizer.isInIndex(project.getProjectDescriptionLocation().orNull())))) {
      final String msg_1 = IssueCodes.getMessageForNON_REGISTERED_PROJECT(id);
      this.addIssue(msg_1, ref.astRepresentation, null, IssueCodes.NON_REGISTERED_PROJECT, id);
      return;
    }
    if ((Objects.equal(currentProjectName, id) && (!allowReflexive))) {
      this.addProjectReferencesItselfIssue(ref.astRepresentation);
      return;
    } else {
      boolean _apply = projectPredicate.apply(project);
      boolean _not_1 = (!_apply);
      if (_not_1) {
        this.addInvalidProjectTypeIssue(ref.astRepresentation, id, 
          project.getProjectType(), sectionLabel);
        return;
      }
    }
    final boolean ignoreVersion = (currentProject.isExternal() && description.isHasNestedNodeModulesFolder());
    if ((!ignoreVersion)) {
      this.checkVersions(currentProject, ref, id, allProjects);
    }
    ProjectType _projectType = description.getProjectType();
    boolean _tripleNotEquals = (_projectType != ProjectType.DEFINITION);
    if (_tripleNotEquals) {
      this.checkImplProjectPresentForReferencedTypeDef(ref, project, allReferencedProjectNames);
    }
  }
  
  private void checkImplProjectPresentForReferencedTypeDef(final N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference ref, final IN4JSProject referencedProject, final Set<String> allReferencedProjectNames) {
    ProjectType _projectType = referencedProject.getProjectType();
    boolean _tripleEquals = (_projectType == ProjectType.DEFINITION);
    if (_tripleEquals) {
      final String nameOfProjectDefinedByReferencedProject = referencedProject.getDefinesPackageName();
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(nameOfProjectDefinedByReferencedProject);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        boolean _contains = allReferencedProjectNames.contains(nameOfProjectDefinedByReferencedProject);
        boolean _not_1 = (!_contains);
        if (_not_1) {
          final String msg = IssueCodes.getMessageForPKGJ_IMPL_PROJECT_IS_MISSING_FOR_TYPE_DEF(nameOfProjectDefinedByReferencedProject, ref.referencedProjectName);
          this.addIssue(msg, ref.astRepresentation, IssueCodes.PKGJ_IMPL_PROJECT_IS_MISSING_FOR_TYPE_DEF);
        }
      }
    }
  }
  
  private void checkForDuplicateProjectReferences(final Multimap<String, N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> validProjectRefs) {
    final String currentVendor = this.getProjectDescription().getVendorId();
    final Consumer<String> _function = (String it) -> {
      int _size = validProjectRefs.get(it).size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        final HashMultimap<String, N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> referencesByNameAndVendor = HashMultimap.<String, N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>create();
        final Consumer<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> _function_1 = (N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference it_1) -> {
          referencesByNameAndVendor.put(currentVendor, it_1);
        };
        validProjectRefs.get(it).forEach(_function_1);
        final Consumer<String> _function_2 = (String it_1) -> {
          final Set<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> mappedRefs = referencesByNameAndVendor.get(it_1);
          int _size_1 = mappedRefs.size();
          boolean _greaterThan_1 = (_size_1 > 1);
          if (_greaterThan_1) {
            final Function1<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, Integer> _function_3 = (N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference it_2) -> {
              return Integer.valueOf(NodeModelUtils.findActualNodeFor(it_2.astRepresentation).getOffset());
            };
            final Consumer<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> _function_4 = (N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference ref) -> {
              this.addDuplicateProjectReferenceIssue(ref.astRepresentation, ref.referencedProjectName);
            };
            IterableExtensions.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference>tail(IterableExtensions.<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference, Integer>sortBy(mappedRefs, _function_3)).forEach(_function_4);
          }
        };
        referencesByNameAndVendor.keySet().forEach(_function_2);
      }
    };
    validProjectRefs.asMap().keySet().forEach(_function);
  }
  
  /**
   * Checks that the given {@code reference}s are also declared as explicit project dependencies
   * under {@code dependencies} or {@code devDependencies}.
   */
  private void checkDeclaredDependencies(final Iterable<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> references, final String sectionLabel) {
    final Map<String, ProjectDependency> declaredDependencies = this.getDeclaredProjectDependencies();
    final Consumer<N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference> _function = (N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference reference) -> {
      boolean _containsKey = declaredDependencies.containsKey(reference.referencedProjectName);
      boolean _not = (!_containsKey);
      if (_not) {
        this.addIssue(IssueCodes.getMessageForPKGJ_PROJECT_REFERENCE_MUST_BE_DEPENDENCY(reference.referencedProjectName, sectionLabel), 
          reference.astRepresentation, IssueCodes.PKGJ_PROJECT_REFERENCE_MUST_BE_DEPENDENCY);
      }
    };
    references.forEach(_function);
  }
  
  /**
   * Checks if version constraint of the project reference is satisfied by any available project.
   */
  private void checkVersions(final IN4JSProject curPrj, final N4JSProjectSetupJsonValidatorExtension.ValidationProjectReference ref, final String id, final Map<String, IN4JSProject> allProjects) {
    final NPMVersionRequirement desiredVersion = ref.npmVersion;
    if ((desiredVersion == null)) {
      return;
    }
    final IN4JSProject depProject = allProjects.get(id);
    final VersionNumber availableVersion = depProject.getVersion();
    final boolean availableVersionMatches = SemverMatcher.matches(availableVersion, desiredVersion);
    if (availableVersionMatches) {
      return;
    }
    final boolean curPrjShadows = this.shadowingInfoHelper.isShadowingProject(curPrj);
    final boolean dependencyShadows = this.shadowingInfoHelper.isShadowingProject(depProject);
    final String desiredStr = SemverSerializer.serialize(desiredVersion);
    final String availableStr = SemverSerializer.serialize(availableVersion);
    if ((curPrjShadows || dependencyShadows)) {
      String _xifexpression = null;
      if (curPrjShadows) {
        _xifexpression = "shadowing ";
      } else {
        _xifexpression = "";
      }
      final String curPrjShadowsStr = _xifexpression;
      String _xifexpression_1 = null;
      if (dependencyShadows) {
        _xifexpression_1 = "shadowed ";
      } else {
        _xifexpression_1 = "";
      }
      final String dependencyShadowsStr = _xifexpression_1;
      final String msg = IssueCodes.getMessageForNO_MATCHING_VERSION_SHADOWING(curPrjShadowsStr, dependencyShadowsStr, id, desiredStr, availableStr);
      this.addIssue(msg, ref.astRepresentation, IssueCodes.NO_MATCHING_VERSION_SHADOWING);
    } else {
      final String msg_1 = IssueCodes.getMessageForNO_MATCHING_VERSION(id, desiredStr, availableStr);
      this.addIssue(msg_1, ref.astRepresentation, null, IssueCodes.NO_MATCHING_VERSION, id, desiredVersion.toString());
    }
  }
  
  /**
   * Returns the {@link ProjectDescription} that can be created based on the information
   * to be found in the currently validated {@link JSONDocument}.
   * 
   * @See {@link ProjectDescriptionLoader}
   */
  protected ProjectDescription getProjectDescription() {
    final Supplier<ProjectDescription> _function = () -> {
      ProjectDescription _xblockexpression = null;
      {
        final JSONDocument doc = this.getDocument();
        _xblockexpression = this.projectDescriptionLoader.loadProjectDescriptionAtLocation(doc.eResource().getURI().trimSegments(1), doc);
      }
      return _xblockexpression;
    };
    return this.<ProjectDescription>contextMemoize(N4JSProjectSetupJsonValidatorExtension.PROJECT_DESCRIPTION_CACHE, _function);
  }
  
  /**
   * Returns a cached view on all declared project dependencies mapped to the dependency project ID.
   * 
   * @See {@link #getProjectDescription()}.
   */
  protected Map<String, ProjectDependency> getDeclaredProjectDependencies() {
    final Supplier<Map<String, ProjectDependency>> _function = () -> {
      final ProjectDescription description = this.getProjectDescription();
      final Function1<ProjectDependency, String> _function_1 = (ProjectDependency d) -> {
        return d.getProjectName();
      };
      return IterableExtensions.<String, ProjectDependency>toMap(description.getProjectDependencies(), _function_1);
    };
    return this.<Map<String, ProjectDependency>>contextMemoize(N4JSProjectSetupJsonValidatorExtension.DECLARED_DEPENDENCIES_CACHE, _function);
  }
  
  private CharSequence getLabel(final ProjectType it) {
    CharSequence _xifexpression = null;
    if ((null == it)) {
      StringConcatenation _builder = new StringConcatenation();
      _xifexpression = _builder;
    } else {
      _xifexpression = this.upperUnderscoreToHumanReadable(it.toString());
    }
    return _xifexpression;
  }
  
  private String upperUnderscoreToHumanReadable(final String s) {
    return Strings.nullToEmpty(s).replaceAll("_", " ").toLowerCase();
  }
  
  private void addProjectReferencesItselfIssue(final EObject target) {
    this.addIssue(IssueCodes.getMessageForPROJECT_REFERENCES_ITSELF(), target, IssueCodes.PROJECT_REFERENCES_ITSELF);
  }
  
  private void addInvalidProjectTypeIssue(final EObject target, final String projectName, final ProjectType type, final String sectionLabel) {
    this.addIssue(IssueCodes.getMessageForINVALID_PROJECT_TYPE_REF(projectName, this.getLabel(type), sectionLabel), target, IssueCodes.INVALID_PROJECT_TYPE_REF);
  }
  
  private void addDuplicateProjectReferenceIssue(final EObject target, final String name) {
    this.addIssue(IssueCodes.getMessageForDUPLICATE_PROJECT_REF(name), target, IssueCodes.DUPLICATE_PROJECT_REF);
  }
  
  /**
   * Adds an issue to every non-null element in {@code preferredTargets}.
   * 
   * If {@code preferredTargets} is empty (or contains null entries only), adds an issue to
   * the {@code name} property of the {@code package.json} file.
   * 
   * If there is no {@code name} property, adds an issue to the whole document (see {@link #getDocument()}).
   */
  private void addIssuePreferred(final Iterable<? extends EObject> preferredTargets, final String message, final String issueCode) {
    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.filterNull(preferredTargets));
    boolean _not = (!_isEmpty);
    if (_not) {
      final Consumer<EObject> _function = (EObject t) -> {
        this.addIssue(message, t, issueCode);
      };
      IterableExtensions.filterNull(preferredTargets).forEach(_function);
      return;
    }
    final JSONValue nameValue = this.getSingleDocumentValue(PackageJsonProperties.NAME);
    if ((nameValue != null)) {
      this.addIssue(message, nameValue, issueCode);
      return;
    }
    this.addIssue(message, this.getDocument(), issueCode);
  }
  
  /**
   * Returns a map between all available project IDs and their corresponding
   * {@link IN4JSProject} representations.
   * 
   * The result of this method is cached in the validation context.
   */
  private Map<String, IN4JSProject> getAllProjectsByName() {
    final Supplier<Map<String, IN4JSProject>> _function = () -> {
      final Map<String, IN4JSProject> res = new HashMap<String, IN4JSProject>();
      final Consumer<IN4JSProject> _function_1 = (IN4JSProject p) -> {
        res.put(p.getProjectName(), p);
      };
      this._iN4JSCore.findAllProjects().forEach(_function_1);
      List<org.eclipse.xtext.util.Pair<URI, ProjectDescription>> _projectsIncludingUnnecessary = this.extWS.getProjectsIncludingUnnecessary();
      for (final org.eclipse.xtext.util.Pair<URI, ProjectDescription> pair : _projectsIncludingUnnecessary) {
        {
          final URI location = pair.getFirst();
          final IN4JSProject project = this._iN4JSCore.findProject(location).orNull();
          if (((!this.shadowingInfoHelper.isShadowedProject(project)) && (!res.containsKey(project.getProjectName())))) {
            res.put(project.getProjectName(), project);
          }
        }
      }
      return res;
    };
    return this.<Map<String, IN4JSProject>>contextMemoize(N4JSProjectSetupJsonValidatorExtension.ALL_EXISTING_PROJECT_CACHE, _function);
  }
  
  /**
   * Transforms a {@link ProjectTypePredicate} into a predicate for {@link IN4JSProject}.
   */
  private Predicate<IN4JSProject> forN4jsProjects(final Predicate<ProjectType> predicate) {
    final Predicate<IN4JSProject> _function = (IN4JSProject it) -> {
      return predicate.apply(it.getProjectType());
    };
    return _function;
  }
}
