/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.tests.codegen;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import org.eclipse.n4js.tests.codegen.Classifier;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;

/**
 * Generates code for a module containing imports and classifiers.
 */
@SuppressWarnings("all")
public class Module {
  private String name;
  
  private List<Classifier<?>> classifiers;
  
  private Map<String, List<String>> imports;
  
  /**
   * Creates a new instance with the given parameters.
   * 
   * @param name the module name without extension
   */
  public Module(final String name) {
    this.name = Objects.<String>requireNonNull(name);
  }
  
  /**
   * Returns the name of this module.
   * 
   * @return the name of this module
   */
  public String getName() {
    return this.name;
  }
  
  /**
   * Adds the given classifier to the module built by this builder.
   * 
   * @param classifier the classifier to add
   */
  public Module addClassifier(final Classifier<?> classifier) {
    if ((this.classifiers == null)) {
      this.classifiers = CollectionLiterals.<Classifier<?>>newLinkedList();
    }
    this.classifiers.add(Objects.<Classifier<?>>requireNonNull(classifier));
    return this;
  }
  
  /**
   * Adds an import to the module built by this builder.
   * 
   * @param importedType the name of the type to be imported
   * @param sourceModule the module containing the imported type
   */
  public Module addImport(final String importedType, final Module sourceModule) {
    return this.addImport(importedType, sourceModule.name);
  }
  
  /**
   * Adds an import to the module built by this builder.
   * 
   * @param importedType the classifier representing the type to be imported
   * @param sourceModule the module containing the imported type
   */
  public Module addImport(final Classifier<?> importedType, final Module sourceModule) {
    return this.addImport(importedType.getName(), sourceModule);
  }
  
  /**
   * Adds an import to the module built by this builder.
   * 
   * @param importedType the name of the type to be imported
   * @param sourceModule the name of the module containing the imported type
   */
  public Module addImport(final String importedType, final String sourceModule) {
    if ((this.imports == null)) {
      this.imports = CollectionLiterals.<String, List<String>>newHashMap();
    }
    List<String> importedTypesForModule = this.imports.get(Objects.<Object>requireNonNull(sourceModule));
    if ((importedTypesForModule == null)) {
      importedTypesForModule = CollectionLiterals.<String>newLinkedList();
      this.imports.put(sourceModule, importedTypesForModule);
    }
    importedTypesForModule.add(Objects.<String>requireNonNull(importedType));
    return this;
  }
  
  /**
   * Creates this module as a file in the given parent directory, which must already exist.
   * 
   * @param parentDirectory a file representing the parent directory
   */
  public void create(final File parentDirectory) {
    try {
      Objects.<File>requireNonNull(parentDirectory);
      boolean _exists = parentDirectory.exists();
      boolean _not = (!_exists);
      if (_not) {
        throw new IOException((("Directory \'" + parentDirectory) + "\' does not exist"));
      }
      boolean _isDirectory = parentDirectory.isDirectory();
      boolean _not_1 = (!_isDirectory);
      if (_not_1) {
        throw new IOException((("\'" + parentDirectory) + "\' is not a directory"));
      }
      final File filePath = new File(parentDirectory, (this.name + ".n4js"));
      FileWriter out = null;
      try {
        FileWriter _fileWriter = new FileWriter(filePath);
        out = _fileWriter;
        out.write(this.generate().toString());
      } finally {
        if ((out != null)) {
          out.close();
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Generates the N4JS code for this module.
   */
  public CharSequence generate() {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _hasImports = this.hasImports();
      if (_hasImports) {
        CharSequence _generateImports = this.generateImports();
        _builder.append(_generateImports);
        _builder.newLineIfNotEmpty();
        _builder.newLine();
      }
    }
    CharSequence _generateClassifiers = this.generateClassifiers();
    _builder.append(_generateClassifiers);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  private CharSequence generateImports() {
    StringConcatenation _builder = new StringConcatenation();
    {
      Set<Map.Entry<String, List<String>>> _entrySet = this.imports.entrySet();
      for(final Map.Entry<String, List<String>> entry : _entrySet) {
        _builder.append("import { ");
        {
          List<String> _value = entry.getValue();
          boolean _hasElements = false;
          for(final String type : _value) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "");
            }
            _builder.append(type);
          }
        }
        _builder.append(" } from \"");
        String _key = entry.getKey();
        _builder.append(_key);
        _builder.append("\";");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
  
  private CharSequence generateClassifiers() {
    StringConcatenation _builder = new StringConcatenation();
    {
      for(final Classifier<?> classifier : this.classifiers) {
        CharSequence _generate = classifier.generate();
        _builder.append(_generate);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
  
  private boolean hasImports() {
    return ((this.imports != null) && (!this.imports.isEmpty()));
  }
}
