/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.generator.GeneratorOption;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.CatchBlock;
import org.eclipse.n4js.n4JS.DestructNode;
import org.eclipse.n4js.n4JS.DestructureUtils;
import org.eclipse.n4js.n4JS.EqualityOperator;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.ForStatement;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.IndexedAccessExpression;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParenExpression;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclarationOrBinding;
import org.eclipse.n4js.n4JS.VariableEnvironmentElement;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.n4JS.VariableStatementKeyword;
import org.eclipse.n4js.n4JS.WithStatement;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.es.transform.StaticPolyfillTransformation;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryIMOnly;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Transforms ES6 destructuring patterns into equivalent ES5 code. If the target engine supports ES6 destructuring
 * patterns, this transformation can simply be deactivated.
 * <p>
 * For details on destructuring patterns, see documentation of class {@link DestructNode}.
 */
@TransformationDependency.Optional(GeneratorOption.Destructuring)
@TransformationDependency.ExcludesAfter(StaticPolyfillTransformation.class)
@SuppressWarnings("all")
public class DestructuringTransformation extends Transformation {
  /**
   * Counts destructuring patterns per variable environment. Used to avoid name clashes of helper variables.
   */
  private final Map<VariableEnvironmentElement, Integer> destructsPerScope = CollectionLiterals.<VariableEnvironmentElement, Integer>newHashMap();
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Function1<VariableStatement, Boolean> _function = (VariableStatement it) -> {
      return Boolean.valueOf(DestructureUtils.containsDestructuringPattern(it));
    };
    final List<VariableStatement> destructBindings = IterableExtensions.<VariableStatement>toList(IterableExtensions.<VariableStatement>filter(this.<VariableStatement>collectNodes(this.getState().im, VariableStatement.class, true), _function));
    final Function1<AssignmentExpression, Boolean> _function_1 = (AssignmentExpression it) -> {
      return Boolean.valueOf(DestructureUtils.isTopOfDestructuringAssignment(it));
    };
    final Function1<AssignmentExpression, Boolean> _function_2 = (AssignmentExpression it) -> {
      return Boolean.valueOf(DestructureUtils.isRoot(it));
    };
    final List<AssignmentExpression> destructAssignments = IterableExtensions.<AssignmentExpression>toList(IterableExtensions.<AssignmentExpression>filter(IterableExtensions.<AssignmentExpression>filter(this.<AssignmentExpression>collectNodes(this.getState().im, AssignmentExpression.class, true), _function_1), _function_2));
    final Function1<ForStatement, Boolean> _function_3 = (ForStatement it) -> {
      return Boolean.valueOf((DestructureUtils.containsDestructuringPattern(it) || DestructureUtils.isTopOfDestructuringForStatement(it)));
    };
    final List<ForStatement> destructForStmnts = IterableExtensions.<ForStatement>toList(IterableExtensions.<ForStatement>filter(this.<ForStatement>collectNodes(this.getState().im, ForStatement.class, true), _function_3));
    final Consumer<VariableStatement> _function_4 = (VariableStatement it) -> {
      this.transformDestructuringBindings(it);
    };
    destructBindings.forEach(_function_4);
    final Consumer<AssignmentExpression> _function_5 = (AssignmentExpression it) -> {
      this.transformDestructuringAssignment(it);
    };
    destructAssignments.forEach(_function_5);
    final Consumer<ForStatement> _function_6 = (ForStatement it) -> {
      this.transformForStatementWithDestructuring(it);
    };
    destructForStmnts.forEach(_function_6);
  }
  
  /**
   * Transforms not a single destructuring binding but all destructuring bindings in the given variable statement.
   */
  private void transformDestructuringBindings(final VariableStatement stmnt) {
    final List<VariableDeclaration> newVarDecls = this.computeVariableDeclarations(stmnt.getVarDeclsOrBindings());
    stmnt.getVarDeclsOrBindings().clear();
    EList<VariableDeclarationOrBinding> _varDeclsOrBindings = stmnt.getVarDeclsOrBindings();
    Iterables.<VariableDeclarationOrBinding>addAll(_varDeclsOrBindings, newVarDecls);
  }
  
  /**
   * Transforms (a single) destructuring assignment.
   */
  public void transformDestructuringAssignment(final AssignmentExpression expr) {
    String fparName = null;
    FunctionExpression helperFun = null;
    fparName = ("$destruct" + "Param0");
    final FormalParameter fpar = TranspilerBuilderBlocks._FormalParameter(fparName);
    helperFun = TranspilerBuilderBlocks._FunExpr(false, null, fpar);
    final EList<Statement> helperFunContents = helperFun.getBody().getStatements();
    final DestructNode rootNode = DestructNode.unify(expr);
    final ArrayList<VariableDeclaration> helperVars = CollectionLiterals.<VariableDeclaration>newArrayList();
    final ArrayList<Pair<SymbolTableEntry, ? extends Expression>> simpleAssignments = CollectionLiterals.<Pair<SymbolTableEntry, ? extends Expression>>newArrayList();
    this.traverse(helperVars, simpleAssignments, rootNode, expr.getRhs(), fparName);
    final Function1<VariableDeclaration, VariableStatement> _function = (VariableDeclaration it) -> {
      return TranspilerBuilderBlocks._VariableStatement(it);
    };
    List<VariableStatement> _map = ListExtensions.<VariableDeclaration, VariableStatement>map(helperVars, _function);
    Iterables.<Statement>addAll(helperFunContents, _map);
    final Function1<Pair<SymbolTableEntry, ? extends Expression>, ExpressionStatement> _function_1 = (Pair<SymbolTableEntry, ? extends Expression> it) -> {
      return TranspilerBuilderBlocks._ExprStmnt(
        TranspilerBuilderBlocks._AssignmentExpr(
          this.__NSSafe_IdentRef(it.getKey()), 
          it.getValue()));
    };
    List<ExpressionStatement> _map_1 = ListExtensions.<Pair<SymbolTableEntry, ? extends Expression>, ExpressionStatement>map(simpleAssignments, _function_1);
    Iterables.<Statement>addAll(helperFunContents, _map_1);
    final SymbolTableEntry firstHelperVarSTE = this.findSymbolTableEntryForElement(helperVars.get(0), false);
    ReturnStatement __ReturnStmnt = TranspilerBuilderBlocks._ReturnStmnt(TranspilerBuilderBlocks._IdentRef(firstHelperVarSTE));
    helperFunContents.add(__ReturnStmnt);
    final ParameterizedCallExpression callExpr = TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._Parenthesis(helperFun), expr.getRhs());
    this.replace(expr, callExpr);
  }
  
  private void transformForStatementWithDestructuring(final ForStatement stmnt) {
    boolean _isForPlain = stmnt.isForPlain();
    if (_isForPlain) {
      boolean _isEmpty = stmnt.getVarDeclsOrBindings().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final List<VariableDeclaration> newVarDecls = this.computeVariableDeclarations(stmnt.getVarDeclsOrBindings());
        stmnt.getVarDeclsOrBindings().clear();
        EList<VariableDeclarationOrBinding> _varDeclsOrBindings = stmnt.getVarDeclsOrBindings();
        Iterables.<VariableDeclarationOrBinding>addAll(_varDeclsOrBindings, newVarDecls);
      } else {
      }
    } else {
      final int depth = DestructuringTransformation.getNestingDepth(stmnt);
      final VariableDeclaration iterVar = TranspilerBuilderBlocks._VariableDeclaration(("$destructStep$" + Integer.valueOf(depth)));
      final SymbolTableEntryIMOnly iterVarSTE = this.createSymbolTableEntryIMOnly(iterVar);
      boolean needDeclarations = false;
      VariableStatementKeyword varStmtKeyword = VariableStatementKeyword.VAR;
      final ArrayList<VariableDeclaration> helperVars = CollectionLiterals.<VariableDeclaration>newArrayList();
      final ArrayList<Pair<SymbolTableEntry, ? extends Expression>> simpleAssignments = CollectionLiterals.<Pair<SymbolTableEntry, ? extends Expression>>newArrayList();
      boolean _isEmpty_1 = stmnt.getVarDeclsOrBindings().isEmpty();
      boolean _not_1 = (!_isEmpty_1);
      if (_not_1) {
        this.assertTrue("there should be exactly one VariableBinding in stmnt.varDeclsOrBindings", 
          ((stmnt.getVarDeclsOrBindings().size() == 1) && (stmnt.getVarDeclsOrBindings().get(0) instanceof VariableBinding)));
        VariableDeclarationOrBinding _head = IterableExtensions.<VariableDeclarationOrBinding>head(stmnt.getVarDeclsOrBindings());
        final DestructNode rootNode = DestructNode.unify(((VariableBinding) _head));
        this.traverse(helperVars, simpleAssignments, rootNode, TranspilerBuilderBlocks._IdentRef(iterVarSTE), null);
        needDeclarations = true;
        varStmtKeyword = stmnt.getVarStmtKeyword();
      } else {
        if (((stmnt.getInitExpr() instanceof ArrayLiteral) || (stmnt.getInitExpr() instanceof ObjectLiteral))) {
          final DestructNode rootNode_1 = DestructNode.unify(stmnt);
          this.traverse(helperVars, simpleAssignments, rootNode_1, TranspilerBuilderBlocks._IdentRef(iterVarSTE), null);
          needDeclarations = false;
        } else {
          throw new IllegalArgumentException();
        }
      }
      Statement _statement = stmnt.getStatement();
      boolean _not_2 = (!(_statement instanceof Block));
      if (_not_2) {
        stmnt.setStatement(TranspilerBuilderBlocks._Block(stmnt.getStatement()));
      }
      Statement _statement_1 = stmnt.getStatement();
      final Block body = ((Block) _statement_1);
      final ArrayList<Statement> toBeInserted = CollectionLiterals.<Statement>newArrayList();
      if (needDeclarations) {
        final Function1<Pair<SymbolTableEntry, ? extends Expression>, VariableDeclaration> _function = (Pair<SymbolTableEntry, ? extends Expression> it) -> {
          final VariableDeclaration varDecl = DestructuringTransformation.getVariableDeclarationFromSTE(it.getKey());
          varDecl.setExpression(it.getValue());
          return varDecl;
        };
        VariableStatement __VariableStatement = TranspilerBuilderBlocks._VariableStatement(varStmtKeyword, ((VariableDeclaration[])Conversions.unwrapArray(ListExtensions.<Pair<SymbolTableEntry, ? extends Expression>, VariableDeclaration>map(simpleAssignments, _function), VariableDeclaration.class)));
        toBeInserted.add(__VariableStatement);
      } else {
        VariableStatement __VariableStatement_1 = TranspilerBuilderBlocks._VariableStatement(((VariableDeclaration[])Conversions.unwrapArray(helperVars, VariableDeclaration.class)));
        toBeInserted.add(__VariableStatement_1);
        final Function1<Pair<SymbolTableEntry, ? extends Expression>, ExpressionStatement> _function_1 = (Pair<SymbolTableEntry, ? extends Expression> it) -> {
          return TranspilerBuilderBlocks._ExprStmnt(
            TranspilerBuilderBlocks._AssignmentExpr(
              this.__NSSafe_IdentRef(it.getKey()), 
              it.getValue()));
        };
        List<ExpressionStatement> _map = ListExtensions.<Pair<SymbolTableEntry, ? extends Expression>, ExpressionStatement>map(simpleAssignments, _function_1);
        Iterables.<Statement>addAll(toBeInserted, _map);
      }
      body.getStatements().addAll(0, toBeInserted);
      stmnt.setInitExpr(null);
      stmnt.getVarDeclsOrBindings().clear();
      EList<VariableDeclarationOrBinding> _varDeclsOrBindings_1 = stmnt.getVarDeclsOrBindings();
      _varDeclsOrBindings_1.add(iterVar);
    }
  }
  
  /**
   * Breaks down all VariableBindings in the given list into simple variable declarations and returns a list containing
   * the variable declarations that were contained in the given list from the beginning plus those created when
   * breaking down the variable bindings. The order of the elements in the given list is preserved!
   */
  private List<VariableDeclaration> computeVariableDeclarations(final List<VariableDeclarationOrBinding> varDeclsOrBindings) {
    final ArrayList<VariableDeclaration> result = CollectionLiterals.<VariableDeclaration>newArrayList();
    ArrayList<VariableDeclarationOrBinding> _arrayList = new ArrayList<VariableDeclarationOrBinding>(varDeclsOrBindings);
    for (final VariableDeclarationOrBinding vdeclOrBinding : _arrayList) {
      if ((vdeclOrBinding instanceof VariableDeclaration)) {
        result.add(((VariableDeclaration)vdeclOrBinding));
      } else {
        if ((vdeclOrBinding instanceof VariableBinding)) {
          final DestructNode rootNode = DestructNode.unify(((VariableBinding)vdeclOrBinding));
          final ArrayList<VariableDeclaration> helperVars = CollectionLiterals.<VariableDeclaration>newArrayList();
          final ArrayList<Pair<SymbolTableEntry, ? extends Expression>> simpleAssignments = CollectionLiterals.<Pair<SymbolTableEntry, ? extends Expression>>newArrayList();
          this.traverse(helperVars, simpleAssignments, rootNode, ((VariableBinding)vdeclOrBinding).getExpression(), null);
          final Function1<Pair<SymbolTableEntry, ? extends Expression>, VariableDeclaration> _function = (Pair<SymbolTableEntry, ? extends Expression> it) -> {
            VariableDeclaration varDecl = DestructuringTransformation.getVariableDeclarationFromSTE(it.getKey());
            varDecl.setExpression(it.getValue());
            return varDecl;
          };
          List<VariableDeclaration> _map = ListExtensions.<Pair<SymbolTableEntry, ? extends Expression>, VariableDeclaration>map(simpleAssignments, _function);
          Iterables.<VariableDeclaration>addAll(result, _map);
        }
      }
    }
    return result;
  }
  
  private void traverse(final List<VariableDeclaration> helperVars, final List<Pair<SymbolTableEntry, ? extends Expression>> simpleAssignments, final DestructNode rootNode, final Expression value, final String fparName) {
    VariableEnvironmentElement _elvis = null;
    VariableEnvironmentElement _scope = N4JSASTUtils.getScope(rootNode.getAstElement(), false);
    if (_scope != null) {
      _elvis = _scope;
    } else {
      _elvis = this.getState().im;
    }
    final VariableEnvironmentElement scope = _elvis;
    final BiFunction<Integer, Integer, Integer> _function = (Integer i, Integer j) -> {
      return Integer.valueOf(((i).intValue() + (j).intValue()));
    };
    Integer _merge = this.destructsPerScope.merge(scope, Integer.valueOf(1), _function);
    final int n = ((_merge).intValue() - 1);
    this.traverse(helperVars, simpleAssignments, rootNode.getNestedNodes(), value, fparName, Integer.toString(n));
  }
  
  /**
   * Breaks down the destructuring pattern, represented by the given {@link DestructNode}s, into a number of simple
   * assignments (without destructuring) that will be added to argument 'simpleAssignments'. The order of those simple
   * assignments matters. Nested patterns as returned by {@link DestructNode#getNestedNodes()} are also broken down.
   * fparName, if not null, is the parameter name of the enclosing function.
   */
  private void traverse(final List<VariableDeclaration> helperVars, final List<Pair<SymbolTableEntry, ? extends Expression>> simpleAssignments, final DestructNode[] nodes, final Expression value, final String fparName, final String helperVarSuffix) {
    final int len = nodes.length;
    final Function1<DestructNode, Boolean> _function = (DestructNode it) -> {
      return Boolean.valueOf(it.isPositional());
    };
    final boolean isPositionalPattern = IterableExtensions.<DestructNode>exists(((Iterable<DestructNode>)Conversions.doWrapArray(nodes)), _function);
    final boolean isRest = ((isPositionalPattern && (!((List<DestructNode>)Conversions.doWrapArray(nodes)).isEmpty())) && IterableExtensions.<DestructNode>last(((Iterable<DestructNode>)Conversions.doWrapArray(nodes))).isRest());
    final String currHelperVarName = ("$destruct" + helperVarSuffix);
    final VariableDeclaration currHelperVarDecl = TranspilerBuilderBlocks._VariableDeclaration(currHelperVarName);
    helperVars.add(currHelperVarDecl);
    final SymbolTableEntry currHelperVarSTE = this.findSymbolTableEntryForElement(currHelperVarDecl, true);
    VariableDeclaration _variableDeclarationFromSTE = DestructuringTransformation.getVariableDeclarationFromSTE(currHelperVarSTE);
    boolean _tripleEquals = (_variableDeclarationFromSTE == currHelperVarDecl);
    this.assertTrue("", _tripleEquals);
    this.assertTrue("", currHelperVarSTE.getElementsOfThisName().contains(currHelperVarDecl));
    SymbolTableEntryInternal $sliceToArrayForDestructSTE = this.steFor_$sliceToArrayForDestruct();
    if (isRest) {
      ParameterizedCallExpression __CallExpr = TranspilerBuilderBlocks._CallExpr(
        TranspilerBuilderBlocks._Snippet("function(arr){return Array.isArray(arr) ? arr : Array.from(arr);}"), value);
      Pair<SymbolTableEntry, ParameterizedCallExpression> _mappedTo = Pair.<SymbolTableEntry, ParameterizedCallExpression>of(currHelperVarSTE, __CallExpr);
      simpleAssignments.add(_mappedTo);
    } else {
      Expression _xifexpression = null;
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(fparName);
      if (_isNullOrEmpty) {
        _xifexpression = value;
      } else {
        IdentifierRef_IM _xblockexpression = null;
        {
          final SymbolTableEntryInternal fparSTE = this.getSymbolTableEntryInternal(fparName, true);
          _xblockexpression = TranspilerBuilderBlocks._IdentRef(fparSTE);
        }
        _xifexpression = _xblockexpression;
      }
      final Expression passValue = _xifexpression;
      if (isPositionalPattern) {
        ParameterizedCallExpression __CallExpr_1 = TranspilerBuilderBlocks._CallExpr(
          TranspilerBuilderBlocks._IdentRef($sliceToArrayForDestructSTE), 
          TranspilerBuilderBlocks._Parenthesis(passValue), 
          TranspilerBuilderBlocks._NumericLiteral(len));
        Pair<SymbolTableEntry, ParameterizedCallExpression> _mappedTo_1 = Pair.<SymbolTableEntry, ParameterizedCallExpression>of(currHelperVarSTE, __CallExpr_1);
        simpleAssignments.add(_mappedTo_1);
      } else {
        ParenExpression __Parenthesis = TranspilerBuilderBlocks._Parenthesis(passValue);
        Pair<SymbolTableEntry, ParenExpression> _mappedTo_2 = Pair.<SymbolTableEntry, ParenExpression>of(currHelperVarSTE, __Parenthesis);
        simpleAssignments.add(_mappedTo_2);
      }
    }
    final SymbolTableEntryOriginal sliceSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.arrayType(this.getState().G), "slice", false, false, true);
    int nestedPatternsCounter = 0;
    for (int i = 0; (i < len); i++) {
      {
        final DestructNode currNode = nodes[i];
        Expression _xifexpression_1 = null;
        if ((isRest && (i == (len - 1)))) {
          _xifexpression_1 = TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._PropertyAccessExpr(currHelperVarSTE, sliceSTE), TranspilerBuilderBlocks._NumericLiteral(i));
        } else {
          IndexedAccessExpression _xifexpression_2 = null;
          if (isPositionalPattern) {
            _xifexpression_2 = TranspilerBuilderBlocks._IndexAccessExpr(currHelperVarSTE, TranspilerBuilderBlocks._NumericLiteral(i));
          } else {
            _xifexpression_2 = TranspilerBuilderBlocks._IndexAccessExpr(currHelperVarSTE, TranspilerBuilderBlocks._StringLiteral(currNode.getPropName()));
          }
          _xifexpression_1 = _xifexpression_2;
        }
        final Expression currValueRaw = _xifexpression_1;
        Expression _xifexpression_3 = null;
        Expression _defaultExpr = currNode.getDefaultExpr();
        boolean _tripleNotEquals = (_defaultExpr != null);
        if (_tripleNotEquals) {
          _xifexpression_3 = TranspilerBuilderBlocks._ConditionalExpr(
            TranspilerBuilderBlocks._EqualityExpr(
              this.<Expression>copy(currValueRaw), 
              EqualityOperator.SAME, 
              this.undefinedRef()), 
            TranspilerBuilderBlocks._Parenthesis(
              currNode.getDefaultExpr()), currValueRaw);
        } else {
          _xifexpression_3 = currValueRaw;
        }
        final Expression currValue = _xifexpression_3;
        if (((currNode.getVarRef() != null) || (currNode.getVarDecl() != null))) {
          EObject _elvis = null;
          IdentifierRef _varRef = currNode.getVarRef();
          if (_varRef != null) {
            _elvis = _varRef;
          } else {
            VariableDeclaration _varDecl = currNode.getVarDecl();
            _elvis = _varDecl;
          }
          final EObject varSource = ((EObject)_elvis);
          SymbolTableEntry _switchResult = null;
          boolean _matched = false;
          if (varSource instanceof IdentifierRef_IM) {
            _matched=true;
            _switchResult = ((IdentifierRef_IM)varSource).getId_IM();
          }
          if (!_matched) {
            if (varSource instanceof VariableDeclaration) {
              _matched=true;
              _switchResult = this.findSymbolTableEntryForElement(((NamedElement)varSource), true);
            }
          }
          final SymbolTableEntry varSTE = _switchResult;
          Pair<SymbolTableEntry, Expression> _mappedTo_3 = Pair.<SymbolTableEntry, Expression>of(varSTE, currValue);
          simpleAssignments.add(_mappedTo_3);
        } else {
          if (((currNode.getNestedNodes() != null) && (!((List<DestructNode>)Conversions.doWrapArray(currNode.getNestedNodes())).isEmpty()))) {
            nestedPatternsCounter++;
            this.traverse(helperVars, simpleAssignments, currNode.getNestedNodes(), currValue, null, ((helperVarSuffix + "$") + Integer.valueOf(nestedPatternsCounter)));
          } else {
          }
        }
      }
    }
  }
  
  private static final int getNestingDepth(final ForStatement stmnt) {
    int d = 0;
    EObject obj = stmnt;
    while (((obj = obj.eContainer()) != null)) {
      if ((obj instanceof ForStatement)) {
        d++;
      }
    }
    return d;
  }
  
  /**
   * Returns a list of statements that are the root statements of the next outer variable environment directly
   * containing the given AST node.
   */
  public static EList<? super Statement> getContainingVariableEnvironmentContent(final EObject astNode) {
    final VariableEnvironmentElement vee = EcoreUtil2.<VariableEnvironmentElement>getContainerOfType(astNode.eContainer(), VariableEnvironmentElement.class);
    if ((vee == null)) {
      throw new IllegalArgumentException("given AST node does not have an outer variable environment");
    }
    EList<? super Statement> _switchResult = null;
    boolean _matched = false;
    if (vee instanceof Script) {
      _matched=true;
      _switchResult = ((Script)vee).getScriptElements();
    }
    if (!_matched) {
      if (vee instanceof FunctionOrFieldAccessor) {
        _matched=true;
        _switchResult = ((FunctionOrFieldAccessor)vee).getBody().getStatements();
      }
    }
    if (!_matched) {
      if (vee instanceof CatchBlock) {
        _matched=true;
        _switchResult = ((CatchBlock)vee).getBlock().getStatements();
      }
    }
    if (!_matched) {
      if (vee instanceof PropertyAssignment) {
        _matched=true;
        _switchResult = DestructuringTransformation.getContainingVariableEnvironmentContent(vee);
      }
    }
    if (!_matched) {
      if (vee instanceof WithStatement) {
        _matched=true;
        EList<Statement> _xblockexpression = null;
        {
          Statement _statement = ((WithStatement)vee).getStatement();
          boolean _not = (!(_statement instanceof Block));
          if (_not) {
            ((WithStatement)vee).setStatement(TranspilerBuilderBlocks._Block(((WithStatement)vee).getStatement()));
          }
          Statement _statement_1 = ((WithStatement)vee).getStatement();
          _xblockexpression = ((Block) _statement_1).getStatements();
        }
        _switchResult = _xblockexpression;
      }
    }
    return _switchResult;
  }
  
  private static VariableDeclaration getVariableDeclarationFromSTE(final SymbolTableEntry ste) {
    return IterableExtensions.<VariableDeclaration>head(Iterables.<VariableDeclaration>filter(ste.getElementsOfThisName(), VariableDeclaration.class));
  }
}
