/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.SuperLiteral;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.assistants.TypeAssistant;
import org.eclipse.n4js.transpiler.es.assistants.DelegationAssistant;
import org.eclipse.n4js.transpiler.es.transform.ClassDeclarationTransformation;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Dependencies:
 * <ul>
 * <li>excludesBefore {@link ClassDeclarationTransformation}:
 *     this transformation relies on N4MemberDeclarations which are removed by ClassDeclarationTransformation.
 * </ul>
 */
@TransformationDependency.ExcludesBefore(ClassDeclarationTransformation.class)
@SuppressWarnings("all")
public class SuperLiteralTransformation extends Transformation {
  @Inject
  private TypeAssistant typeAssistant;
  
  @Inject
  private DelegationAssistant delegationAssistant;
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<N4MemberDeclaration> _function = (N4MemberDeclaration it) -> {
      this.transformSuper(it);
    };
    this.<N4MemberDeclaration>collectNodes(this.getState().im, N4MemberDeclaration.class, false).forEach(_function);
  }
  
  private void transformSuper(final N4MemberDeclaration memDecl) {
    final N4ClassifierDefinition owner = memDecl.getOwner();
    if ((owner instanceof N4ClassDeclaration)) {
      if ((memDecl instanceof FunctionOrFieldAccessor)) {
        final Block body = ((FunctionOrFieldAccessor)memDecl).getBody();
        if ((body != null)) {
          this.transformSuper(((N4ClassDeclaration)owner), body);
        }
      }
    }
  }
  
  private void transformSuper(final N4ClassDeclaration classDecl, final Block block) {
    final Procedure1<SuperLiteral> _function = (SuperLiteral sl) -> {
      final EObject p = sl.eContainer();
      if ((p instanceof ParameterizedCallExpression)) {
        Expression _target = ((ParameterizedCallExpression)p).getTarget();
        boolean _tripleEquals = (_target == sl);
        if (_tripleEquals) {
          this.transformSuperCall(classDecl, ((ParameterizedCallExpression)p));
        }
      } else {
        if ((p instanceof ParameterizedPropertyAccessExpression_IM)) {
          Expression _target_1 = ((ParameterizedPropertyAccessExpression_IM)p).getTarget();
          boolean _tripleEquals_1 = (_target_1 == sl);
          if (_tripleEquals_1) {
            this.transformSuperAccess(classDecl, ((ParameterizedPropertyAccessExpression_IM)p));
          }
        }
      }
    };
    IteratorExtensions.<SuperLiteral>forEach(Iterators.<SuperLiteral>filter(block.eAllContents(), SuperLiteral.class), _function);
  }
  
  private void transformSuperCall(final N4ClassDeclaration classDecl, final ParameterizedCallExpression callExpr) {
    final SymbolTableEntryOriginal superClassSTE = this.typeAssistant.getSuperClassSTE(classDecl);
    final SymbolTableEntryOriginal prototypeSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.objectType(this.getState().G), "prototype", false, true, true);
    final SymbolTableEntryOriginal constructorSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.objectType(this.getState().G), "constructor", false, false, true);
    final SymbolTableEntryOriginal callSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.functionType(this.getState().G), "call", false, false, true);
    this.setTarget(callExpr, this.__NSSafe_PropertyAccessExpr(superClassSTE, prototypeSTE, constructorSTE, callSTE));
    this.addArgument(callExpr, 0, TranspilerBuilderBlocks._ThisLiteral());
    this.getState().info.markAsExplicitSuperCall(callExpr);
  }
  
  /**
   * This convenience method allows other transformations to extract the arguments from an explicit super call, i.e.
   * from a statement tagged with {@code InformationRegistry$Tag#explicitSuperCall}.
   */
  public static Expression[] getArgumentsFromExplicitSuperCall(final Statement superCallStmnt) {
    if ((superCallStmnt == null)) {
      throw new IllegalArgumentException("superCallStmnt may not be null");
    }
    if ((superCallStmnt instanceof ExpressionStatement)) {
      final Expression expr = ((ExpressionStatement)superCallStmnt).getExpression();
      if ((expr instanceof ParameterizedCallExpression)) {
        final Function1<Argument, Expression> _function = (Argument it) -> {
          return it.getExpression();
        };
        return ((Expression[])Conversions.unwrapArray(IterableExtensions.<Expression>drop(ListExtensions.<Argument, Expression>map(((ParameterizedCallExpression)expr).getArguments(), _function), 1), Expression.class));
      }
    }
    throw new IllegalArgumentException("explicit super call has an unexpected structure");
  }
  
  private void transformSuperAccess(final N4ClassDeclaration classDecl, final ParameterizedPropertyAccessExpression_IM accExpr) {
    SymbolTableEntry _property_IM = accExpr.getProperty_IM();
    final SymbolTableEntryOriginal propertySTE = ((SymbolTableEntryOriginal) _property_IM);
    final N4MemberDeclaration containingFunctionOrAccessor = EcoreUtil2.<N4MemberDeclaration>getContainerOfType(accExpr, N4MemberDeclaration.class);
    final boolean isStatic = ((containingFunctionOrAccessor != null) && containingFunctionOrAccessor.isStatic());
    Expression _xifexpression = null;
    boolean _isMethod = this.isMethod(propertySTE);
    if (_isMethod) {
      _xifexpression = this.delegationAssistant.createAccessToSuperClass(classDecl, isStatic);
    } else {
      _xifexpression = this.delegationAssistant.createAccessToSuperClassBequestingMember(classDecl, isStatic, propertySTE);
    }
    final Expression superTarget = _xifexpression;
    boolean _isMethod_1 = this.isMethod(propertySTE);
    if (_isMethod_1) {
      this.transformSuperMethodAccess(accExpr, superTarget);
    } else {
      if ((this.isGetter(propertySTE) || this.isSetter(propertySTE))) {
        this.transformSuperFieldAccessorAccess(accExpr, superTarget, propertySTE);
      } else {
        throw new IllegalStateException("super member access to a SymbolTableEntry that could not be identified as a method or field accessor");
      }
    }
  }
  
  private void transformSuperMethodAccess(final ParameterizedPropertyAccessExpression_IM accExpr, final Expression superTarget) {
    this.setTarget(accExpr, superTarget);
    final EObject parent = accExpr.eContainer();
    if ((parent instanceof ParameterizedCallExpression)) {
      final SymbolTableEntryOriginal callSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.functionType(this.getState().G), "call", false, false, true);
      this.setTarget(((ParameterizedCallExpression)parent), TranspilerBuilderBlocks._PropertyAccessExpr(accExpr, callSTE));
      this.addArgument(((ParameterizedCallExpression)parent), 0, TranspilerBuilderBlocks._ThisLiteral());
    }
  }
  
  private void transformSuperFieldAccessorAccess(final ParameterizedPropertyAccessExpression_IM accExpr, final Expression superTarget, final SymbolTableEntryOriginal propertySTE) {
    final boolean isSetter = this.isSetter(propertySTE);
    final SymbolTableEntryOriginal ObjectSTE = this.getSymbolTableEntryOriginal(RuleEnvironmentExtensions.objectType(this.getState().G), true);
    final SymbolTableEntryOriginal getOwnPropertyDescriptorSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.objectType(this.getState().G), "getOwnPropertyDescriptor", false, true, true);
    final SymbolTableEntryOriginal callSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.functionType(this.getState().G), "call", false, false, true);
    ParameterizedCallExpression __CallExpr = TranspilerBuilderBlocks._CallExpr(
      TranspilerBuilderBlocks._PropertyAccessExpr(ObjectSTE, getOwnPropertyDescriptorSTE), superTarget, 
      TranspilerBuilderBlocks._StringLiteralForSTE(propertySTE));
    SymbolTableEntryInternal _xifexpression = null;
    if (isSetter) {
      _xifexpression = this.steFor_set();
    } else {
      _xifexpression = this.steFor_get();
    }
    final ParameterizedCallExpression replacement = TranspilerBuilderBlocks._CallExpr(
      TranspilerBuilderBlocks._PropertyAccessExpr(__CallExpr, _xifexpression, callSTE), 
      TranspilerBuilderBlocks._ThisLiteral());
    if (isSetter) {
      EObject _eContainer = accExpr.eContainer();
      final AssignmentExpression parent = ((AssignmentExpression) _eContainer);
      final Expression value = parent.getRhs();
      EList<Argument> _arguments = replacement.getArguments();
      Argument __Argument = TranspilerBuilderBlocks._Argument(value);
      _arguments.add(__Argument);
      this.replace(parent, replacement);
    } else {
      this.replace(accExpr, replacement);
    }
  }
  
  private boolean isMethod(final SymbolTableEntryOriginal ste) {
    return ((ste.getOriginalTarget() instanceof TMethod) || IterableExtensions.<NamedElement>exists(ste.getElementsOfThisName(), ((Function1<NamedElement, Boolean>) (NamedElement it) -> {
      return Boolean.valueOf((it instanceof N4MethodDeclaration));
    })));
  }
  
  private boolean isGetter(final SymbolTableEntryOriginal ste) {
    return ((ste.getOriginalTarget() instanceof TGetter) || IterableExtensions.<NamedElement>exists(ste.getElementsOfThisName(), ((Function1<NamedElement, Boolean>) (NamedElement it) -> {
      return Boolean.valueOf((it instanceof N4GetterDeclaration));
    })));
  }
  
  private boolean isSetter(final SymbolTableEntryOriginal ste) {
    return ((ste.getOriginalTarget() instanceof TSetter) || IterableExtensions.<NamedElement>exists(ste.getElementsOfThisName(), ((Function1<NamedElement, Boolean>) (NamedElement it) -> {
      return Boolean.valueOf((it instanceof N4SetterDeclaration));
    })));
  }
}
