/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.operations;

import com.google.common.base.Objects;
import java.util.Collection;
import java.util.Collections;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.NamedElement;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.transpiler.PreparationStep;
import org.eclipse.n4js.transpiler.TranspilerState;
import org.eclipse.n4js.transpiler.im.ImFactory;
import org.eclipse.n4js.transpiler.im.ImPackage;
import org.eclipse.n4js.transpiler.im.ReferencingElement_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryIMOnly;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.transpiler.im.VersionedNamedImportSpecifier_IM;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.NameAndAccess;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;

@SuppressWarnings("all")
public class SymbolTableManagement {
  /**
   * Create a symbol table entry for a given original target (either a TModule element OR a variable in the original
   * AST, in case of non-exported top-level variables, local variables, formal parameters, etc.).
   */
  public static SymbolTableEntryOriginal createSymbolTableEntryOriginal(final TranspilerState state, final IdentifiableElement originalTarget) {
    if ((originalTarget == null)) {
      throw new IllegalArgumentException("original target may not be null");
    }
    final SymbolTableEntryOriginal newEntry = ImFactory.eINSTANCE.createSymbolTableEntryOriginal();
    newEntry.setName(originalTarget.getName());
    newEntry.setOriginalTarget(originalTarget);
    if ((originalTarget instanceof NamedElement)) {
      EList<NamedElement> _elementsOfThisName = newEntry.getElementsOfThisName();
      _elementsOfThisName.add(((NamedElement) originalTarget));
    }
    SymbolTableManagement.addOriginal(state, newEntry);
    return newEntry;
  }
  
  /**
   * add a {@link SymbolTableEntryOriginal}
   */
  public static void addOriginal(final TranspilerState state, final SymbolTableEntryOriginal steOriginal) {
    SymbolTableManagement.addOriginal(state.steCache, steOriginal);
  }
  
  /**
   * NOTE: Internal usage in preparation step, please call {@link #addOriginal(TranspilerState,SymbolTableEntryOriginal)}
   */
  public static void addOriginal(final TranspilerState.STECache steCache, final SymbolTableEntryOriginal steOriginal) {
    final SymbolTableEntryOriginal old = steCache.mapOriginal.put(steOriginal.getOriginalTarget(), steOriginal);
    if ((old != null)) {
      throw new IllegalStateException(
        ("It is not allowed to register more then one STEOriginal for the same original Target. Already had: " + old));
    }
    steCache.im.getSymbolTable().getEntries().add(steOriginal);
    SymbolTableManagement.inverseMap(steCache, steOriginal);
  }
  
  private static void inverseMap(final TranspilerState.STECache steManager, final SymbolTableEntryOriginal steOriginal) {
    final Consumer<NamedElement> _function = (NamedElement ele) -> {
      steManager.mapNamedElement_2_STE.put(ele, steOriginal);
    };
    steOriginal.getElementsOfThisName().forEach(_function);
  }
  
  /**
   * Create a symbol table entry for an element in the intermediate model. This should only be used if the element
   * in the IM does <b>not</b> have a corresponding original target (either a TModule element or an element
   * in the original AST, in case of non-exported variables), for example because it was newly created by an AST
   * transformation.
   */
  public static SymbolTableEntryIMOnly createSymbolTableEntryIMOnly(final TranspilerState state, final NamedElement elementInIM) {
    if ((elementInIM == null)) {
      throw new IllegalArgumentException("element in intermediate model may not be null");
    }
    final SymbolTableEntryIMOnly newEntry = ImFactory.eINSTANCE.createSymbolTableEntryIMOnly();
    newEntry.setName(elementInIM.getName());
    EList<NamedElement> _elementsOfThisName = newEntry.getElementsOfThisName();
    _elementsOfThisName.add(elementInIM);
    SymbolTableManagement.addIMOnly(state, newEntry);
    return newEntry;
  }
  
  /**
   * Create an <em>internal</em> symbol table entry. They are special and should be used only in rare exception cases.
   * See {@link SymbolTableEntryInternal} for details.
   */
  public static SymbolTableEntryInternal createSymbolTableEntryInternal(final TranspilerState state, final String name) {
    if ((name == null)) {
      throw new IllegalArgumentException("name may not be null");
    }
    final SymbolTableEntryInternal newEntry = ImFactory.eINSTANCE.createSymbolTableEntryInternal();
    newEntry.setName(name);
    SymbolTableManagement.addInteral(state, newEntry);
    return newEntry;
  }
  
  /**
   * add a {@link SymbolTableEntryInternal}
   */
  private static void addInteral(final TranspilerState state, final SymbolTableEntryInternal ste) {
    final SymbolTableEntryInternal old = state.steCache.mapInternal.put(ste.getName(), ste);
    if ((old != null)) {
      throw new IllegalStateException(
        ("It is not allowed to put the same SymbolTableEntryInternal twice into the Symboltable " + old));
    }
    state.im.getSymbolTable().getEntries().add(ste);
  }
  
  /**
   * Search an STE by original target and create it if not found.
   */
  public static SymbolTableEntryOriginal getSymbolTableEntryOriginal(final TranspilerState state, final IdentifiableElement originalTarget, final boolean create) {
    if ((originalTarget == null)) {
      throw new IllegalArgumentException("original target may not be null");
    }
    final SymbolTableEntryOriginal existingEntry = SymbolTableManagement.getSteOriginal(state, originalTarget);
    if ((existingEntry != null)) {
      return existingEntry;
    }
    if (create) {
      return SymbolTableManagement.createSymbolTableEntryOriginal(state, originalTarget);
    }
    return null;
  }
  
  /**
   * Convenience method for {@link #getSymbolTableEntryOriginal(TranspilerState, IdentifiableElement, boolean},
   * allowing to retrieve the member by name and access from its parent classifier.
   */
  public static SymbolTableEntryOriginal getSymbolTableEntryForMember(final TranspilerState state, final TClassifier type, final String memberName, final boolean writeAccess, final boolean staticAccess, final boolean create) {
    if ((((type == null) || (memberName == null)) || memberName.isEmpty())) {
      throw new IllegalArgumentException("type may not be null and memberName may not be null or empty");
    }
    final TMember m = type.findOwnedMember(memberName, writeAccess, staticAccess);
    if ((m == null)) {
      final NameAndAccess nameAndAccess = new NameAndAccess(memberName, writeAccess, staticAccess);
      throw new IllegalArgumentException(("no such member found in given type: " + nameAndAccess));
    }
    return SymbolTableManagement.getSymbolTableEntryOriginal(state, m, create);
  }
  
  /**
   * Search an internal STE by name and create it if not found.
   */
  public static SymbolTableEntryInternal getSymbolTableEntryInternal(final TranspilerState state, final String name, final boolean create) {
    if (((name == null) || name.isEmpty())) {
      throw new IllegalArgumentException("name may not be null or empty");
    }
    final SymbolTableEntryInternal existingEntry = SymbolTableManagement.getSteInternal(state, name);
    if ((existingEntry != null)) {
      return existingEntry;
    }
    if (create) {
      return SymbolTableManagement.createSymbolTableEntryInternal(state, name);
    }
    return null;
  }
  
  /**
   * Will look up the STE for the given named element in the IM. If not found and <code>create</code> is set to
   * <code>true</code> a {@code SymbolTableEntryIMOnly} is created, otherwise <code>null</code> is returned.
   * <p>
   * <b>WARNING:</b> during look up it will find both {@link SymbolTableEntryOriginal}s and {@link SymbolTableEntryIMOnly}s,
   * but when creating a new STE, it will always create a {@code SymbolTableEntryIMOnly} which is invalid if there
   * exists an original target for the given <code>elementInIM</code> (then a {@link SymbolTableEntryOriginal} would
   * have to be created)!. In such a case, this method must not be used.<br>
   * Most of the time, this won't be the case and it is safe to use this method, because all
   * {@code SymbolTableEntryOriginal}s will be created up-front during the {@link PreparationStep}; in some special
   * cases, however, a new element is introduced into the IM that actually has an original target (so far, static
   * polyfills are the only case of this).
   */
  public static SymbolTableEntry findSymbolTableEntryForElement(final TranspilerState state, final NamedElement elementInIM, final boolean create) {
    if ((elementInIM == null)) {
      throw new IllegalArgumentException("element in intermediate model may not be null");
    }
    final SymbolTableEntry existingEntry = SymbolTableManagement.byElementsOfThisName(state, elementInIM);
    if ((existingEntry != null)) {
      return existingEntry;
    }
    if (create) {
      return SymbolTableManagement.createSymbolTableEntryIMOnly(state, elementInIM);
    }
    return null;
  }
  
  /**
   * Search STE for the given name space import.
   */
  public static SymbolTableEntryOriginal findSymbolTableEntryForNamespaceImport(final TranspilerState state, final NamespaceImportSpecifier importspec) {
    final Predicate<SymbolTableEntryOriginal> _function = (SymbolTableEntryOriginal it) -> {
      ImportSpecifier _importSpecifier = it.getImportSpecifier();
      return (_importSpecifier == importspec);
    };
    final Predicate<SymbolTableEntryOriginal> _function_1 = (SymbolTableEntryOriginal it) -> {
      IdentifiableElement _originalTarget = it.getOriginalTarget();
      return (_originalTarget instanceof ModuleNamespaceVirtualType);
    };
    return state.steCache.mapOriginal.values().parallelStream().filter(_function).filter(_function_1).findAny().orElse(null);
  }
  
  public static void rewireSymbolTable(final TranspilerState state, final EObject from, final EObject to) {
    if (((!SymbolTableManagement.requiresRewiringOfSymbolTable(from)) && (!SymbolTableManagement.requiresRewiringOfSymbolTable(to)))) {
      return;
    }
    if (((from instanceof ReferencingElement_IM) && (to instanceof ReferencingElement_IM))) {
      final EReference eRefThatMightPointToOriginal = ImPackage.eINSTANCE.getSymbolTableEntry_ReferencingElements();
      final Consumer<SymbolTableEntry> _function = (SymbolTableEntry it) -> {
        SymbolTableManagement.<EObject, EObject>replaceInEReference(it, eRefThatMightPointToOriginal, from, to);
      };
      state.im.getSymbolTable().getEntries().parallelStream().forEach(_function);
    } else {
      if (((from instanceof ImportSpecifier) && (to instanceof ImportSpecifier))) {
        final EReference eRefThatMightPointToOriginal_1 = ImPackage.eINSTANCE.getSymbolTableEntryOriginal_ImportSpecifier();
        final Predicate<SymbolTableEntry> _function_1 = (SymbolTableEntry it) -> {
          return (it instanceof SymbolTableEntryOriginal);
        };
        final Consumer<SymbolTableEntry> _function_2 = (SymbolTableEntry it) -> {
          SymbolTableManagement.<EObject, EObject>replaceInEReference(it, eRefThatMightPointToOriginal_1, from, to);
        };
        state.im.getSymbolTable().getEntries().parallelStream().filter(_function_1).forEach(_function_2);
      } else {
        if (((from instanceof NamedElement) && (to instanceof NamedElement))) {
          final EReference eRefThatMightPointToOriginal_2 = ImPackage.eINSTANCE.getSymbolTableEntry_ElementsOfThisName();
          final SymbolTableEntry steFrom = SymbolTableManagement.byElementsOfThisName(state, ((NamedElement) from));
          if ((steFrom != null)) {
            SymbolTableManagement.<EObject, EObject>replaceInEReference(steFrom, eRefThatMightPointToOriginal_2, from, to);
            SymbolTableManagement.replacedElementOfThisName(state, steFrom, ((NamedElement) from), ((NamedElement) to));
          }
        } else {
          String _name = from.eClass().getName();
          String _plus = ("rewiring symbol table entries from type " + _name);
          String _plus_1 = (_plus + 
            " to type ");
          String _name_1 = to.eClass().getName();
          String _plus_2 = (_plus_1 + _name_1);
          String _plus_3 = (_plus_2 + " is not supported yet");
          throw new IllegalArgumentException(_plus_3);
        }
      }
    }
  }
  
  private static boolean requiresRewiringOfSymbolTable(final EObject obj) {
    return (((obj instanceof ReferencingElement_IM) || (obj instanceof ImportSpecifier)) || (obj instanceof NamedElement));
  }
  
  private static <T extends EObject, TN extends T> void replaceInEReference(final EObject obj, final EReference eRef, final T original, final TN replacement) {
    boolean _isMany = eRef.isMany();
    if (_isMany) {
      Object _eGet = obj.eGet(eRef);
      final EList<? super T> l = ((EList<? super T>) _eGet);
      int _size = l.size();
      ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _size, true);
      for (final Integer idx : _doubleDotLessThan) {
        Object _get = l.get((idx).intValue());
        boolean _tripleEquals = (_get == original);
        if (_tripleEquals) {
          l.set((idx).intValue(), replacement);
        }
      }
    } else {
      Object _eGet_1 = obj.eGet(eRef);
      boolean _tripleEquals_1 = (_eGet_1 == original);
      if (_tripleEquals_1) {
        obj.eSet(eRef, replacement);
      }
    }
  }
  
  /**
   * add a {@link SymbolTableEntryIMOnly}
   */
  public static void addIMOnly(final TranspilerState state, final SymbolTableEntryIMOnly only) {
    int _size = only.getElementsOfThisName().size();
    boolean _tripleNotEquals = (_size != 1);
    if (_tripleNotEquals) {
      int _size_1 = only.getElementsOfThisName().size();
      String _plus = ("got a STEImOnly with elmentsOfThisName != 1 : " + Integer.valueOf(_size_1));
      throw new IllegalArgumentException(_plus);
    }
    final SymbolTableEntry old = state.steCache.mapNamedElement_2_STE.put(only.getElementsOfThisName().get(0), only);
    if ((old != null)) {
      NamedElement _get = only.getElementsOfThisName().get(0);
      String _plus_1 = ("tries to install STEImOnly but already had one for the NamedElmeent = " + _get);
      throw new IllegalStateException(_plus_1);
    }
    state.im.getSymbolTable().getEntries().add(only);
  }
  
  /**
   * lookup a {@link SymbolTableEntryIMOnly} associated to an {@link IdentifiableElement}
   */
  public static SymbolTableEntryOriginal getSteOriginal(final TranspilerState state, final IdentifiableElement element) {
    return state.steCache.mapOriginal.get(element);
  }
  
  /**
   * lookup an {@link SymbolTableEntryInternal} based on a plain name ({@link String})
   */
  public static SymbolTableEntryInternal getSteInternal(final TranspilerState state, final String name) {
    return state.steCache.mapInternal.get(name);
  }
  
  /**
   * lookup a {@link SymbolTableEntry} based on a {@link NamedElement} contained in the IM
   */
  public static SymbolTableEntry byElementsOfThisName(final TranspilerState state, final NamedElement elementInIM) {
    final SymbolTableEntry lookup = state.steCache.mapNamedElement_2_STE.get(elementInIM);
    if ((lookup != null)) {
      boolean _contains = lookup.getElementsOfThisName().contains(elementInIM);
      if (_contains) {
        return lookup;
      }
      throw new IllegalStateException(((("Did find STE by NamedElement which is not contained in the list STE.elementsOfThisName. elementInIM=" + elementInIM) + "  found wrong STE=") + lookup));
    }
    return null;
  }
  
  /**
   * Update data structure for NamedElements after the list of {@link SymbolTableEntry#getElementsOfThisName()} of
   * {@code entry} has been modified
   * 
   * @param entry
   *            the updated STE (wherein elmentsOfThisName has been modified to contain {@code to} instead of
   *            {@code from}
   * @param from
   *            old NamedElement
   * @param to
   *            new NamedElement
   */
  public static void replacedElementOfThisName(final TranspilerState state, final SymbolTableEntry entry, final NamedElement from, final NamedElement to) {
    final SymbolTableEntry steRegisteredWithFrom = state.steCache.mapNamedElement_2_STE.get(from);
    boolean _notEquals = (!Objects.equal(steRegisteredWithFrom, entry));
    if (_notEquals) {
      throw new IllegalArgumentException(
        (((((("This method must be called directly after the replacement and only once." + "Expected from=") + from) + " to be related to entry=") + entry) + " in mapNamedElement_2_STE but found: ") + steRegisteredWithFrom));
    }
    state.steCache.mapNamedElement_2_STE.remove(from);
    state.steCache.mapNamedElement_2_STE.put(to, entry);
  }
  
  public static SymbolTableEntryOriginal findSymbolTableEntryForNamedImport(final TranspilerState state, final NamedImportSpecifier importspec) {
    final Predicate<SymbolTableEntryOriginal> _function = (SymbolTableEntryOriginal it) -> {
      ImportSpecifier _importSpecifier = it.getImportSpecifier();
      return Objects.equal(_importSpecifier, importspec);
    };
    return state.steCache.mapOriginal.values().parallelStream().filter(_function).findAny().orElse(null);
  }
  
  /**
   * Finds and returns all STEs that hold a reference to the given {@link VersionedNamedImportSpecifier_IM}
   * 
   * In case of the import of an unversioned type, this method defaults
   * to {@link SymbolTableManagement.findSymbolTableEntryForNamedImport(TranspilerState, NamedImportSpecifier)}.
   */
  public static Collection<SymbolTableEntryOriginal> findSymbolTableEntriesForVersionedTypeImport(final TranspilerState state, final VersionedNamedImportSpecifier_IM importspec) {
    boolean _isVersionedTypeImport = importspec.isVersionedTypeImport();
    boolean _not = (!_isVersionedTypeImport);
    if (_not) {
      SymbolTableEntryOriginal _findSymbolTableEntryForNamedImport = SymbolTableManagement.findSymbolTableEntryForNamedImport(state, importspec);
      return Collections.<SymbolTableEntryOriginal>unmodifiableList(CollectionLiterals.<SymbolTableEntryOriginal>newArrayList(_findSymbolTableEntryForNamedImport));
    }
    final Predicate<SymbolTableEntryOriginal> _function = (SymbolTableEntryOriginal it) -> {
      ImportSpecifier _importSpecifier = it.getImportSpecifier();
      return Objects.equal(_importSpecifier, importspec);
    };
    return state.steCache.mapOriginal.values().parallelStream().unordered().filter(_function).distinct().limit(importspec.getImportedTypeVersions().size()).collect(Collectors.<SymbolTableEntryOriginal>toList());
  }
  
  public static void rename(final TranspilerState state, final SymbolTableEntry entry, final String name) {
    if ((entry instanceof SymbolTableEntryInternal)) {
      throw new UnsupportedOperationException(("cannot rename internal STEs " + entry));
    } else {
      if ((entry instanceof SymbolTableEntryIMOnly)) {
        ((SymbolTableEntryIMOnly)entry).setName(name);
      } else {
        if ((entry instanceof SymbolTableEntryOriginal)) {
          ((SymbolTableEntryOriginal)entry).setName(name);
          ImportSpecifier _importSpecifier = ((SymbolTableEntryOriginal)entry).getImportSpecifier();
          boolean _tripleNotEquals = (_importSpecifier != null);
          if (_tripleNotEquals) {
            throw new UnsupportedOperationException(
              "renaming of symbol table entries not tested yet for imported elements!");
          }
        } else {
          throw new UnsupportedOperationException(
            ("Rename request for SymboltableEntries of unkown type : " + entry));
        }
      }
    }
  }
}
