/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.HashSet;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.function.Consumer;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.xtend.lib.macro.AbstractClassProcessor;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtend.lib.macro.declaration.AnnotationReference;
import org.eclipse.xtend.lib.macro.declaration.MutableClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableFieldDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableMethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.Type;
import org.eclipse.xtend.lib.macro.declaration.TypeReference;
import org.eclipse.xtend.lib.macro.declaration.Visibility;
import org.eclipse.xtend.lib.macro.file.Path;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IntegerRange;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.eclipse.xtext.xbase.lib.util.ReflectExtensions;

/**
 * See annotation NLS
 */
@SuppressWarnings("all")
public abstract class AbstractNLSProcessor extends AbstractClassProcessor {
  @Extension
  protected ReflectExtensions reflExt = new ReflectExtensions();
  
  protected final static String BUNDLE_NAME_FIELD = "BUNDLE_NAME";
  
  protected final static String RESOURCE_BUNDLE_FIELD = "RESOURCE_BUNDLE";
  
  protected final static String nlsClass = "org.eclipse.osgi.util.NLS";
  
  protected abstract Class<?> getAnnotationType();
  
  public String getAnnotationName() {
    return this.getAnnotationType().getSimpleName();
  }
  
  @Override
  public void doTransform(final MutableClassDeclaration annotatedClass, @Extension final TransformationContext context) {
    final AnnotationReference nlsAnnotation = this.getNLSAnnotation(annotatedClass, context);
    if ((nlsAnnotation == null)) {
      throw new NullPointerException("Unable to retrieve the annotation");
    }
    Type _findTypeGlobally = context.findTypeGlobally(AbstractNLSProcessor.nlsClass);
    boolean _tripleEquals = (_findTypeGlobally == null);
    if (_tripleEquals) {
      context.addError(nlsAnnotation, (AbstractNLSProcessor.nlsClass + " isn\'t on the classpath."));
    }
    final String propertyFileNameValue = this.getNLSAnnotationPropertyValue(nlsAnnotation, context);
    final InputStream propertiesFileInputStream = this.getPropertiesFile(annotatedClass, context, propertyFileNameValue, nlsAnnotation);
    final Properties properties = this.loadPropertiesFile(propertiesFileInputStream, context, nlsAnnotation);
    this.addMembers(annotatedClass, nlsAnnotation, context, propertyFileNameValue);
    final Function1<Map.Entry<Object, Object>, String> _function = (Map.Entry<Object, Object> it) -> {
      return String.valueOf(it.getKey());
    };
    final Consumer<Map.Entry<Object, Object>> _function_1 = (Map.Entry<Object, Object> it) -> {
      this.addField(it, annotatedClass, nlsAnnotation, context);
      this.addMethod(it, annotatedClass, nlsAnnotation, context);
    };
    IterableExtensions.<Map.Entry<Object, Object>, String>sortBy(properties.entrySet(), _function).forEach(_function_1);
  }
  
  /**
   * May be overridden by sub classes to add or remove members which are to be created.
   */
  protected void addMembers(final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, final TransformationContext context, final String propertyFileNameValue) {
    this.addBundleNameField(annotatedClass, nlsAnnotation, context, propertyFileNameValue);
    this.addResourceBundleField(annotatedClass, nlsAnnotation, context);
    this.addStaticBlock(annotatedClass, nlsAnnotation, context);
    this.addGetStringMethod(annotatedClass, nlsAnnotation, context);
  }
  
  protected void addStaticBlock(final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    final String fieldName = "INITIALIZER";
    this.checkForExistentField(annotatedClass, fieldName, context, nlsAnnotation);
    final Procedure1<MutableFieldDeclaration> _function = (MutableFieldDeclaration it) -> {
      it.setVisibility(Visibility.PRIVATE);
      it.setStatic(true);
      it.setFinal(true);
      it.setType(context.getString());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("new ");
          _builder.append(Function0.class);
          _builder.append("<");
          _builder.append(String.class);
          _builder.append(">() {");
          _builder.newLineIfNotEmpty();
          _builder.append("    ");
          _builder.append("public ");
          TypeReference _string = context.getString();
          _builder.append(_string, "    ");
          _builder.append(" apply() {");
          _builder.newLineIfNotEmpty();
          _builder.append("      ");
          _builder.append(AbstractNLSProcessor.nlsClass, "      ");
          _builder.append(".initializeMessages(");
          String _simpleName = annotatedClass.findDeclaredField(
            AbstractNLSProcessor.BUNDLE_NAME_FIELD).getSimpleName();
          _builder.append(_simpleName, "      ");
          _builder.append(", ");
          TypeReference _newTypeReference = context.newTypeReference(annotatedClass);
          _builder.append(_newTypeReference, "      ");
          _builder.append(".class);");
          _builder.newLineIfNotEmpty();
          _builder.append("      ");
          _builder.append("return \"\";");
          _builder.newLine();
          _builder.append("    ");
          _builder.append("}");
          _builder.newLine();
          _builder.append("  ");
          _builder.append("}.apply();");
          _builder.newLine();
        }
      };
      it.setInitializer(_client);
    };
    annotatedClass.addField(fieldName, _function);
  }
  
  protected void addBundleNameField(final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context, final String propertyFileName) {
    this.checkForExistentField(annotatedClass, AbstractNLSProcessor.BUNDLE_NAME_FIELD, context, nlsAnnotation);
    final Procedure1<MutableFieldDeclaration> _function = (MutableFieldDeclaration it) -> {
      it.setVisibility(Visibility.PRIVATE);
      it.setStatic(true);
      it.setFinal(true);
      it.setType(context.getString());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          TypeReference _newTypeReference = context.newTypeReference(annotatedClass);
          _builder.append(_newTypeReference);
          _builder.append(".class.getPackage().getName() + \".");
          String _replace = propertyFileName.replace(".properties", "");
          _builder.append(_replace);
          _builder.append("\"");
        }
      };
      it.setInitializer(_client);
    };
    annotatedClass.addField(AbstractNLSProcessor.BUNDLE_NAME_FIELD, _function);
  }
  
  protected void addResourceBundleField(final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    this.checkForExistentField(annotatedClass, AbstractNLSProcessor.RESOURCE_BUNDLE_FIELD, context, nlsAnnotation);
    final Procedure1<MutableFieldDeclaration> _function = (MutableFieldDeclaration it) -> {
      it.setVisibility(Visibility.PRIVATE);
      it.setStatic(true);
      it.setFinal(true);
      it.setType(context.newTypeReference(ResourceBundle.class));
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append(ResourceBundle.class);
          _builder.append(".getBundle(");
          String _simpleName = annotatedClass.findDeclaredField(AbstractNLSProcessor.BUNDLE_NAME_FIELD).getSimpleName();
          _builder.append(_simpleName);
          _builder.append(")");
        }
      };
      it.setInitializer(_client);
    };
    annotatedClass.addField(AbstractNLSProcessor.RESOURCE_BUNDLE_FIELD, _function);
  }
  
  protected void addGetStringMethod(final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    final String methodName = "getString";
    this.checkForExistentMethod(annotatedClass, methodName, context, nlsAnnotation, 1);
    final Procedure1<MutableMethodDeclaration> _function = (MutableMethodDeclaration it) -> {
      it.setVisibility(Visibility.PRIVATE);
      it.setStatic(true);
      it.setReturnType(context.getString());
      it.addParameter("key", context.getString());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("try {");
          _builder.newLine();
          _builder.append("\t");
          TypeReference _string = context.getString();
          _builder.append(_string, "\t");
          _builder.append(" value = ");
          String _simpleName = annotatedClass.findDeclaredField(AbstractNLSProcessor.RESOURCE_BUNDLE_FIELD).getSimpleName();
          _builder.append(_simpleName, "\t");
          _builder.append(".getString(key);");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append("return value;");
          _builder.newLine();
          _builder.append("} catch (");
          _builder.append(MissingResourceException.class);
          _builder.append(" e) {");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append("return \'!\' + key + \'!\';");
          _builder.newLine();
          _builder.append("}");
          _builder.newLine();
        }
      };
      it.setBody(_client);
    };
    annotatedClass.addMethod(methodName, _function);
  }
  
  protected void addField(final Map.Entry<Object, Object> entry, final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    Object _key = entry.getKey();
    final String fieldName = ((String) _key);
    this.checkForExistentField(annotatedClass, fieldName, context, nlsAnnotation);
    final Procedure1<MutableFieldDeclaration> _function = (MutableFieldDeclaration it) -> {
      it.setVisibility(Visibility.PUBLIC);
      it.setStatic(true);
      it.setFinal(true);
      it.setType(context.getString());
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("\"");
          Object _key = entry.getKey();
          _builder.append(_key);
          _builder.append("\"");
        }
      };
      it.setInitializer(_client);
      try {
        final Object delegate = this.reflExt.invoke(it, "getDelegate");
        this.reflExt.invoke(delegate, "setConstant", Boolean.valueOf(true));
      } catch (final Throwable _t) {
        if (_t instanceof NoSuchFieldException) {
          final NoSuchFieldException exc = (NoSuchFieldException)_t;
          throw new RuntimeException(exc);
        } else if (_t instanceof IllegalAccessException) {
          final IllegalAccessException exc_1 = (IllegalAccessException)_t;
          throw new RuntimeException(exc_1);
        } else if (_t instanceof NoSuchMethodException) {
          final NoSuchMethodException exc_2 = (NoSuchMethodException)_t;
          throw new RuntimeException(exc_2);
        } else if (_t instanceof InvocationTargetException) {
          final InvocationTargetException exc_3 = (InvocationTargetException)_t;
          throw new RuntimeException(exc_3);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    };
    annotatedClass.addField(fieldName, _function);
  }
  
  protected void checkForExistentField(final MutableClassDeclaration annotatedClass, final String fieldName, @Extension final TransformationContext context, final AnnotationReference nlsAnnotation) {
    MutableFieldDeclaration _findDeclaredField = annotatedClass.findDeclaredField(fieldName);
    boolean _tripleNotEquals = (_findDeclaredField != null);
    if (_tripleNotEquals) {
      context.addError(nlsAnnotation, (("Field " + fieldName) + " already present in class."));
    }
  }
  
  protected void addMethod(final Map.Entry<Object, Object> entry, final MutableClassDeclaration annotatedClass, final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    Object _value = entry.getValue();
    final String message = ((String) _value);
    final int wildcardCount = this.getWildcardCount(message);
    Iterable<String> _xifexpression = null;
    if ((wildcardCount > 0)) {
      final Function1<Integer, String> _function = (Integer it) -> {
        return ("param" + it);
      };
      _xifexpression = IterableExtensions.<Integer, String>map(new IntegerRange(0, (wildcardCount - 1)), _function);
    } else {
      _xifexpression = CollectionLiterals.<String>newArrayList();
    }
    final Iterable<String> params = _xifexpression;
    Object _key = entry.getKey();
    final String msgMethodName = ("msg" + _key);
    this.checkForExistentMethod(annotatedClass, msgMethodName, context, nlsAnnotation, IterableExtensions.size(params));
    final Procedure1<MutableMethodDeclaration> _function_1 = (MutableMethodDeclaration it) -> {
      it.setVisibility(Visibility.PUBLIC);
      it.setStatic(true);
      it.setReturnType(context.getString());
      final Consumer<String> _function_2 = (String param) -> {
        it.addParameter(param, context.getObject());
      };
      params.forEach(_function_2);
      it.setDocComment(message);
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("return ");
          _builder.append(AbstractNLSProcessor.nlsClass);
          _builder.append(".bind(getString(");
          Object _key = entry.getKey();
          _builder.append(_key);
          _builder.append("), new Object [] { ");
          String _join = IterableExtensions.join(params, ", ");
          _builder.append(_join);
          _builder.append(" });");
        }
      };
      it.setBody(_client);
    };
    annotatedClass.addMethod(msgMethodName, _function_1);
  }
  
  protected void checkForExistentMethod(final MutableClassDeclaration annotatedClass, final String methodName, @Extension final TransformationContext context, final AnnotationReference nlsAnnotation, final int parameterListSize) {
    final MutableMethodDeclaration existentMethod = annotatedClass.findDeclaredMethod(methodName);
    if ((existentMethod != null)) {
      int _size = IterableExtensions.size(existentMethod.getParameters());
      boolean _equals = (_size == parameterListSize);
      if (_equals) {
        context.addError(nlsAnnotation, (((("Method " + methodName) + "/") + Integer.valueOf(parameterListSize)) + " already present in class."));
      }
    }
  }
  
  protected int getWildcardCount(final String unboundMessage) {
    int _xblockexpression = (int) 0;
    {
      final Pattern pattern = Pattern.compile("\\{\\d*\\}");
      final Matcher matcher = pattern.matcher(unboundMessage);
      final HashSet<String> matches = CollectionLiterals.<String>newHashSet();
      while (matcher.find()) {
        {
          final MatchResult matchResult = matcher.toMatchResult();
          matches.add(matchResult.group());
        }
      }
      _xblockexpression = matches.size();
    }
    return _xblockexpression;
  }
  
  private AnnotationReference getNLSAnnotation(final MutableClassDeclaration annotatedClass, @Extension final TransformationContext context) {
    return annotatedClass.findAnnotation(context.newTypeReference(this.getAnnotationType()).getType());
  }
  
  private String getNLSAnnotationPropertyValue(final AnnotationReference nlsAnnotation, @Extension final TransformationContext context) {
    String _xblockexpression = null;
    {
      Object _value = nlsAnnotation.getValue("propertyFileName");
      final String value = ((String) _value);
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(value);
      if (_isNullOrEmpty) {
        String _annotationName = this.getAnnotationName();
        String _plus = (_annotationName + " requires non empty propertyFileName property value.");
        context.addError(nlsAnnotation, _plus);
      }
      _xblockexpression = value;
    }
    return _xblockexpression;
  }
  
  private InputStream getPropertiesFile(final MutableClassDeclaration annotatedClass, @Extension final TransformationContext context, final String propertyFileName, final AnnotationReference nlsAnnotation) {
    InputStream _xblockexpression = null;
    {
      Path _filePath = annotatedClass.getCompilationUnit().getFilePath();
      Path _parent = null;
      if (_filePath!=null) {
        _parent=_filePath.getParent();
      }
      final Path folder = _parent;
      if (((folder == null) || (!context.exists(folder)))) {
        String _qualifiedName = annotatedClass.getQualifiedName();
        String _plus = ("Cannot find folder for class " + _qualifiedName);
        String _plus_1 = (_plus + ": ");
        String _plus_2 = (_plus_1 + folder);
        context.addError(nlsAnnotation, _plus_2);
        String _qualifiedName_1 = annotatedClass.getQualifiedName();
        String _plus_3 = ("Cannot find folder for class " + _qualifiedName_1);
        String _plus_4 = (_plus_3 + ": ");
        String _plus_5 = (_plus_4 + folder);
        throw new IllegalArgumentException(_plus_5);
      }
      String _xifexpression = null;
      boolean _endsWith = propertyFileName.endsWith(".properties");
      boolean _not = (!_endsWith);
      if (_not) {
        _xifexpression = ".properties";
      } else {
        _xifexpression = "";
      }
      String _plus_6 = (propertyFileName + _xifexpression);
      final Path propertiesFilePath = folder.append(_plus_6);
      boolean _exists = context.exists(propertiesFilePath);
      boolean _not_1 = (!_exists);
      if (_not_1) {
        String _lastSegment = propertiesFilePath.getLastSegment();
        String _plus_7 = (_lastSegment + " doesn\'t exist.");
        context.addError(nlsAnnotation, _plus_7);
      }
      _xblockexpression = context.getContentsAsStream(propertiesFilePath);
    }
    return _xblockexpression;
  }
  
  private Properties loadPropertiesFile(final InputStream propertiesFile, @Extension final TransformationContext transformationContext, final AnnotationReference nlsAnnotation) {
    Properties _xblockexpression = null;
    {
      final Properties properties = new Properties();
      try {
        properties.load(propertiesFile);
      } catch (final Throwable _t) {
        if (_t instanceof IOException) {
          final IOException ioe = (IOException)_t;
          String _message = ioe.getMessage();
          String _plus = ("Cannot load properties file: " + _message);
          transformationContext.addError(nlsAnnotation, _plus);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      _xblockexpression = properties;
    }
    return _xblockexpression;
  }
}
