/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.util.List;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * A predicate like {@link Function1 function} implementation with logical conjunction capabilities.
 * In other words the current AND function will be evaluated to {@code true} if all wrapped functions
 * are evaluated to {@code true}. Otherwise it will be evaluated to {@code false}.
 */
@SuppressWarnings("all")
public class AndFunction1<F extends Object> implements Function1<F, Boolean> {
  private final List<Function1<? super F, Boolean>> functions;
  
  /**
   * Creates a new function instance that will represent a logical conjunction.
   */
  public static <F extends Object> AndFunction1<F> conjunctionOf(final Function1<? super F, Boolean> first, final Function1<? super F, Boolean>... others) {
    List<Function1<? super F, Boolean>> _asList = Lists.<Function1<? super F, Boolean>>asList(
      Preconditions.<Function1<? super F, Boolean>>checkNotNull(first, "first"), 
      Preconditions.<Function1<? super F, Boolean>[]>checkNotNull(others, "others"));
    return new AndFunction1<F>(_asList);
  }
  
  /**
   * Creates a new logical AND predicate function with the iterable of wrapped functions.
   */
  protected AndFunction1(final Iterable<? extends Function1<? super F, Boolean>> toAdd) {
    this.functions = ImmutableList.<Function1<? super F, Boolean>>builder().addAll(Preconditions.<Iterable<? extends Function1<? super F, Boolean>>>checkNotNull(toAdd, "others")).build();
  }
  
  @Override
  public Boolean apply(final F p) {
    for (final Function1<? super F, Boolean> f : this.functions) {
      Boolean _apply = f.apply(p);
      boolean _not = (!(_apply).booleanValue());
      if (_not) {
        return Boolean.valueOf(false);
      }
    }
    return Boolean.valueOf(true);
  }
  
  /**
   * Returns with a new AND function instance by copying the current instance and appending the argument
   * to to it.
   */
  public AndFunction1<F> and(final Function1<? super F, Boolean>... toAdd) {
    AndFunction1<F> _xifexpression = null;
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(((Iterable<?>)Conversions.doWrapArray(toAdd)));
    if (_isNullOrEmpty) {
      _xifexpression = this;
    } else {
      List<Function1<? super F, Boolean>> _asList = Lists.<Function1<? super F, Boolean>>asList(this, toAdd);
      _xifexpression = new AndFunction1<F>(_asList);
    }
    return _xifexpression;
  }
}
