/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.xpect.config;

import com.google.common.base.Objects;
import java.util.List;
import org.eclipse.n4js.xpect.config.VarDef;
import org.eclipse.xpect.setup.XpectSetupComponent;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Configuration of switches and values affecting the performance of the test-run.
 * This component may have a name (String) on Construction
 * In the Content block you can define variables by stating  {@link VarDef}  components.
 * 
 * <pre>
 * Config {
 *   content_assist_timeout = 1000
 *   VarDef "objectProposals" {
 *     MemberList  "Object" "methods" "public" {}
 *   }
 *   VarDef "keywords" {
 *     StringList {
 *       s="++" s="--" s="abstract" s="break" s="class"
 *     }
 *   }
 * }
 * </pre>
 */
@XpectSetupComponent
@SuppressWarnings("all")
public class Config {
  private List<VarDef> vars = CollectionLiterals.<VarDef>newArrayList();
  
  public String name;
  
  private long contentAssistTimeout = 2000L;
  
  private String content_assist_kind = "n4js";
  
  public Config() {
    this.name = "Default_Ui_Root_Config";
  }
  
  public Config(final String name) {
    this.name = name;
  }
  
  public boolean add(final VarDef v) {
    return this.vars.add(v);
  }
  
  public VarDef getVar(final String sname) {
    final Function1<VarDef, Boolean> _function = (VarDef it) -> {
      return Boolean.valueOf(Objects.equal(it.name, sname));
    };
    return IterableExtensions.<VarDef>findFirst(this.vars, _function);
  }
  
  /**
   * Timeout in milliseconds for the content-assist computation. Defaults to 2000ms.
   * @param timeoutInMsec  timeout value in ms
   */
  public void setContent_assist_timeout(final long timeoutInMsec) {
    this.contentAssistTimeout = timeoutInMsec;
  }
  
  public long getContent_assist_timeout() {
    return this.contentAssistTimeout;
  }
  
  /**
   * Optionally specify the content assist kind.
   * Cycling in content-assist (multiple ctrl+space hits.) changes kinds, to test them we can
   * specify the kind in the test.
   * 
   * Default kind is 'n4js'.
   * 
   * @param kind key String identifying the default content assist kind.
   */
  public void setContent_assist_kind(final String kind) {
    this.content_assist_kind = kind;
  }
  
  public String getContent_assist_kind() {
    return this.content_assist_kind;
  }
  
  @Override
  public String toString() {
    return (((("Config(" + this.name) + ")[") + this.vars) + "]");
  }
}
