/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.antlr;

import com.google.common.base.Charsets;
import com.google.common.io.Files;
import com.google.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import org.eclipse.n4js.antlr.CodeIntoGrammarInjector;
import org.eclipse.n4js.antlr.UnicodeKeywordHelper;
import org.eclipse.n4js.antlr.n4js.AutomaticSemicolonInjector;
import org.eclipse.n4js.antlr.n4js.NoLineTerminatorHandlingInjector;
import org.eclipse.n4js.antlr.n4js.RegExDisambiguationInjector;
import org.eclipse.n4js.antlr.n4js.TemplateLiteralDisambiguationInjector;
import org.eclipse.xtext.Grammar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xtext.generator.model.IXtextGeneratorFileSystemAccess;
import org.eclipse.xtext.xtext.generator.parser.antlr.AntlrGrammarGenerator;
import org.eclipse.xtext.xtext.generator.parser.antlr.AntlrOptions;

/**
 * Customized production ANTLR grammar generator.
 */
@SuppressWarnings("all")
public class N4JSAntlrGrammarGenerator extends AntlrGrammarGenerator {
  @Inject
  private AutomaticSemicolonInjector semicolonInjector;
  
  @Inject
  private RegExDisambiguationInjector regExDisambiguationInjector;
  
  @Inject
  private TemplateLiteralDisambiguationInjector templateLiteralDisambiguationInjector;
  
  @Inject
  private NoLineTerminatorHandlingInjector noLineTerminatorHandlingInjector;
  
  private List<CodeIntoGrammarInjector> steps;
  
  @Inject
  public List<CodeIntoGrammarInjector> initialize() {
    return this.steps = Collections.<CodeIntoGrammarInjector>unmodifiableList(CollectionLiterals.<CodeIntoGrammarInjector>newArrayList(this.semicolonInjector, this.regExDisambiguationInjector, this.templateLiteralDisambiguationInjector, this.noLineTerminatorHandlingInjector));
  }
  
  /**
   * Replace specified extensions with custom implementation for unicode keyword lexer rules
   */
  @Override
  protected String toAntlrKeywordRule(final String keyword, final AntlrOptions options) {
    return UnicodeKeywordHelper.toUnicodeKeyword(keyword);
  }
  
  @Override
  public void generate(final Grammar it, final AntlrOptions options, final IXtextGeneratorFileSystemAccess fsa) {
    super.generate(it, options, fsa);
    this.injectCode(it, fsa);
  }
  
  public void injectCode(final Grammar it, final IXtextGeneratorFileSystemAccess fsa) {
    String _path = fsa.getPath();
    final String outletPath = (_path + "/");
    String _grammarFileName = this.getGrammarNaming().getLexerGrammar(it).getGrammarFileName();
    final String lexerGrammarFileName = (outletPath + _grammarFileName);
    String _grammarFileName_1 = this.getGrammarNaming().getParserGrammar(it).getGrammarFileName();
    final String parserGrammarFileName = (outletPath + _grammarFileName_1);
    this.injectCode(lexerGrammarFileName, parserGrammarFileName);
  }
  
  public void injectCode(final String lexerGrammarFileName, final String parserGrammarFileName) {
    if ((lexerGrammarFileName != null)) {
      try {
        File _file = new File(lexerGrammarFileName);
        String lexerGrammarContent = Files.toString(_file, Charsets.UTF_8);
        lexerGrammarContent = this.processLexerGrammar(lexerGrammarContent);
        File _file_1 = new File(lexerGrammarFileName);
        Files.write(lexerGrammarContent, _file_1, Charsets.UTF_8);
      } catch (final Throwable _t) {
        if (_t instanceof IOException) {
          throw new RuntimeException();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    if ((parserGrammarFileName != null)) {
      try {
        File _file_2 = new File(parserGrammarFileName);
        String parserGrammarContent = Files.toString(_file_2, Charsets.UTF_8);
        parserGrammarContent = this.processParserGrammar(parserGrammarContent);
        File _file_3 = new File(parserGrammarFileName);
        Files.write(parserGrammarContent, _file_3, Charsets.UTF_8);
      } catch (final Throwable _t_1) {
        if (_t_1 instanceof IOException) {
          throw new RuntimeException();
        } else {
          throw Exceptions.sneakyThrow(_t_1);
        }
      }
    }
  }
  
  private String processParserGrammar(final String grammarContent) {
    String result = grammarContent;
    for (final CodeIntoGrammarInjector step : this.steps) {
      result = step.processParserGrammar(result);
    }
    return result;
  }
  
  private String processLexerGrammar(final String grammarContent) {
    String result = grammarContent;
    for (final CodeIntoGrammarInjector step : this.steps) {
      result = step.processLexerGrammar(result);
    }
    return result;
  }
}
