/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.runner.nodejs;

import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Template for generating N4JS ELF code for node, see N4JSDesign document, chap. 17 : Execution, section 17.2 N4JS Execution And
 * Linking File.
 * 
 * This file is responsible for bootstrapping
 * the node engine by configuring symbolic links to dependencies, adding user provided data
 * to the global scope, initializing init modules and invoking node runtime environment.
 * 
 * @param list
 *            of runtime modules to be bootstrapped
 * @param entryPoint
 *            of the code to be executed
 * @param executionData
 *            that is expected by execution module
 * @return elf data in format for used JS engine
 */
@SuppressWarnings("all")
public class NodeBootScriptTemplate {
  /**
   * {@code pathNodeModules} is {@code /absolute/path/to/node_modules/}
   * {@code fileToInvoke} is {@code ./run.js}
   */
  public static String getRunScriptCore(final String pathNodeModules, final String executionData, final List<String> initModules, final String fileToInvoke, final Set<String> scopeNames, final Map<Path, String> path2names) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isEmpty = path2names.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        _builder.append("//link dependencies");
        _builder.newLine();
        _builder.append("const path = require(\'path\')");
        _builder.newLine();
        _builder.append("const fs = require(\'fs\')");
        _builder.newLine();
        _builder.append("const os = require(\"os\");");
        _builder.newLine();
        {
          boolean _hasElements = false;
          for(final String scopeName : scopeNames) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate("\n", "");
            }
            _builder.append("fs.mkdirSync(\'");
            _builder.append(pathNodeModules);
            _builder.append("/");
            _builder.append(scopeName);
            _builder.append("\');");
            _builder.newLineIfNotEmpty();
          }
        }
        {
          Set<Map.Entry<Path, String>> _entrySet = path2names.entrySet();
          boolean _hasElements_1 = false;
          for(final Map.Entry<Path, String> path2name : _entrySet) {
            if (!_hasElements_1) {
              _hasElements_1 = true;
            } else {
              _builder.appendImmediate("\n", "");
            }
            _builder.append("if(!fs.existsSync(\'");
            _builder.append(pathNodeModules);
            _builder.append("/");
            String _value = path2name.getValue();
            _builder.append(_value);
            _builder.append("\'))");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("fs.symlinkSync(\'");
            String _string = path2name.getKey().toString();
            _builder.append(_string, "\t");
            _builder.append("\', \'");
            _builder.append(pathNodeModules, "\t");
            _builder.append("/");
            String _value_1 = path2name.getValue();
            _builder.append(_value_1, "\t");
            _builder.append("\', \'dir\');");
            _builder.newLineIfNotEmpty();
          }
        }
      } else {
        _builder.append("//no dependencies to link");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(executionData);
      boolean _not_1 = (!_isNullOrEmpty);
      if (_not_1) {
        _builder.append("/*");
        _builder.newLine();
        _builder.append(" ");
        _builder.append("* In this form execution module needs to read prop \'$executionData\' from global scope (also would be good idea");
        _builder.newLine();
        _builder.append(" ");
        _builder.append("* to remove it). It would be possible that execution module exports function that takes this data as parameter");
        _builder.newLine();
        _builder.append(" ");
        _builder.append("* but then we need to change order of things in ELF file, that is execution module has to be loaded, its export");
        _builder.newLine();
        _builder.append(" ");
        _builder.append("* function assigned to variable and called with this data below.");
        _builder.newLine();
        _builder.append(" ");
        _builder.append("*");
        _builder.newLine();
        _builder.append(" ");
        _builder.append("* keep it in sync");
        _builder.newLine();
        _builder.append(" ");
        _builder.append("*/");
        _builder.newLine();
        _builder.append("global.$executionData = ");
        _builder.append(executionData);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      } else {
        _builder.append("//no execution data provided");
        _builder.newLine();
      }
    }
    _builder.newLine();
    {
      boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(initModules);
      boolean _not_2 = (!_isNullOrEmpty_1);
      if (_not_2) {
        _builder.append("g");
        _builder.newLineIfNotEmpty();
        _builder.append("\t\t");
        _builder.append("// invoke init modules");
        _builder.newLine();
        {
          boolean _hasElements_2 = false;
          for(final String initModule : initModules) {
            if (!_hasElements_2) {
              _hasElements_2 = true;
            } else {
              _builder.appendImmediate("\n", "\t\t");
            }
            _builder.append("\t\t");
            _builder.append("require(\'");
            _builder.append(initModule, "\t\t");
            _builder.append("\')");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t\t");
        _builder.append("//no init modules to invoke");
        _builder.newLine();
      }
    }
    _builder.newLine();
    _builder.append("require(\'");
    _builder.append(fileToInvoke);
    _builder.append("\')");
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }
}
