/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.migrations;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.eclipse.n4js.n4idl.migrations.TypeDistanceComputer;
import org.eclipse.n4js.ts.types.TMigration;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * A {@link MigrationMatcher} accumulates all migration candidates handed to its {@link #match}
 * method and keeps track of the best match (and thus distance) matched so far.
 */
@SuppressWarnings("all")
public class MigrationMatcher {
  /**
   * Creates a new empty {@link MigrationMatcher}.
   */
  public static MigrationMatcher emptyMatcher() {
    return new MigrationMatcher();
  }
  
  private double bestDistance;
  
  private Set<TMigration> matches = new HashSet<TMigration>();
  
  /**
   * Initializes a new {@link MigrationMatcher}.
   * 
   * @param match The first migration match or {@code null} for an empty matcher.
   * @param distance The initially best distance (for empty matcher, pass {@link TypeDistanceComputer#MAX_DISTANCE}).
   */
  private MigrationMatcher() {
    this.bestDistance = TypeDistanceComputer.MAX_DISTANCE;
  }
  
  /**
   * Matches the given match and distance with this matcher.
   * 
   * Returns an {@link MigrationMatcher} which also considers the given
   * match.
   */
  public void match(final TMigration match, final double distance) {
    if ((this.bestDistance < distance)) {
    } else {
      if ((this.bestDistance == distance)) {
        this.matches.add(match);
      } else {
        HashSet<TMigration> _hashSet = new HashSet<TMigration>(Collections.<TMigration>unmodifiableList(CollectionLiterals.<TMigration>newArrayList(match)));
        this.matches = _hashSet;
        this.bestDistance = distance;
      }
    }
  }
  
  /**
   * Returns {@code true} iff this matcher has found the
   * perfect match (best distance == 0).
   */
  public boolean hasPerfectMatch() {
    double _distance = this.getDistance();
    return (_distance == 0);
  }
  
  /**
   * Returns the best distance this matcher has matched.
   * 
   * {@link #allMatches} can be assumed to be of this distance.
   */
  public double getDistance() {
    return this.bestDistance;
  }
  
  /**
   * Returns all matches matched by this {@link MigrationMatcher}.
   * 
   * All returned candidates can be assumed to be of the same {@link #distance}.
   */
  public List<TMigration> getAllMatches() {
    return IterableExtensions.<TMigration>toList(this.matches);
  }
  
  /**
   * Returns any match from this matcher or {@code null} if no
   * match has been found yet.
   */
  public TMigration anyMatch() {
    return IterableExtensions.<TMigration>head(this.matches);
  }
  
  /**
   * Returns {@code true} iff this matcher has not matched any migration yet.
   */
  public boolean isEmpty() {
    int _size = this.matches.size();
    return (_size == 0);
  }
}
