/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.ASTProcessor;
import org.eclipse.n4js.postprocessing.AbstractPolyProcessor;
import org.eclipse.n4js.postprocessing.AbstractProcessor;
import org.eclipse.n4js.ts.typeRefs.DeferredTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.InferenceVariable;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.constraints.InferenceContext;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;

/**
 * {@link PolyProcessor} delegates here for processing array literals.
 * 
 * @see PolyProcessor#inferType(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,ASTMetaInfoCache)
 * @see PolyProcessor#processExpr(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,TypeRef,InferenceContext,ASTMetaInfoCache)
 */
@Singleton
@SuppressWarnings("all")
class PolyProcessor_FunctionExpression extends AbstractPolyProcessor {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private ASTProcessor astProcessor;
  
  /**
   * BEFORE CHANGING THIS METHOD, READ THIS:
   * {@link PolyProcessor#processExpr(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,TypeRef,InferenceContext,ASTMetaInfoCache)}
   */
  TypeRef processFunctionExpression(final RuleEnvironment G, final FunctionExpression funExpr, final TypeRef expectedTypeRef, final InferenceContext infCtx, final ASTMetaInfoCache cache) {
    Type _definedType = funExpr.getDefinedType();
    final TFunction fun = ((TFunction) _definedType);
    boolean _isPoly = this.isPoly(funExpr);
    boolean _not = (!_isPoly);
    if (_not) {
      final FunctionTypeExpression funTE = TypeUtils.createFunctionTypeExpression(null, Collections.<TypeVariable>unmodifiableList(CollectionLiterals.<TypeVariable>newArrayList()), fun.getFpars(), fun.getReturnTypeRef());
      return funTE;
    }
    final FunctionTypeExpression funTE_1 = TypeRefsFactory.eINSTANCE.createFunctionTypeExpression();
    TypeRef _declaredThisType = fun.getDeclaredThisType();
    boolean _tripleNotEquals = (_declaredThisType != null);
    if (_tripleNotEquals) {
      funTE_1.setDeclaredThisType(TypeUtils.<TypeRef>copy(fun.getDeclaredThisType()));
    }
    boolean _isEmpty = fun.getTypeVars().isEmpty();
    boolean _not_1 = (!_isEmpty);
    if (_not_1) {
      EList<TypeVariable> _ownedTypeVars = funTE_1.getOwnedTypeVars();
      final Function1<TypeVariable, TypeVariable> _function = (TypeVariable tv) -> {
        return TypeUtils.<TypeVariable>copy(tv);
      };
      List<TypeVariable> _map = ListExtensions.<TypeVariable, TypeVariable>map(fun.getTypeVars(), _function);
      Iterables.<TypeVariable>addAll(_ownedTypeVars, _map);
    }
    this.processFormalParameters(G, cache, infCtx, funExpr, funTE_1, expectedTypeRef);
    this.processReturnType(G, cache, infCtx, funExpr, funTE_1);
    funTE_1.setReturnValueMarkedOptional(((expectedTypeRef instanceof FunctionTypeExprOrRef) && ((FunctionTypeExprOrRef) expectedTypeRef).isReturnValueOptional()));
    FunctionTypeExpression _xifexpression = null;
    boolean _isEmpty_1 = fun.getTypeVars().isEmpty();
    if (_isEmpty_1) {
      _xifexpression = funTE_1;
    } else {
      FunctionTypeExpression _xblockexpression = null;
      {
        final RuleEnvironment Gx = RuleEnvironmentExtensions.newRuleEnvironment(G);
        final Function1<TypeVariable, ParameterizedTypeRef> _function_1 = (TypeVariable it) -> {
          return TypeUtils.createTypeRef(it);
        };
        RuleEnvironmentExtensions.addTypeMappings(Gx, fun.getTypeVars(), ListExtensions.<TypeVariable, ParameterizedTypeRef>map(funTE_1.getOwnedTypeVars(), _function_1));
        TypeRef _substTypeVariables = this.ts.substTypeVariables(Gx, funTE_1);
        _xblockexpression = ((FunctionTypeExpression) _substTypeVariables);
      }
      _xifexpression = _xblockexpression;
    }
    final FunctionTypeExpression resultTypeRef = _xifexpression;
    final Consumer<Optional<Map<InferenceVariable, TypeRef>>> _function_1 = (Optional<Map<InferenceVariable, TypeRef>> solution) -> {
      this.handleOnSolved(G, cache, infCtx, funExpr, resultTypeRef, solution);
    };
    infCtx.onSolved(_function_1);
    return resultTypeRef;
  }
  
  /**
   * Process formal parameters and also introduce inference variables for types of fpars, where needed.
   */
  private void processFormalParameters(final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceContext infCtx, final FunctionExpression funExpr, final FunctionTypeExpression funTE, final TypeRef expectedTypeRef) {
    Type _definedType = funExpr.getDefinedType();
    final TFunction fun = ((TFunction) _definedType);
    FunctionTypeExprOrRef _xifexpression = null;
    if ((expectedTypeRef instanceof FunctionTypeExprOrRef)) {
      _xifexpression = ((FunctionTypeExprOrRef)expectedTypeRef);
    }
    final FunctionTypeExprOrRef expectedFunctionTypeExprOrRef = _xifexpression;
    final int len = Math.min(funExpr.getFpars().size(), fun.getFpars().size());
    final Map<FormalParameter, TFormalParameter> typeRefMap = new HashMap<FormalParameter, TFormalParameter>();
    for (int i = 0; (i < len); i++) {
      {
        final FormalParameter fparAST = funExpr.getFpars().get(i);
        final TFormalParameter fparT = fun.getFpars().get(i);
        final TFormalParameter fparTCopy = TypeUtils.<TFormalParameter>copy(fparT);
        EList<TFormalParameter> _fpars = funTE.getFpars();
        _fpars.add(fparTCopy);
        typeRefMap.put(fparAST, fparTCopy);
        TypeRef _declaredTypeRef = fparAST.getDeclaredTypeRef();
        boolean _tripleEquals = (_declaredTypeRef == null);
        if (_tripleEquals) {
          TypeRef _typeRef = null;
          if (fparTCopy!=null) {
            _typeRef=fparTCopy.getTypeRef();
          }
          AbstractProcessor.assertTrueIfRigid(cache, "type of formal parameter in TModule should be a DeferredTypeRef", 
            (_typeRef instanceof DeferredTypeRef));
          final InferenceVariable iv = infCtx.newInferenceVariable();
          fparTCopy.setTypeRef(TypeUtils.createTypeRef(iv));
        }
      }
    }
    Set<Map.Entry<FormalParameter, TFormalParameter>> _entrySet = typeRefMap.entrySet();
    for (final Map.Entry<FormalParameter, TFormalParameter> fparPair : _entrySet) {
      {
        final FormalParameter fparAST = fparPair.getKey();
        final TFormalParameter fparTCopy = fparPair.getValue();
        TypeRef _declaredTypeRef = fparAST.getDeclaredTypeRef();
        boolean _tripleEquals = (_declaredTypeRef == null);
        if (_tripleEquals) {
          Type _declaredType = fparTCopy.getTypeRef().getDeclaredType();
          final InferenceVariable iv = ((InferenceVariable) _declaredType);
          this.addConstraintForDefaultInitializers(funExpr, fparAST, fparTCopy, G, cache, iv, infCtx, typeRefMap);
          this.inferOptionalityFromExpectedFpar(funExpr, funTE, expectedFunctionTypeExprOrRef, fparAST, fparTCopy);
        }
      }
    }
  }
  
  /**
   * When a function expression contains an initializer (in a default parameter),
   * the type of this initializer is taken into account when calculating the parameter's type.
   */
  private void addConstraintForDefaultInitializers(final FunctionExpression funExpr, final FormalParameter fparAST, final TFormalParameter fparT, final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceVariable iv, final InferenceContext infCtx, final Map<FormalParameter, TFormalParameter> typeRefMap) {
    boolean _isHasInitializerAssignment = fparAST.isHasInitializerAssignment();
    if (_isHasInitializerAssignment) {
      EObject _eContainer = fparAST.eContainer();
      final EList<FormalParameter> allFPars = ((FunctionDefinition) _eContainer).getFpars();
      final Expression fparInitializer = fparAST.getInitializer();
      boolean refIsInitializer = false;
      final boolean isPostponed = cache.postponedSubTrees.contains(fparInitializer);
      if ((fparInitializer instanceof IdentifierRef)) {
        final IdentifiableElement id = ((IdentifierRef)fparInitializer).getId();
        refIsInitializer = allFPars.contains(id);
      }
      if (refIsInitializer) {
        final IdentifierRef iRef = ((IdentifierRef) fparInitializer);
        IdentifiableElement _id = iRef.getId();
        final FormalParameter fparam = ((FormalParameter) _id);
        final TFormalParameter fparTCopy = typeRefMap.get(fparam);
        final TypeRef tRef = fparTCopy.getTypeRef();
        infCtx.addConstraint(TypeUtils.createTypeRef(iv), TypeUtils.<TypeRef>copy(tRef), Variance.CONTRA);
      } else {
        if ((!isPostponed)) {
          ParameterizedTypeRef _xifexpression = null;
          EObject _eContainer_1 = fparT.eContainer();
          if ((_eContainer_1 instanceof ContainerType<?>)) {
            EObject _eContainer_2 = fparT.eContainer();
            _xifexpression = TypeUtils.createTypeRef(((ContainerType<?>) _eContainer_2));
          } else {
            _xifexpression = null;
          }
          final ParameterizedTypeRef context = _xifexpression;
          final RuleEnvironment G_withContext = this.ts.createRuleEnvironmentForContext(context, RuleEnvironmentExtensions.getContextResource(G));
          TypeRef _xifexpression_1 = null;
          if ((fparInitializer != null)) {
            _xifexpression_1 = this.ts.type(G_withContext, fparInitializer);
          } else {
            _xifexpression_1 = RuleEnvironmentExtensions.undefinedTypeRef(G);
          }
          final TypeRef iniTypeRef = _xifexpression_1;
          final TypeRef iniTypeRefSubst = this.ts.substTypeVariables(G_withContext, iniTypeRef);
          infCtx.addConstraint(TypeUtils.createTypeRef(iv), TypeUtils.<TypeRef>copy(iniTypeRefSubst), Variance.CONTRA);
        }
      }
    }
  }
  
  /**
   * if the corresponding fpar in the type expectation is optional, we make the fpar in the
   * function expression optional as well
   * Example:
   * 		let fun: {function(string=)} = function(p) {};
   */
  private void inferOptionalityFromExpectedFpar(final FunctionExpression funExpr, final FunctionTypeExpression funTE, final FunctionTypeExprOrRef expectedFunctionTypeExprOrRef, final FormalParameter fparAST, final TFormalParameter fparTCopy) {
    if ((expectedFunctionTypeExprOrRef != null)) {
      final int fparIdx = funExpr.getFpars().indexOf(fparAST);
      final TFormalParameter fparExpected = expectedFunctionTypeExprOrRef.getFparForArgIdx(fparIdx);
      if ((((fparExpected != null) && fparExpected.isOptional()) && (!fparExpected.isVariadic()))) {
        TFormalParameter _last = IterableExtensions.<TFormalParameter>last(funTE.getFpars());
        _last.setHasInitializerAssignment(true);
        final Procedure0 _function = () -> {
          fparAST.setHasInitializerAssignment(true);
          fparTCopy.setHasInitializerAssignment(true);
        };
        EcoreUtilN4.doWithDeliver(false, _function, fparAST, fparTCopy);
      }
    }
  }
  
  /**
   * Processes return type (also introduce inference variable for return type, if needed)
   */
  private void processReturnType(final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceContext infCtx, final FunctionExpression funExpr, final FunctionTypeExpression funTE) {
    Type _definedType = funExpr.getDefinedType();
    final TFunction fun = ((TFunction) _definedType);
    TypeRef returnTypeRef = null;
    TypeRef _returnTypeRef = funExpr.getReturnTypeRef();
    boolean _tripleNotEquals = (_returnTypeRef != null);
    if (_tripleNotEquals) {
      returnTypeRef = TypeUtils.<TypeRef>copy(fun.getReturnTypeRef());
    } else {
      TypeRef _returnTypeRef_1 = fun.getReturnTypeRef();
      AbstractProcessor.assertTrueIfRigid(cache, "return type of TFunction in TModule should be a DeferredTypeRef", 
        (_returnTypeRef_1 instanceof DeferredTypeRef));
      boolean _isReturningValue = this.isReturningValue(funExpr);
      if (_isReturningValue) {
        final InferenceVariable iv = infCtx.newInferenceVariable();
        returnTypeRef = TypeUtils.createTypeRef(iv);
      } else {
        returnTypeRef = RuleEnvironmentExtensions.voidTypeRef(G);
      }
      returnTypeRef = N4JSLanguageUtils.makePromiseIfAsync(funExpr, returnTypeRef, RuleEnvironmentExtensions.getBuiltInTypeScope(G));
      returnTypeRef = N4JSLanguageUtils.makeGeneratorIfGeneratorFunction(funExpr, returnTypeRef, RuleEnvironmentExtensions.getBuiltInTypeScope(G));
    }
    funTE.setReturnTypeRef(returnTypeRef);
  }
  
  /**
   * Writes final types to cache
   */
  private void handleOnSolved(final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceContext infCtx, final FunctionExpression funExpr, final FunctionTypeExpression resultTypeRef, final Optional<Map<InferenceVariable, TypeRef>> solution) {
    Map<InferenceVariable, TypeRef> _xifexpression = null;
    boolean _isPresent = solution.isPresent();
    if (_isPresent) {
      _xifexpression = solution.get();
    } else {
      _xifexpression = this.createPseudoSolution(infCtx, RuleEnvironmentExtensions.anyTypeRef(G));
    }
    final Map<InferenceVariable, TypeRef> solution2 = _xifexpression;
    TypeRef _applySolution = this.applySolution(resultTypeRef, G, solution2);
    final FunctionTypeExprOrRef resultSolved = ((FunctionTypeExprOrRef) _applySolution);
    cache.storeType(funExpr, resultSolved);
    Type _definedType = funExpr.getDefinedType();
    final TFunction fun = ((TFunction) _definedType);
    this.replaceDeferredTypeRefs(fun, resultSolved);
    boolean _isReturnValueMarkedOptional = fun.isReturnValueMarkedOptional();
    boolean _isReturnValueOptional = resultSolved.isReturnValueOptional();
    boolean _tripleNotEquals = (Boolean.valueOf(_isReturnValueMarkedOptional) != Boolean.valueOf(_isReturnValueOptional));
    if (_tripleNotEquals) {
      final Procedure0 _function = () -> {
        fun.setReturnValueMarkedOptional(resultSolved.isReturnValueOptional());
      };
      EcoreUtilN4.doWithDeliver(false, _function, fun);
    }
    final int len = Math.min(funExpr.getFpars().size(), fun.getFpars().size());
    for (int i = 0; (i < len); i++) {
      {
        final FormalParameter fparAST = funExpr.getFpars().get(i);
        final TFormalParameter fparT = fun.getFpars().get(i);
        final TypeRef fparTw = TypeUtils.wrapIfVariadic(RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope, fparT.getTypeRef(), fparAST);
        cache.storeType(fparAST, fparTw);
      }
    }
    if ((funExpr instanceof ArrowFunction)) {
      AbstractProcessor.log(0, "===START of special handling of single-expression arrow function");
      this.tweakReturnTypeOfSingleExpressionArrowFunction(G, cache, ((ArrowFunction)funExpr), resultSolved);
      AbstractProcessor.log(0, "===END of special handling of single-expression arrow function");
    }
  }
  
  /**
   * Handling of a very specific special case of single-expression arrow functions.
   * <p>
   * If the given arrow function is a single-expression arrow function, this method changes the return type of the
   * given function type reference from non-void to void if the non-void return type would lead to a type error later
   * on (for details see code of this method).
   * <p>
   * This tweak is only required because our poor man's return type inferencer in the types builder infers a wrong
   * non-void return type in some cases, which is corrected in this method.
   * <p>
   * Example:
   * <pre>
   * function foo(f : {function(): void}) {}
   * function none(): void {}
   * foo(() => none()); // will show bogus error when disabling this method
   * </pre>
   */
  private void tweakReturnTypeOfSingleExpressionArrowFunction(final RuleEnvironment G, final ASTMetaInfoCache cache, final ArrowFunction arrFun, final FunctionTypeExprOrRef arrFunTypeRef) {
    boolean _isSingleExprImplicitReturn = arrFun.isSingleExprImplicitReturn();
    boolean _not = (!_isSingleExprImplicitReturn);
    if (_not) {
      return;
    }
    final Block block = arrFun.getBody();
    if ((block == null)) {
      return;
    }
    boolean _remove = cache.postponedSubTrees.remove(block);
    boolean _not_1 = (!_remove);
    if (_not_1) {
      URI _uRI = arrFun.eResource().getURI();
      String _plus = ("body of single-expression arrow function not among postponed subtrees, in resource: " + _uRI);
      throw new IllegalStateException(_plus);
    }
    this.astProcessor.processSubtree(G, block, cache, 1);
    boolean didTweakReturnType = false;
    final Expression expr = arrFun.getSingleExpression();
    if ((expr == null)) {
      return;
    }
    final TypeRef exprTypeRef = cache.getType(expr);
    boolean _isVoid = TypeUtils.isVoid(exprTypeRef);
    if (_isVoid) {
      if ((arrFunTypeRef instanceof FunctionTypeExpression)) {
        boolean _isVoid_1 = TypeUtils.isVoid(((FunctionTypeExpression)arrFunTypeRef).getReturnTypeRef());
        boolean _not_2 = (!_isVoid_1);
        if (_not_2) {
          final FunctionTypeExprOrRef outerTypeExpectation = this.expectedTypeForArrowFunction(G, arrFun);
          TypeRef _returnTypeRef = null;
          if (outerTypeExpectation!=null) {
            _returnTypeRef=outerTypeExpectation.getReturnTypeRef();
          }
          final TypeRef outerReturnTypeExpectation = _returnTypeRef;
          if (((outerTypeExpectation == null) || ((outerReturnTypeExpectation != null) && TypeUtils.isVoid(outerReturnTypeExpectation)))) {
            boolean _isDEBUG_LOG = AbstractProcessor.isDEBUG_LOG();
            if (_isDEBUG_LOG) {
              TypeRef _returnTypeRef_1 = ((FunctionTypeExpression)arrFunTypeRef).getReturnTypeRef();
              String _typeRefAsString = null;
              if (_returnTypeRef_1!=null) {
                _typeRefAsString=_returnTypeRef_1.getTypeRefAsString();
              }
              String _plus_1 = ("tweaking return type from " + _typeRefAsString);
              String _plus_2 = (_plus_1 + " to void");
              AbstractProcessor.log(1, _plus_2);
            }
            final Procedure0 _function = () -> {
              ((FunctionTypeExpression)arrFunTypeRef).setReturnTypeRef(RuleEnvironmentExtensions.voidTypeRef(G));
            };
            EcoreUtilN4.doWithDeliver(false, _function, arrFunTypeRef);
            boolean _isDEBUG_LOG_1 = AbstractProcessor.isDEBUG_LOG();
            if (_isDEBUG_LOG_1) {
              String _typeRefAsString_1 = ((FunctionTypeExpression)arrFunTypeRef).getTypeRefAsString();
              String _plus_3 = ("tweaked type of arrow function is: " + _typeRefAsString_1);
              AbstractProcessor.log(1, _plus_3);
            }
            didTweakReturnType = true;
          }
        }
      }
    }
    if ((!didTweakReturnType)) {
      AbstractProcessor.log(1, "tweaking of return type not required");
    }
  }
  
  /**
   * Replaces all DeferredTypeRefs in the given TFunction (i.e. in the fpars' types and the return type)
   * by the corresponding types in 'result'. Argument 'result' must not contain any DeferredTypeRefs and,
   * when this method returns, also the given TFunction 'fun' won't contain DeferredTypeRefs anymore.
   * Will throw exception if 'fun' and 'result' do not match (e.g. 'result' has fewer fpars than 'fun').
   */
  private void replaceDeferredTypeRefs(final TFunction fun, final FunctionTypeExprOrRef result) {
    final int len = ((Object[])Conversions.unwrapArray(fun.getFpars(), Object.class)).length;
    for (int i = 0; (i < len); i++) {
      {
        final TFormalParameter funFpar = fun.getFpars().get(i);
        TypeRef _typeRef = funFpar.getTypeRef();
        if ((_typeRef instanceof DeferredTypeRef)) {
          final int idx = i;
          final Procedure0 _function = () -> {
            funFpar.setTypeRef(TypeUtils.<TypeRef>copy(result.getFpars().get(idx).getTypeRef()));
          };
          EcoreUtilN4.doWithDeliver(false, _function, funFpar);
        }
      }
    }
    TypeRef _returnTypeRef = fun.getReturnTypeRef();
    if ((_returnTypeRef instanceof DeferredTypeRef)) {
      final Procedure0 _function = () -> {
        fun.setReturnTypeRef(TypeUtils.<TypeRef>copy(result.getReturnTypeRef()));
      };
      EcoreUtilN4.doWithDeliver(false, _function, fun);
    }
  }
  
  private FunctionTypeExprOrRef expectedTypeForArrowFunction(final RuleEnvironment G, final ArrowFunction fe) {
    final RuleEnvironment G_new = RuleEnvironmentExtensions.newRuleEnvironment(G);
    final TypeRef tr = this.ts.expectedType(G_new, fe.eContainer(), fe);
    if ((tr instanceof FunctionTypeExprOrRef)) {
      return ((FunctionTypeExprOrRef)tr);
    }
    return null;
  }
}
