/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4idl.versioning.MigrationUtils;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.DeferredTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TMigration;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesbuilder.AbstractFunctionDefinitionTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4IDLMigrationTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSFormalParameterTypesBuilder;
import org.eclipse.n4js.typesbuilder.N4JSTypesBuilderHelper;
import org.eclipse.n4js.typesbuilder.VersionedTypesBuilderUtil;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Type builder for function declaration or expression builder.
 */
@Singleton
@SuppressWarnings("all")
public class N4JSFunctionDefinitionTypesBuilder extends AbstractFunctionDefinitionTypesBuilder {
  @Inject
  @Extension
  private N4JSFormalParameterTypesBuilder _n4JSFormalParameterTypesBuilder;
  
  @Inject
  @Extension
  private N4JSTypesBuilderHelper _n4JSTypesBuilderHelper;
  
  @Inject
  @Extension
  private N4IDLMigrationTypesBuilder _n4IDLMigrationTypesBuilder;
  
  boolean relinkTFunction(final FunctionDeclaration functionDecl, final TModule target, final boolean preLinkingPhase, final int idx) {
    Object _eGet = functionDecl.eGet(N4JSPackage.eINSTANCE.getTypeDefiningElement_DefinedType(), false);
    final EObject functionDefinedType = ((EObject) _eGet);
    if (((functionDefinedType != null) && (!functionDefinedType.eIsProxy()))) {
      throw new IllegalStateException("TFunction already created for FunctionDeclaration");
    }
    String _name = functionDecl.getName();
    boolean _tripleEquals = (_name == null);
    if (_tripleEquals) {
      return false;
    }
    Type _get = target.getTopLevelTypes().get(idx);
    final TFunction functionType = ((TFunction) _get);
    this._n4JSTypesBuilderHelper.ensureEqualName(functionDecl, functionType);
    this.relinkFormalParameters(functionType, functionDecl, preLinkingPhase);
    functionType.setAstElement(functionDecl);
    functionDecl.setDefinedType(functionType);
    return true;
  }
  
  /**
   * Creates TFunction for the given function declaration and adds it to the modules top level types
   * (as function declarations are only allowed on top level).
   * 
   * @param functionDecl declaration for which the TFunction is created, must not be linked to a TFunction yet (i.e. its defined type must be null).
   * @param target the module to which the newly created TFunction is added
   */
  void createTFunction(final FunctionDeclaration functionDecl, final TModule target, final boolean preLinkingPhase) {
    Object _eGet = functionDecl.eGet(N4JSPackage.eINSTANCE.getTypeDefiningElement_DefinedType(), false);
    final EObject functionDefinedType = ((EObject) _eGet);
    if (((functionDefinedType != null) && (!functionDefinedType.eIsProxy()))) {
      throw new IllegalStateException("TFunction already created for FunctionDeclaration");
    }
    String _name = functionDecl.getName();
    boolean _tripleEquals = (_name == null);
    if (_tripleEquals) {
      return;
    }
    final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(functionDecl.eResource().getResourceSet());
    final TFunction functionType = this.createAndLinkTFunction(functionDecl, preLinkingPhase);
    this.addFormalParameters(functionType, functionDecl, builtInTypeScope, preLinkingPhase);
    this._n4JSTypesBuilderHelper.<FunctionDeclaration>setTypeAccessModifier(functionType, functionDecl);
    this._n4JSTypesBuilderHelper.setProvidedByRuntime(functionType, functionDecl, preLinkingPhase);
    this.setReturnType(functionType, functionDecl, builtInTypeScope, preLinkingPhase);
    this.addTypeVariables(functionType, functionDecl, preLinkingPhase);
    this._n4JSTypesBuilderHelper.setDeclaredThisTypeFromAnnotation(functionType, functionDecl, preLinkingPhase);
    this._n4JSTypesBuilderHelper.copyAnnotations(functionType, functionDecl, preLinkingPhase);
    functionType.setDeclaredAsync(functionDecl.isAsync());
    functionType.setDeclaredGenerator(functionDecl.isGenerator());
    VersionedTypesBuilderUtil.setTypeVersion(functionType, functionDecl);
    EList<Type> _topLevelTypes = target.getTopLevelTypes();
    _topLevelTypes.add(functionType);
    boolean _isMigrationDefinition = MigrationUtils.isMigrationDefinition(functionDecl);
    if (_isMigrationDefinition) {
      this._n4IDLMigrationTypesBuilder.initialiseTMigration(functionDecl, ((TMigration) functionType), preLinkingPhase);
    }
  }
  
  /**
   * Creates TFunction for the given function expression and adds it to the module. Note that this method applies
   * only to expressions that define a function, not to function type expressions that merely define a function
   * type (the latter are represented in the AST <em>and</em> TModule by a node of type <code>FunctionTypeExpression</code>
   * from Types.xcore).
   * <p>
   * Creating a TFunction for a function expression becomes a bit tricky when type inference has to be used to
   * infer the types of one or more formal parameters and/or the return value. These are the steps involved:
   * <ol>
   * <li>method {@link #createTFunction(FunctionExpression,TModule,boolean)} creates an initial TFunction in
   *     which the type of every fpar may(!) be a ComputedTypeRef and the return type may(!) be a ComputedTypeRef.
   *     ComputedTypeRefs are only used if there is no declared type available.
   * <li>when the first(!) of these ComputedTypeRefs is resolved (and only for the first!), then method
   *     {@link #resolveTFunction(ComputedTypeRef,TFunction,FunctionExpression,BuiltInTypeScope)}
   *     is invoked. This method will handle the resolution of all ComputedTypeRefs of the given TFunction in
   *     one step (to avoid unnecessary repeated inference of the expected type; note: caching does not help
   *     here, because we call judgment 'expectedTypeIn' and not 'type').
   * </ol>
   */
  void createTFunction(final FunctionExpression functionExpr, final TModule target, final boolean preLinkingPhase) {
    Object _eGet = functionExpr.eGet(N4JSPackage.eINSTANCE.getTypeDefiningElement_DefinedType(), false);
    final EObject functionDefinedType = ((EObject) _eGet);
    if (((functionDefinedType != null) && (!functionDefinedType.eIsProxy()))) {
      throw new IllegalStateException("TFunction already created for FunctionExpression");
    }
    final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(functionExpr.eResource().getResourceSet());
    final TFunction functionType = this.createAndLinkTFunction(functionExpr, preLinkingPhase);
    this.addFormalParametersWithInferredType(functionType, functionExpr, builtInTypeScope, preLinkingPhase);
    this.setReturnTypeWithInferredType(functionType, functionExpr, builtInTypeScope, preLinkingPhase);
    this.addTypeVariables(functionType, functionExpr, preLinkingPhase);
    this._n4JSTypesBuilderHelper.setDeclaredThisTypeFromAnnotation(functionType, functionExpr, preLinkingPhase);
    this._n4JSTypesBuilderHelper.copyAnnotations(functionType, functionExpr, preLinkingPhase);
    EList<Type> _internalTypes = target.getInternalTypes();
    _internalTypes.add(functionType);
  }
  
  /**
   * Same as {@link AbstractFunctionDefinitionTypesBuilder#addFormalParameters(TFunction,FunctionDefinition,BuiltInTypeScope,boolean)},
   * but uses a ComputedTypeRef as the fpar's type if the type has to be inferred.
   */
  private void addFormalParametersWithInferredType(final TFunction functionType, final FunctionExpression functionExpr, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    final Function1<FormalParameter, TFormalParameter> _function = (FormalParameter it) -> {
      return this._n4JSFormalParameterTypesBuilder.createFormalParameter(it, 
        TypeUtils.createDeferredTypeRef(), builtInTypeScope, preLinkingPhase);
    };
    Iterables.<TFormalParameter>addAll(functionType.getFpars(), 
      IterableExtensions.<TFormalParameter>filterNull(ListExtensions.<FormalParameter, TFormalParameter>map(functionExpr.getFpars(), _function)));
  }
  
  /**
   * Same as {@link AbstractFunctionDefinitionTypesBuilder#setReturnType(TFunction,FunctionDefinition,BuiltInTypeScope,boolean)},
   * but uses a ComputedTypeRef as the return type if the type has to be inferred.
   */
  private void setReturnTypeWithInferredType(final TFunction functionType, final FunctionExpression functionExpr, final BuiltInTypeScope builtInTypeScope, final boolean preLinkingPhase) {
    if ((!preLinkingPhase)) {
      TypeRef _elvis = null;
      TypeRef _copyWithProxies = TypeUtils.<TypeRef>copyWithProxies(functionExpr.getReturnTypeRef());
      if (_copyWithProxies != null) {
        _elvis = _copyWithProxies;
      } else {
        DeferredTypeRef _createDeferredTypeRef = TypeUtils.createDeferredTypeRef();
        _elvis = _createDeferredTypeRef;
      }
      functionType.setReturnTypeRef(_elvis);
    }
  }
  
  private TFunction createAndLinkTFunction(final FunctionDefinition functionDef, final boolean preLinkingPhase) {
    final TFunction functionType = this.createTFunction(functionDef);
    if ((functionDef instanceof FunctionDeclaration)) {
      functionType.setExportedName(((FunctionDeclaration)functionDef).getExportedName());
      functionType.setExternal(((FunctionDeclaration)functionDef).isExternal());
    }
    functionType.setName(functionDef.getName());
    functionType.setDeclaredAsync(functionDef.isAsync());
    functionType.setDeclaredGenerator(functionDef.isGenerator());
    functionType.setAstElement(functionDef);
    functionDef.setDefinedType(functionType);
    return functionType;
  }
  
  /**
   * Creates a new plain instance of {@link TFunction} or of the subtype {@link TMigration}.
   * 
   * @see N4IDLMigrationTypesBuilder#isMigrationDeclaration
   */
  private TFunction createTFunction(final FunctionDefinition functionDef) {
    boolean _isMigrationDefinition = MigrationUtils.isMigrationDefinition(functionDef);
    if (_isMigrationDefinition) {
      return this._n4IDLMigrationTypesBuilder.createTMigration();
    } else {
      return TypesFactory.eINSTANCE.createTFunction();
    }
  }
}
