/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesbuilder;

import java.util.Arrays;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.StructuralTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TStructField;
import org.eclipse.n4js.ts.types.TStructGetter;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.TStructMethod;
import org.eclipse.n4js.ts.types.TStructSetter;
import org.eclipse.n4js.ts.types.TStructuralType;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Methods for creating types from TypeRefs are collected in this class.
 * <p>
 * The methods of this class might seem different from other createXYZ() methods in the types
 * builder package, in that they take a subclass of TypeRef and not a typical AST node element.
 * However, SturcturalTypeRefs and FunctionTypeExpressions are among those TypeRefs that
 * <em>may</em> appear in the AST and play the role of an AST node. The methods in this class
 * will only be invoked for such TypeRefs that appear in the AST, so these method are, in fact,
 * very similar to the other createXYZ() methods.
 */
@SuppressWarnings("all")
public class N4JSTypesFromTypeRefBuilder {
  /**
   * Creates a TStructuralType in the target module if the StructuralTypeRef has structural
   * members defined (in the with-clause). For more details why this is required, see API
   * doc of StructuralTypeRef.
   */
  void createStructuralType(final StructuralTypeRef structTypeRef, final TModule target) {
    boolean _isEmpty = structTypeRef.getAstStructuralMembers().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final ResourceSet resSet = structTypeRef.eResource().getResourceSet();
      if ((resSet == null)) {
        throw new IllegalArgumentException("structTypeRef must be contained in AST");
      }
      final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(resSet);
      final TStructuralType structType = TypesFactory.eINSTANCE.createTStructuralType();
      final Function1<TStructMember, TStructMember> _function = (TStructMember currStructMember) -> {
        final TStructMember clone = TypeUtils.<TStructMember>copyWithProxies(currStructMember);
        this.applyDefaults(builtInTypeScope, clone);
        clone.setAstElement(currStructMember);
        currStructMember.setDefinedMember(clone);
        return clone;
      };
      structType.getOwnedMembers().addAll(
        ListExtensions.<TStructMember, TStructMember>map(structTypeRef.getAstStructuralMembers(), _function));
      structTypeRef.setStructuralType(structType);
      EList<Type> _internalTypes = target.getInternalTypes();
      _internalTypes.add(structType);
    }
  }
  
  /**
   * Creates a TFunction in the target module if the FunctionTypeExpression is generic.
   * For more details why this is required, see API doc of FunctionTypeExpression.
   */
  void createTFunction(final FunctionTypeExpression fte, final TModule target) {
    boolean _isGeneric = fte.isGeneric();
    if (_isGeneric) {
      final ResourceSet resSet = fte.eResource().getResourceSet();
      if ((resSet == null)) {
        throw new IllegalArgumentException("fte must be contained in AST");
      }
      final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(resSet);
      final TFunction ft = TypesFactory.eINSTANCE.createTFunction();
      final Function1<TypeVariable, TypeVariable> _function = (TypeVariable currTypeVar) -> {
        return TypeUtils.<TypeVariable>copyWithProxies(currTypeVar);
      };
      ft.getTypeVars().addAll(ListExtensions.<TypeVariable, TypeVariable>map(fte.getTypeVars(), _function));
      final Function1<TFormalParameter, TFormalParameter> _function_1 = (TFormalParameter currFpar) -> {
        final TFormalParameter clone = TypeUtils.<TFormalParameter>copyWithProxies(currFpar);
        this.applyDefaults(builtInTypeScope, clone);
        clone.setAstElement(currFpar);
        return clone;
      };
      ft.getFpars().addAll(
        ListExtensions.<TFormalParameter, TFormalParameter>map(fte.getFpars(), _function_1));
      ft.setReturnTypeRef(TypeUtils.<TypeRef>copyWithProxies(fte.getReturnTypeRef()));
      ft.setDeclaredThisType(TypeUtils.<TypeRef>copyWithProxies(fte.getDeclaredThisType()));
      TypeRef _returnTypeRef = ft.getReturnTypeRef();
      boolean _tripleEquals = (_returnTypeRef == null);
      if (_tripleEquals) {
        ft.setReturnTypeRef(builtInTypeScope.getAnyTypeRef());
      }
      fte.setDeclaredType(ft);
      ft.setAstElement(fte);
      EList<Type> _internalTypes = target.getInternalTypes();
      _internalTypes.add(ft);
    }
  }
  
  private void _applyDefaults(final BuiltInTypeScope builtInTypeScope, final TStructField field) {
    TypeRef _typeRef = field.getTypeRef();
    boolean _tripleEquals = (_typeRef == null);
    if (_tripleEquals) {
      field.setTypeRef(builtInTypeScope.getAnyTypeRef());
    }
  }
  
  private void _applyDefaults(final BuiltInTypeScope builtInTypeScope, final TStructGetter getter) {
    TypeRef _declaredTypeRef = getter.getDeclaredTypeRef();
    boolean _tripleEquals = (_declaredTypeRef == null);
    if (_tripleEquals) {
      getter.setDeclaredTypeRef(builtInTypeScope.getAnyTypeRef());
    }
  }
  
  private void _applyDefaults(final BuiltInTypeScope builtInTypeScope, final TStructSetter setter) {
    TFormalParameter _fpar = setter.getFpar();
    boolean _tripleEquals = (_fpar == null);
    if (_tripleEquals) {
      setter.setFpar(TypesFactory.eINSTANCE.createTAnonymousFormalParameter());
    }
    this.applyDefaults(builtInTypeScope, setter.getFpar());
  }
  
  private void _applyDefaults(final BuiltInTypeScope builtInTypeScope, final TStructMethod method) {
    final Consumer<TFormalParameter> _function = (TFormalParameter it) -> {
      this.applyDefaults(builtInTypeScope, it);
    };
    method.getFpars().forEach(_function);
    TypeRef _returnTypeRef = method.getReturnTypeRef();
    boolean _tripleEquals = (_returnTypeRef == null);
    if (_tripleEquals) {
      method.setReturnTypeRef(builtInTypeScope.getVoidTypeRef());
    }
  }
  
  private void _applyDefaults(final BuiltInTypeScope builtInTypeScope, final TFormalParameter fpar) {
    TypeRef _typeRef = fpar.getTypeRef();
    boolean _tripleEquals = (_typeRef == null);
    if (_tripleEquals) {
      fpar.setTypeRef(builtInTypeScope.getAnyTypeRef());
    }
  }
  
  private void applyDefaults(final BuiltInTypeScope builtInTypeScope, final EObject method) {
    if (method instanceof TStructMethod) {
      _applyDefaults(builtInTypeScope, (TStructMethod)method);
      return;
    } else if (method instanceof TStructGetter) {
      _applyDefaults(builtInTypeScope, (TStructGetter)method);
      return;
    } else if (method instanceof TStructSetter) {
      _applyDefaults(builtInTypeScope, (TStructSetter)method);
      return;
    } else if (method instanceof TStructField) {
      _applyDefaults(builtInTypeScope, (TStructField)method);
      return;
    } else if (method instanceof TFormalParameter) {
      _applyDefaults(builtInTypeScope, (TFormalParameter)method);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(builtInTypeScope, method).toString());
    }
  }
}
