/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.validators.N4JSClassValidator;
import org.eclipse.n4js.validation.validators.N4JSEnumValidator;
import org.eclipse.n4js.validation.validators.N4JSFunctionValidator;
import org.eclipse.n4js.validation.validators.N4JSInterfaceValidator;

/**
 * Collecting special validation logic only related to static polyfill modules.
 * IDE-1735
 */
@SuppressWarnings("all")
public class StaticPolyfillValidatorExtension {
  /**
   * §143 (Restriction on static-polyfilling): §143.1 only classes in staticPolyfillModule allowed.
   */
  public static void internalCheckNotInStaticPolyfillModule(final N4InterfaceDeclaration n4InterfaceDeclaration, final N4JSInterfaceValidator host) {
    boolean _isContainedInStaticPolyfillModule = N4JSLanguageUtils.isContainedInStaticPolyfillModule(n4InterfaceDeclaration);
    if (_isContainedInStaticPolyfillModule) {
      final String msg = IssueCodes.getMessageForPOLY_STATIC_POLYFILL_MODULE_ONLY_FILLING_CLASSES();
      host.addIssue(msg, n4InterfaceDeclaration, N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, 
        IssueCodes.POLY_STATIC_POLYFILL_MODULE_ONLY_FILLING_CLASSES);
    }
  }
  
  /**
   * §143 (Restriction on static-polyfilling): §143.1 only classes in staticPolyfillModule allowed.
   */
  public static void internalCheckNotInStaticPolyfillModule(final N4EnumDeclaration n4EnumDecl, final N4JSEnumValidator host) {
    boolean _isContainedInStaticPolyfillModule = N4JSLanguageUtils.isContainedInStaticPolyfillModule(n4EnumDecl);
    if (_isContainedInStaticPolyfillModule) {
      final String msg = IssueCodes.getMessageForPOLY_STATIC_POLYFILL_MODULE_ONLY_FILLING_CLASSES();
      host.addIssue(msg, n4EnumDecl, N4JSPackage.Literals.N4_TYPE_DECLARATION__NAME, 
        IssueCodes.POLY_STATIC_POLYFILL_MODULE_ONLY_FILLING_CLASSES);
    }
  }
  
  /**
   * §143 (Restriction on static-polyfilling): §143.1 only classes in staticPolyfillModule allowed.
   */
  public static void internalCheckNotInStaticPolyfillModule(final FunctionDeclaration functionDeclaration, final N4JSFunctionValidator host) {
    EObject cont = functionDeclaration.eContainer();
    while ((cont instanceof ExportDeclaration)) {
      cont = ((ExportDeclaration)cont).eContainer();
    }
    if ((cont instanceof Script)) {
      boolean _isContainedInStaticPolyfillModule = N4JSLanguageUtils.isContainedInStaticPolyfillModule(functionDeclaration);
      if (_isContainedInStaticPolyfillModule) {
        final String msg = IssueCodes.getMessageForPOLY_STATIC_POLYFILL_MODULE_ONLY_FILLING_CLASSES();
        host.addIssue(msg, functionDeclaration, N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, 
          IssueCodes.POLY_STATIC_POLYFILL_MODULE_ONLY_FILLING_CLASSES);
      }
    }
  }
  
  /**
   * §143 (Restriction on static-polyfilling): §143.4 P must not implement any interfaces
   */
  public static void internalCheckPolyFilledClassWithAdditionalInterface(final N4ClassDeclaration classDeclaration, final N4JSClassValidator host) {
    boolean _isStaticPolyfill = N4JSLanguageUtils.isStaticPolyfill(classDeclaration);
    if (_isStaticPolyfill) {
      boolean _isEmpty = classDeclaration.getImplementedInterfaceRefs().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final String msg = IssueCodes.getMessageForPOLY_IMPLEMENTING_INTERFACE_NOT_ALLOWED();
        host.addIssue(msg, classDeclaration, N4JSPackage.Literals.N4_CLASS_DEFINITION__IMPLEMENTED_INTERFACE_REFS, 
          IssueCodes.POLY_IMPLEMENTING_INTERFACE_NOT_ALLOWED);
      }
    }
  }
}
