/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.tests.codegen;

import com.google.common.base.Objects;
import org.eclipse.n4js.tests.codegen.Fragment;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * Abstract base class for code generators that generate code for members of a {@link Classifier}.
 */
@SuppressWarnings("all")
public abstract class Member<T extends Member<T>> extends Fragment<T> {
  /**
   * Possible visibilities for members.
   */
  public enum Visibility {
    /**
     * Private visibility.
     */
    PRIVATE,
    
    /**
     * Project visibility.
     */
    PROJECT,
    
    /**
     * Internal protected visibility.
     */
    PROTECTED_INTERNAL,
    
    /**
     * Protected visibility.
     */
    PROTECTED,
    
    /**
     * Internal public visibility.
     */
    PUBLIC_INTERNAL,
    
    /**
     * Public visibility.
     */
    PUBLIC;
  }
  
  /**
   * Extensions for the {@link Visibility} enumeration.
   */
  public static class VisibilityExtensions {
    /**
     * Builds a member name from the given name and visibility by appending an appropriate string
     * to the given name.
     * 
     * @param visibility the visibility value
     * @param the member name prefix
     * 
     * @return the newly created member name
     */
    public static String makeName(final Member.Visibility visibility, final String memberName) {
      String _nameExtension = Member.VisibilityExtensions.getNameExtension(visibility);
      return (memberName + _nameExtension);
    }
    
    /**
     * Returns an appropriate member name extension depending on the given visibility.
     * 
     * @param visibility the visibility value
     * 
     * @return the name extension
     */
    public static String getNameExtension(final Member.Visibility visibility) {
      String _switchResult = null;
      if (visibility != null) {
        switch (visibility) {
          case PRIVATE:
            _switchResult = "_private";
            break;
          case PROJECT:
            _switchResult = "_project";
            break;
          case PROTECTED_INTERNAL:
            _switchResult = "_protected_internal";
            break;
          case PROTECTED:
            _switchResult = "_protected";
            break;
          case PUBLIC_INTERNAL:
            _switchResult = "_public_internal";
            break;
          case PUBLIC:
            _switchResult = "_public";
            break;
          default:
            break;
        }
      }
      return _switchResult;
    }
    
    /**
     * Builds an appropriate code fragment for the given member visibility.
     * 
     * @param visibility the visibility value
     * 
     * @return the code fragment
     */
    public static String generate(final Member.Visibility visibility) {
      String _switchResult = null;
      if (visibility != null) {
        switch (visibility) {
          case PRIVATE:
            _switchResult = "private";
            break;
          case PROJECT:
            _switchResult = "project";
            break;
          case PROTECTED_INTERNAL:
            _switchResult = "@Internal protected";
            break;
          case PROTECTED:
            _switchResult = "protected";
            break;
          case PUBLIC_INTERNAL:
            _switchResult = "@Internal public";
            break;
          case PUBLIC:
            _switchResult = "public";
            break;
          default:
            break;
        }
      }
      return _switchResult;
    }
  }
  
  /**
   * Possible values for whether or not a member is static.
   */
  public enum Static {
    /**
     * Static.
     */
    YES,
    
    /**
     * Non-static.
     */
    NO;
  }
  
  /**
   * Possible values for whether or not a member overrides an inherited member.
   */
  public enum Override {
    YES,
    
    NO;
  }
  
  /**
   * Extensions for {@link Static} enumeration.
   */
  public static class StaticExtensions {
    /**
     * Builds a member name from the given name and static specifier by appending an appropriate string
     * to the given name.
     * 
     * @param static_ whether or not the member is static
     * @param the member name prefix
     * 
     * @return the newly created member name
     */
    public static String makeName(final Member.Static static_, final String classifierName) {
      String _nameExtension = Member.StaticExtensions.getNameExtension(static_);
      return (classifierName + _nameExtension);
    }
    
    /**
     * Returns an appropriate member name extension depending on the given static specification.
     * 
     * @param static_ whether or not the member is static
     * 
     * @return the name extension
     */
    public static String getNameExtension(final Member.Static static_) {
      String _switchResult = null;
      if (static_ != null) {
        switch (static_) {
          case YES:
            _switchResult = "_static";
            break;
          case NO:
            _switchResult = "";
            break;
          default:
            break;
        }
      }
      return _switchResult;
    }
    
    /**
     * Returns an appropriate code fragment depending on the given static specification.
     * 
     * @param static_ whether or not the member is static
     * 
     * @return the code fragment
     */
    public static String generate(final Member.Static static_) {
      String _switchResult = null;
      if (static_ != null) {
        switch (static_) {
          case YES:
            _switchResult = "static ";
            break;
          case NO:
            _switchResult = "";
            break;
          default:
            break;
        }
      }
      return _switchResult;
    }
  }
  
  protected Member.Visibility visibility = Member.Visibility.PUBLIC;
  
  protected Member.Static static_ = Member.Static.NO;
  
  protected String name;
  
  protected Member.Override override_ = Member.Override.NO;
  
  /**
   * Creates a new member with the given parameters.
   * 
   * @param name the name of the member
   */
  protected Member(final String name) {
    this.name = name;
  }
  
  /**
   * Sets visibility to project visible.
   */
  public T makeProjectVisible() {
    return this.setVisibility(Member.Visibility.PROJECT);
  }
  
  /**
   * Sets visibility to internal protected.
   */
  public T makeProtectedInternal() {
    return this.setVisibility(Member.Visibility.PROTECTED_INTERNAL);
  }
  
  /**
   * Sets visibility to protected.
   */
  public T makeProtected() {
    return this.setVisibility(Member.Visibility.PROTECTED);
  }
  
  /**
   * Sets visibility to internal public.
   */
  public T makePublicInternal() {
    return this.setVisibility(Member.Visibility.PUBLIC_INTERNAL);
  }
  
  /**
   * Sets visibility to public.
   */
  public T makePublic() {
    return this.setVisibility(Member.Visibility.PUBLIC);
  }
  
  /**
   * Set the visibility.
   * 
   * @param visibility the visibility to set
   * 
   * @return this builder
   */
  public T setVisibility(final Member.Visibility visibility) {
    this.visibility = visibility;
    return ((T) this);
  }
  
  /**
   * Make the member static.
   * 
   * @return this builder
   */
  public T makeStatic() {
    return this.setStatic(Member.Static.YES);
  }
  
  /**
   * Specify whether the member is static.
   * 
   * @param static_ whether or not the member is static
   */
  public T setStatic(final Member.Static static_) {
    this.static_ = static_;
    return ((T) this);
  }
  
  /**
   * Indicates whether this member is static.
   * 
   * @return <code>true</code> if this member is static and <code>false</code> otherwise
   */
  public boolean isStatic() {
    return Objects.equal(this.static_, Member.Static.YES);
  }
  
  /**
   * Set the member to override.
   */
  public T makeOverride() {
    return this.setOverride(Member.Override.YES);
  }
  
  /**
   * Set the override status of the member.
   * 
   * @param override_ the override status
   */
  public T setOverride(final Member.Override override_) {
    this.override_ = override_;
    return ((T) this);
  }
  
  /**
   * Indicates whether this member overrides.
   * 
   * @return <code>true</code> if this member overrides
   */
  public boolean isOverride() {
    return Objects.equal(this.override_, Member.Override.YES);
  }
  
  @java.lang.Override
  public CharSequence generate() {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _generateOverride = this.generateOverride();
    _builder.append(_generateOverride);
    CharSequence _generateVisibility = this.generateVisibility();
    _builder.append(_generateVisibility);
    String _generateStatic = this.generateStatic();
    _builder.append(_generateStatic);
    CharSequence _generateMember = this.generateMember();
    _builder.append(_generateMember);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  private CharSequence generateOverride() {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isOverride = this.isOverride();
      if (_isOverride) {
        _builder.append("@Override ");
      }
    }
    return _builder;
  }
  
  /**
   * Generates a code fragment for the visibility of this member.
   * 
   * @return the code fragment
   */
  private CharSequence generateVisibility() {
    StringConcatenation _builder = new StringConcatenation();
    String _generate = Member.VisibilityExtensions.generate(this.visibility);
    _builder.append(_generate);
    _builder.append(" ");
    return _builder;
  }
  
  /**
   * Generates a code fragment according to whether this member is static or not.
   * 
   * @return the code fragment
   */
  private String generateStatic() {
    return Member.StaticExtensions.generate(this.static_);
  }
  
  /**
   * Abstract method that generates the actual member code.
   * 
   * @return the generated code fragment
   */
  protected abstract CharSequence generateMember();
  
  @java.lang.Override
  public String toString() {
    return this.generate().toString();
  }
}
