/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.inject.Inject;
import java.util.Arrays;
import java.util.function.Consumer;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.AwaitExpression;
import org.eclipse.n4js.n4JS.CastExpression;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.PromisifyExpression;
import org.eclipse.n4js.n4JS.RelationalExpression;
import org.eclipse.n4js.n4JS.RelationalOperator;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TransformationDependency;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.es.transform.AsyncAwaitTransformation;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.PromisifyHelper;
import org.eclipse.n4js.utils.ResourceNameComputer;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Some expressions need special handling, this is done in this transformation.
 * <p>
 * Dependencies:
 * <ul>
 * <li>ExcludesBefore: AsyncAwaitTransformation<br>
 *     otherwise the 'await' expressions have already be converted to 'yield', but we need to find them in order to
 *     support auto-promisify after 'await'; see method {@link #transformExpression(AwaitExpression)}
 * </ul>
 */
@TransformationDependency.ExcludesBefore(AsyncAwaitTransformation.class)
@SuppressWarnings("all")
public class ExpressionTransformation extends Transformation {
  @Inject
  private ResourceNameComputer resourceNameComputer;
  
  @Inject
  private PromisifyHelper promisifyHelper;
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<Expression> _function = (Expression it) -> {
      this.transformExpression(it);
    };
    this.<Expression>collectNodes(this.getState().im, Expression.class, true).forEach(_function);
  }
  
  private void _transformExpression(final Expression relExpr) {
  }
  
  private void _transformExpression(final CastExpression castExpr) {
    this.replace(castExpr, castExpr.getExpression());
  }
  
  private void _transformExpression(final RelationalExpression relExpr) {
    RelationalOperator _op = relExpr.getOp();
    boolean _tripleEquals = (_op == RelationalOperator.INSTANCEOF);
    if (_tripleEquals) {
      final Expression rhs = relExpr.getRhs();
      IdentifiableElement _xifexpression = null;
      if ((rhs instanceof IdentifierRef_IM)) {
        _xifexpression = ((IdentifierRef_IM)rhs).getOriginalTargetOfRewiredTarget();
      }
      final IdentifiableElement rhsType = _xifexpression;
      ParameterizedCallExpression _xifexpression_1 = null;
      if ((rhsType instanceof TInterface)) {
        ParameterizedCallExpression _xblockexpression = null;
        {
          final SymbolTableEntryInternal $implementsSTE = this.steFor_$implements();
          final String fqn = this.resourceNameComputer.getFullyQualifiedTypeName_WITH_LEGACY_SUPPORT(((Type)rhsType));
          _xblockexpression = TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._IdentRef($implementsSTE), relExpr.getLhs(), TranspilerBuilderBlocks._StringLiteral(fqn));
        }
        _xifexpression_1 = _xblockexpression;
      } else {
        ParameterizedCallExpression _xifexpression_2 = null;
        if ((rhsType instanceof TClass)) {
          _xifexpression_2 = null;
        } else {
          ParameterizedCallExpression _xblockexpression_1 = null;
          {
            final SymbolTableEntryInternal $instanceofSTE = this.steFor_$instanceof();
            _xblockexpression_1 = TranspilerBuilderBlocks._CallExpr(TranspilerBuilderBlocks._IdentRef($instanceofSTE), relExpr.getLhs(), relExpr.getRhs());
          }
          _xifexpression_2 = _xblockexpression_1;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      final ParameterizedCallExpression replacement = _xifexpression_1;
      if ((replacement != null)) {
        this.replace(relExpr, replacement);
      }
    }
  }
  
  /**
   * <b>IMPORTANT:</b>
   * here we only do some special handling for the auto-promisify case within an AwaitExpression; the main
   * handling of the AwaitExpression itself is done in a later transformation {@code BlockTransformation.transformBlockAsync(Block)}!!!
   * <p>
   * Changes
   * <pre>await cls.meth(a, b)</pre>
   * to
   * <pre>await $n4promisifyMethod(cls, 'meth', [a, b])</pre>
   * OR
   * <pre>await fun(a, b)</pre>
   * to
   * <pre>await $n4promisifyFunction(fun, [a, b])</pre>
   * assuming that method 'meth' and function 'fun' are annotated with <code>@Promisifiable</code>.
   */
  private void _transformExpression(final AwaitExpression awaitExpr) {
    final AwaitExpression awaitExprOrig = this.getState().tracer.<AwaitExpression>getOriginalASTNodeOfSameType(awaitExpr, false);
    boolean _isAutoPromisify = this.promisifyHelper.isAutoPromisify(awaitExprOrig);
    if (_isAutoPromisify) {
      Expression _expression = awaitExpr.getExpression();
      final ParameterizedCallExpression callExpr = ((ParameterizedCallExpression) _expression);
      final ParameterizedCallExpression replacement = this.promisify(callExpr);
      this.replace(callExpr, replacement);
    }
  }
  
  /**
   * Changes
   * <pre>@Promisify cls.meth(a, b)</pre>
   * to
   * <pre>$n4promisifyMethod(cls, 'meth', [a, b])</pre>
   * OR
   * <pre>@Promisify fun(a, b)</pre>
   * to
   * <pre>$n4promisifyFunction(fun, [a, b])</pre>
   */
  private void _transformExpression(final PromisifyExpression promiExpr) {
    Expression _expression = promiExpr.getExpression();
    final ParameterizedCallExpression callExpr = ((ParameterizedCallExpression) _expression);
    final ParameterizedCallExpression replacement = this.promisify(callExpr);
    this.replace(promiExpr, replacement);
  }
  
  private ParameterizedCallExpression promisify(final ParameterizedCallExpression callExpr) {
    final Expression target = callExpr.getTarget();
    SymbolTableEntry _switchResult = null;
    boolean _matched = false;
    if (target instanceof ParameterizedPropertyAccessExpression_IM) {
      _matched=true;
      _switchResult = ((ParameterizedPropertyAccessExpression_IM)target).getProperty_IM();
    }
    if (!_matched) {
      if (target instanceof IdentifierRef_IM) {
        _matched=true;
        _switchResult = ((IdentifierRef_IM)target).getId_IM();
      }
    }
    final SymbolTableEntry targetSTE = _switchResult;
    if ((targetSTE instanceof SymbolTableEntryOriginal)) {
      final IdentifiableElement originalTarget = ((SymbolTableEntryOriginal)targetSTE).getOriginalTarget();
      if ((originalTarget instanceof TFunction)) {
        ParameterizedTypeRef _createTypeRef = TypeUtils.createTypeRef(((Type)originalTarget));
        final FunctionTypeRef originalTargetTypeRef = ((FunctionTypeRef) _createTypeRef);
        final TypeRef returnTypeRef = this.promisifyHelper.extractPromisifiedReturnType(this.getState().G, originalTargetTypeRef);
        boolean _isUndefined = TypeUtils.isUndefined(IterableExtensions.<TypeArgument>head(IterableExtensions.<TypeArgument>drop(returnTypeRef.getTypeArgs(), 1)));
        final boolean hasErrorValue = (!_isUndefined);
        final boolean hasMoreThan1SuccessValue = RuleEnvironmentExtensions.isIterableN(this.getState().G, IterableExtensions.<TypeArgument>head(returnTypeRef.getTypeArgs()));
        if (((target instanceof ParameterizedPropertyAccessExpression_IM) && (((SymbolTableEntryOriginal)targetSTE).getOriginalTarget() instanceof TMethod))) {
          final Function1<Argument, ArrayElement> _function = (Argument it) -> {
            return TranspilerBuilderBlocks._ArrayElement(it.isSpread(), it.getExpression());
          };
          return TranspilerBuilderBlocks._CallExpr(
            TranspilerBuilderBlocks._IdentRef(this.steFor_$n4promisifyMethod()), 
            ((ParameterizedPropertyAccessExpression_IM) target).getTarget(), 
            TranspilerBuilderBlocks._StringLiteralForSTE(targetSTE), 
            TranspilerBuilderBlocks._ArrLit(((ArrayElement[])Conversions.unwrapArray(ListExtensions.<Argument, ArrayElement>map(callExpr.getArguments(), _function), ArrayElement.class))), 
            TranspilerBuilderBlocks._BooleanLiteral(hasMoreThan1SuccessValue), 
            TranspilerBuilderBlocks._BooleanLiteral((!hasErrorValue)));
        } else {
          final Function1<Argument, ArrayElement> _function_1 = (Argument it) -> {
            return TranspilerBuilderBlocks._ArrayElement(it.isSpread(), it.getExpression());
          };
          return TranspilerBuilderBlocks._CallExpr(
            TranspilerBuilderBlocks._IdentRef(this.steFor_$n4promisifyFunction()), 
            callExpr.getTarget(), 
            TranspilerBuilderBlocks._ArrLit(((ArrayElement[])Conversions.unwrapArray(ListExtensions.<Argument, ArrayElement>map(callExpr.getArguments(), _function_1), ArrayElement.class))), 
            TranspilerBuilderBlocks._BooleanLiteral(hasMoreThan1SuccessValue), 
            TranspilerBuilderBlocks._BooleanLiteral((!hasErrorValue)));
        }
      }
    }
    return callExpr;
  }
  
  private void transformExpression(final Expression awaitExpr) {
    if (awaitExpr instanceof AwaitExpression) {
      _transformExpression((AwaitExpression)awaitExpr);
      return;
    } else if (awaitExpr instanceof CastExpression) {
      _transformExpression((CastExpression)awaitExpr);
      return;
    } else if (awaitExpr instanceof PromisifyExpression) {
      _transformExpression((PromisifyExpression)awaitExpr);
      return;
    } else if (awaitExpr instanceof RelationalExpression) {
      _transformExpression((RelationalExpression)awaitExpr);
      return;
    } else if (awaitExpr != null) {
      _transformExpression(awaitExpr);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(awaitExpr).toString());
    }
  }
}
